/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.state;

import java.util.List;

import de.hfaber.asteroids.game.objects.GameObject;

/**
 * Interface for a strategy to match game objects from the current
 * frame with game objects from a previous frame. 
 * 
 * @author Henning Faber
 */
interface IMatchStrategy {

    /**
     * Determines, if the given game object from the current frame
     * satisfies the precondition of this strategy to be a possible
     * match for the given object from a previous frame.
     * 
     * @param current the game object from the current frame
     * @param prev the game object from a previous frame
     * @param frameGap the difference between the frame numbers of the
     *  frame where this game object belongs to and the frame from where 
     *  the given previous game object orginates 
     * @return <code>true</code> if the two game objects satisfy the
     *  precondition of this strategy, <code>false</code> if not
     */
    public boolean satisfiesPrecondition(GameObject current, GameObject prev,
            int frameGap);
    
    /**
     * Receives a list of all game objects from the current frame that
     * satisfy the precondition of this strategy and could be a possible
     * match for the given game object from a previous frame. The strategy
     * may select and return the best match. If non of the given game
     * objects are considered to be worth as best match, the strategy
     * may return <code>null</code>. 
     * 
     * @param current the game object from the current frame
     * @param matchList the list of game objects from a previous frame,
     *  that satisfy the strategy's precondition 
     * @param frameGap the difference between the frame numbers of the
     *  frame where this game object belongs to and the frame from where 
     *  the given previous game object orginates 
     * @return the game object from the list of matches, which is considered
     *  to be the best match or <code>null</code>, if no best match can
     *  be determined
     */
    public GameObject selectBestMatch(GameObject current, 
            List<? extends GameObject> matchList, int frameGap);
}
