#include <stdio.h>              // printf
#include <cmath>

#include "eigen/vector.h"

#include "game.h"

#include "player.h"      // own interface

#define DELTA_ANGLE          (2.0 * M_PI / 256.0)
#define DELTA_ANGLE_PER_TURN (3.0 * DELTA_ANGLE)

template<class T>
T xNorm(T x)
{
   while (x[0] < -512)
   {
      x[0] += 1024;
   }
   while (x[0] > 511)
   {
      x[0] -= 1024;
   }

   while (x[1] < -384)
   {
      x[1] += 768;
   }
   while (x[1] > 384)
   {
      x[1] -= 768;
   }

   return x;
}



double normAngle(double a)
{
   while (a < 0)
   {
      a += 2.0*M_PI;
   }
   while (a > 2.0*M_PI)
   {
      a -= 2.0*M_PI;
   }
   return a;
}

double normAngle2(double a)
{
   while (a < -M_PI)
   {
      a += 2.0*M_PI;
   }
   while (a > M_PI)
   {
      a -= 2.0*M_PI;
   }
   return a;
}


void Player::AimAngle(
   GameObject&  obj,
   const double initialAngleGuess,
   const double alpha_ship) const
{
   double t_flight, t_aim;
   double prevAimAngle;
   double delta_alpha;
   double delta_alpha_prev;
   // acceptable error: 0.1 degree
   const double delta_alphaMax = 0.1 * M_PI/180.0;
   int iter = 0;
   const int iterMax = 15;      // constant parameter

   Eigen::Vector2d x, x_aim;

   // initial guess
   obj.aimAngle = initialAngleGuess;

   do
   {
      // aiming time: 256 frames per 3 turn around
      // omega = 6 Pi/256 rad/frame
      t_aim = fabs(normAngle2(alpha_ship - obj.aimAngle))
         * 256.0 / (6.0 * M_PI);

      // recalculate x,y of astro at time of fire
      x = obj.x_rel + t_aim * obj.v_rel;

      // calculate projectile flight time
      // t^2 + p * t + q = 0
      // p = 2 * x_t * v_t / (v_t^2 - v_p^2)
      // q = x_t^2 / (v_t^2 - v_p^2)
      // v_p = 8 pixel/frame
      double den = obj.v_rel.norm2() - 8*8;
      double p   = 2.0 * x.dot(obj.v_rel) / den;
      double q   = x.norm2() / den;
      double p2  = p/2.0;
      if (p2*p2 >= q)
      {
         double sq = sqrt(p2*p2 - q);
         double t1 = - p2 + sq;
         double t2 = - p2 - sq;
         if (t2 > 0 && t2 < t1)
         {
            t_flight = t2;
         }
         else
         {
            t_flight = t1;
         }
      }
      else
      {
         t_flight = 0;
      }

      // aiming time + flight time
      obj.impactTime = t_aim + t_flight;

      // new coordinates to aim to, where the impact takes place
      x_aim  = x + t_flight * obj.v_rel;

      prevAimAngle = obj.aimAngle;
      obj.aimAngle = normAngle(atan2(x_aim[1], x_aim[0]));

      // impact distance
      obj.dist = x_aim.norm();
      // allowed angle set off for shooting
      obj.deltaAimAngle = 0.75 * obj.radius / obj.dist;
      //obj.deltaAimAngle = 7.0 / obj.dist;

      delta_alpha_prev = delta_alpha;
      delta_alpha = normAngle2(prevAimAngle - obj.aimAngle);

      if (iter != 0
         && delta_alpha_prev * delta_alpha < 0)
      {
         // reduce overshooting
         obj.aimAngle = 0.5*(obj.aimAngle + prevAimAngle);
      }

#if 0
      if (iter > 9)
      {
         printf(
            "iter=%d, x_rel=%f,%f v=(%f,%f) t_aim=%f,"
            " t_flight=%f, aimAngle=%f\n",
            iter, obj.x_rel[0], obj.x_rel[1], obj.v_rel[0], obj.v_rel[1],
            t_aim, t_flight, obj.aimAngle);
      }
#endif

      ++iter;
   } while (fabs(delta_alpha) > delta_alphaMax && iter < iterMax );

   if (iter>=iterMax)
   {
      // do not aim at this object
      obj.impactTime = 1e10;

      printf("AimAngle(): iter=%d, delta_alpha=%f\n", iter, delta_alpha);
   }
}

double AlphaRel(const int angle_byte, const double alpha_t)
{
   // relative shooting direction angle
   const double gamma = angle_byte * DELTA_ANGLE;
   // relative shooting angle wrt the target: -PI < alpha_rel < PI
   double alpha_rel = gamma - alpha_t;
   while (alpha_rel > M_PI)
   {
      alpha_rel -= 2.0*M_PI;
   }
   while (alpha_rel < -M_PI)
   {
      alpha_rel += 2.0*M_PI;
   }
   return alpha_rel;
}

void shift(int a[], int size)
{
   for (int i = size-1; i>0; --i)
   {
      a[i] = a[i-1];
   }
}

#define HISTARR_SIZE 5
// current ship angle wrt x axis
static int ibeta[HISTARR_SIZE];
static int turn[HISTARR_SIZE];


int CorrectAngleByte(
   const int ping,
   int angle_byte)
{
   // ship graphics turns in units of appox. 2*PI * 1/64 rad = 2*PI * 4/256
   // rad = 5.6 degree
   //
   // one pressing of the key turns shooting direction by 2*PI * 3/256 rad =
   // 4.219 deg

   // adjust the angle byte depending on the turn operation 2 cycles ago
   if (turn[2+ping] == -1)
   {
      angle_byte -= 3;
   }
   else if (turn[2+ping] == 1)
   {
      angle_byte += 3;
   }
   else
   {
      // do nothing
   }

   while (angle_byte < 0)
   {
      angle_byte += 0x100;
   }
   angle_byte %= 0x100;

   // store angle byte before sync
   int orig_angle_byte = angle_byte;

   // try to sync the angle byte with the ship direction
   if (ibeta[0] == 64 || ibeta[0] == 192)
   {
      angle_byte = ibeta[0];
   }
   else if (
      turn[2+ping] == 1           // left turn
      &&
      ibeta[0] == ibeta[1]
      &&
      ibeta[0] != 0                // ambigous directions
      &&
      ibeta[0] != 128)
   {
      // we tried to turn but the ship position did not change -> we may
      // derive the angle byte in this case
      if ((ibeta[0] < 64)
         || ((ibeta[0] > 128) && (ibeta[0] < 192)))
      {
         angle_byte = ibeta[0] + 3;
      }
      else
      {
         angle_byte = ibeta[0];
      }
   }
   else if (
      turn[2+ping] == -1         // right turn
      &&
      ibeta[0] == ibeta[1]
      &&
      ibeta[0] != 0                // ambigous directions
      &&
      ibeta[0] != 128)
   {
      // we tried to turn but the ship position did not change -> we may
      // derive the angle byte in this case
      if ((ibeta[0] < 64)
         || ((ibeta[0] > 128) && (ibeta[0] < 192)))
      {
         angle_byte = ibeta[0];
      }
      else
      {
         angle_byte = ibeta[0] - 3;
      }
   }


   while (angle_byte < 0)
   {
      angle_byte += 0x100;
   }
   angle_byte %= 0x100;

   if (orig_angle_byte != angle_byte)
   {
      printf("angle byte adjustment %d -> %d\n",
         orig_angle_byte, angle_byte);
   }

   return angle_byte;
}



//#define ANGLE_TEST
#ifdef ANGLE_TEST

void Player::MakeTurn(
   GameStatus&       gamestate,
   const GameStatus& lastGamestate,
   KeysPacket&       keys,
   const int         ping)
{
   static int angle_byte = -3;

   if (!gamestate.ship.IsPresent())
   {
      // do nothing if no ship is visible
      return;
   }

   // current ship angle wrt x axis
   const double beta
      = normAngle(atan2(gamestate.ship.view[1], gamestate.ship.view[0]));

   shift(ibeta, HISTARR_SIZE);
   ibeta[0] = lrint(beta/(2.0*M_PI / 256.0)); // in bytes

   printf("Frame# %d, angle byte %d, ship angle %d (%f,%f)\n",
      gamestate.frameno, angle_byte,
      ibeta[0], gamestate.ship.view[0], gamestate.ship.view[1]);

   if (gamestate.nshots > 0)
   {
      const double shot_angle
         = normAngle(
            atan2(gamestate.shots[0].x_rel[1], gamestate.shots[0].x_rel[0]));
      const double shot_angle_byte = shot_angle/(2.0*M_PI / 256.0);
      printf("shot: %f,%f, shot_angle_byte=%f\n",
         gamestate.shots[0].x_rel[0], gamestate.shots[0].x_rel[1],
         shot_angle_byte);
      fire = 0;
   }
   else
   {
      if (fire == 0)
      {
         printf("Sending fire\n");
         keys.fire(true);
         fire = 1;
      }
   }

   keys.left(true);
   angle_byte += 3;
   angle_byte %= 0x100;
}

#else

void Player::MakeTurn(
   GameStatus&       gamestate,
   const GameStatus& lastGamestate,
   KeysPacket&       keys,
   const int         ping)
{
   if (!gamestate.ship.IsPresent())
   {
      // do nothing if no ship is visible
      return;
   }

   static int angle_byte       = 0;
   static double alpha_t       = 0;
   static double delta_alpha_t = 0;

   double min_dist = 1e10;

   Eigen::Vector2d min_dx;
   min_dx.loadZero();

   double impactTimeMin    = 1e10;
   double collisionTimeMin = 1e10;

   // frame difference = time difference
   const int dt  = gamestate.frameno - lastGamestate.frameno;

   // ship angle (float)
   const double beta = normAngle(
      atan2(gamestate.ship.view[1], gamestate.ship.view[0]));

   // ship angle (integer)
   shift(ibeta, HISTARR_SIZE);
   ibeta[0] = lrint(beta/DELTA_ANGLE); // in bytes

   // store last turns
   shift(turn, HISTARR_SIZE);
   turn[0] = 0;              // may be overwritten later


   // velocity of ship wrt background
   gamestate.ship.v = xNorm(gamestate.ship.x - lastGamestate.ship.x) / dt;
   const double v2_ship = gamestate.ship.v.norm2();
#define V_SHIP_MAX 2


   // velocity of saucer wrt background
   if (gamestate.saucer.IsPresent())
   {
      if (lastGamestate.saucer.dt == -1)
      {
         // no x0,y0 available yet, set them
         gamestate.saucer.x0 = gamestate.saucer.x;
         gamestate.saucer.dt = 0;
         gamestate.saucer.v.loadZero();
      }
      else
      {
         gamestate.saucer.x0 = lastGamestate.saucer.x0;
         gamestate.saucer.dt = lastGamestate.saucer.dt + dt;
         gamestate.saucer.v =
            (gamestate.saucer.x - gamestate.saucer.x0)
            / gamestate.saucer.dt;
         const Eigen::Vector2d v_simple(
            (gamestate.saucer.x - lastGamestate.saucer.x)
            / dt);
         const Eigen::Vector2d dv(
            gamestate.saucer.v - v_simple);
         if (dv.norm2() > 2*2)
         {
            // abnormal jump in calculated velocity
            gamestate.saucer.x0 = gamestate.saucer.x;
            gamestate.saucer.dt = 0;
            gamestate.saucer.v.loadZero();
         }
      }

      // velocity of saucer wrt ship
      gamestate.saucer.v_rel = gamestate.saucer.v - gamestate.ship.v;
   }

   // impact of shot in saucer
   gamestate.saucer.impactCountdown
      = lastGamestate.saucer.impactCountdown;
   if (gamestate.saucer.impactCountdown > 0)
   {
      --gamestate.saucer.impactCountdown;
   }


   // Determine asteroid distances, velocity, aimAngle, collision parameters
   for (int i=0; i<gamestate.nasteroids; ++i)
   {
      // determine distance to ship for collision avoidance
      Eigen::Vector2d dx(gamestate.asteroids[i].x_rel);

      // correct distance by asteroid size
      double dist = dx.norm2()
         - gamestate.asteroids[i].radius * gamestate.asteroids[i].radius;
      if (dist < min_dist)
      {
         min_dist = dist;
         min_dx   = dx;
      }

      // DEBUG
      // printf("old astro# %d, x=%d, y=%d, vx=%f, vy=%f\n",
      //    i, lastGamestate.asteroids[i].x, lastGamestate.asteroids[i].y,
      //    lastGamestate.asteroids[i].vx, lastGamestate.asteroids[i].vy);
      // printf("new astro# %d, x=%d, y=%d, vx=%f, vy=%f\n",
      //    i, gamestate.asteroids[i].x, gamestate.asteroids[i].y,
      //    gamestate.asteroids[i].vx, gamestate.asteroids[i].vy);

      // calc velocity
      if (
         gamestate.asteroids[i].type == lastGamestate.asteroids[i].type
         &&
         gamestate.asteroids[i].sf   == lastGamestate.asteroids[i].sf)
      {
         // probable new coordinates of old asteroid

         // calculate velocity
         if (lastGamestate.asteroids[i].dt == -1)
         {
            // no x0,y0 available yet, set them
            gamestate.asteroids[i].x0 = gamestate.asteroids[i].x;
            gamestate.asteroids[i].dt = 0;
            gamestate.asteroids[i].v.loadZero();
         }
         else
         {
            gamestate.asteroids[i].x0 = lastGamestate.asteroids[i].x0;
            gamestate.asteroids[i].dt = lastGamestate.asteroids[i].dt + dt;
            gamestate.asteroids[i].v =
               (gamestate.asteroids[i].x - gamestate.asteroids[i].x0)
               / gamestate.asteroids[i].dt;
            const Eigen::Vector2d v_simple(
               (gamestate.asteroids[i].x - lastGamestate.asteroids[i].x)
               / dt);
            const Eigen::Vector2d dv(
               gamestate.asteroids[i].v - v_simple);
            if (dv.norm2() > 2*2)
            {
               // abnormal jump in calculated velocity
               gamestate.asteroids[i].x0 = gamestate.asteroids[i].x;
               gamestate.asteroids[i].dt = 0;
               gamestate.asteroids[i].v.loadZero();
            }
         }

         // printf(
         //    "astro# %d, x=%d, y=%d, vx=%f, vy=%f, x0=%d, y0=%d, dt=%d\n",
         //    i, gamestate.asteroids[i].x, gamestate.asteroids[i].y,
         //    gamestate.asteroids[i].vx, gamestate.asteroids[i].vy,
         //    gamestate.asteroids[i].x0, gamestate.asteroids[i].y0,
         //    gamestate.asteroids[i].dt);
      }
      // else: astro was replaced by smaller one or is a new one

      // calc rel. asteroid speed wrt ship
      gamestate.asteroids[i].v_rel
         = gamestate.asteroids[i].v - gamestate.ship.v;


      // collision parameters
      // t_col = - \vec x0 \vec v / v^2
      gamestate.asteroids[i].collisionTime = 1e10;
      const double v2 = gamestate.asteroids[i].v_rel.norm2();
      if (v2 > 0)
      {
         // astro is closing in
         const double collisionTime
            = -gamestate.asteroids[i].x_rel.dot(gamestate.asteroids[i].v_rel)
            / v2;
         if (collisionTime > 0 && collisionTime < 180)
         {
            Eigen::Vector2d bx(
               xNorm(gamestate.asteroids[i].x_rel
                  + collisionTime * gamestate.asteroids[i].v_rel));
            // correct distance by asteroid size
            // and decrease by ship size
            const double b2 = bx.norm2()
               - gamestate.asteroids[i].radius * gamestate.asteroids[i].radius
               - 30*30;
            if (b2 < 0)
            {
               // collision danger!
               gamestate.asteroids[i].collisionTime = collisionTime;
            }
         }
      }

      // impact of shot
      gamestate.asteroids[i].impactCountdown
         = lastGamestate.asteroids[i].impactCountdown;
      if (gamestate.asteroids[i].impactCountdown > 0)
      {
         --gamestate.asteroids[i].impactCountdown;
      }
   }

   // calculate shooting direction and impact time for each target
   int target_idx = 0xFF;

   // look for best asteroid candidate with the smallest impact time
   impactTimeMin    = 1e10;
   collisionTimeMin = 1e10;
   alpha_t          = 0; // no astro as target
   delta_alpha_t    = 0;

   for (int i=0; i<gamestate.nasteroids; ++i)
   {
      // calculate aim angle and impact time for each asteroid
      AimAngle(
         gamestate.asteroids[i], lastGamestate.asteroids[i].aimAngle, beta);
   }
   // calculate aim angle and impact time for saucer
   if (gamestate.saucer.IsPresent())
   {
      AimAngle(gamestate.saucer, lastGamestate.saucer.aimAngle, beta);
   }


   if (gamestate.saucer.IsPresent()
      &&
      // and not already shot at
      (gamestate.saucer.impactCountdown <= 0
         || gamestate.nasteroids == 0))
   {
      // if there is a saucer, attack it first
      alpha_t       = gamestate.saucer.aimAngle;
      delta_alpha_t = gamestate.saucer.deltaAimAngle;
      impactTimeMin = gamestate.saucer.impactTime;
      // Abstand um den ungefhren Radius des UFOs korrigieren
      const double dist
         = gamestate.saucer.x_rel.norm2()
         - gamestate.saucer.radius * gamestate.saucer.radius;

      // only for collision prevention
      if (dist < min_dist)
      {
         min_dist = dist;
      }
   }
   else
   {
      // find best other target
      for (int i=0; i<gamestate.nasteroids; ++i)
      {
         if (
            // smallest impact time
            gamestate.asteroids[i].impactTime < impactTimeMin
            &&
            // and not already shot at
            (gamestate.asteroids[i].impactCountdown <= 0
               || gamestate.nasteroids == 1)
            &&
            // and within reach (shot distance = 570)
            gamestate.asteroids[i].dist < 570)
         {
            impactTimeMin = gamestate.asteroids[i].impactTime;
            alpha_t       = gamestate.asteroids[i].aimAngle;
            delta_alpha_t = gamestate.asteroids[i].deltaAimAngle;
            target_idx    = i;
         }
      }
   }

#if 0
   if (target_idx != 0xFF)
   {
      printf(
         "Target: astro# %d, impact time=%f, x=%f,%f, v=%f,%f\n",
         target_idx,
         impactTimeMin,
         gamestate.asteroids[target_idx].x[0],
         gamestate.asteroids[target_idx].x[1],
         gamestate.asteroids[target_idx].v[0],
         gamestate.asteroids[target_idx].v[1]);
   }
#endif

   // angle byte correction
   angle_byte = CorrectAngleByte(ping, angle_byte);

   // relative shooting angle wrt the target: -PI < alpha_rel < PI
   double alpha_rel = AlphaRel(angle_byte, alpha_t);

   // no target found?
   if (delta_alpha_t <= 0)
   {
      // go back to the middle of the screen
      // printf("INFO: no target found\n");

      // y  = 128 bis 895
      // x =    0 bis 1024
      // mitte = (512, 384 + 128) = (512, 512)

      Eigen::Vector2d x(
         -gamestate.ship.x + Eigen::Vector2d(512.0, 512.0));
      alpha_t   = normAngle(atan2(x[1], x[0]));
      alpha_rel = AlphaRel(angle_byte, alpha_t);

      if ((x.norm2() > 250*250)
         && (fabs(alpha_rel) < M_PI/2.0)
         && v2_ship < V_SHIP_MAX * V_SHIP_MAX)
      {
         keys.thrust(true);
      }
   }
   else
   {
#if 0
      printf("frame# %d, aiming at %f +- %f\n",
         gamestate.frameno,
         alpha_t*180.0/M_PI, delta_alpha_t*180.0/M_PI);
#endif
   }

   // Flucht, wenn Kollision unausweichlich
   if (min_dist < 35*35)
   {
      keys.hyperspace(true);
   }

   if (fire_state == FIRED)
   {
      fire_state = FIRED_IN_LAST_CYCLE;
   }

   if (state == DEFAULT)
   {
      // turn ship if neccessary
      if (alpha_rel
         > M_PI * 3.0 / 256.0
         - (turn[1]+turn[2]) * DELTA_ANGLE_PER_TURN)
      {
         turn[0] = -1;
         keys.right(true);
      }
      else if (alpha_rel
         < -M_PI * 3.0 / 256.0
         - (turn[1]+turn[2]) * DELTA_ANGLE_PER_TURN)
      {
         turn[0] = 1;
         keys.left(true);
      }
      else
      {
         turn[0] = 0;
      }

      // cycle through targets to fire en passant
      for (int i=0; i<gamestate.nasteroids; ++i)
      {
         // press fire if we are on target in the next frame
         if (fabs(AlphaRel(
                  angle_byte, gamestate.asteroids[i].aimAngle)
               + turn[1+ping] * DELTA_ANGLE_PER_TURN)
            < gamestate.asteroids[i].deltaAimAngle
            &&
            // and within reach (shot distance = 570)
            gamestate.asteroids[i].dist < 570
            &&
            (fire_state == OFF)
            &&
            // current target is not final target
            (i != target_idx))
         {
            fire_state = FIRED;

            keys.fire(true);

            // reset impact countdown value
            gamestate.asteroids[i].impactCountdown = (int)
               (1.05f * gamestate.asteroids[i].x_rel.norm() / 8.0);
         }
      }

      // press fire if we are on target in the next frame
      double alpha_rel_next
         = alpha_rel + turn[1+ping] * DELTA_ANGLE_PER_TURN;
      if (fabs(alpha_rel_next) < delta_alpha_t
         &&
         (fire_state == OFF))
      {
         fire_state = FIRED;

         keys.fire(true);

         if (target_idx != 0xFF)
         {
            // reset impact countdown value
            gamestate.asteroids[target_idx].impactCountdown = (int)
               (1.05f * gamestate.asteroids[target_idx].x_rel.norm() / 8.0);

            printf("frame# %d, fireing at %f +- %f, sf=%d\n",
               gamestate.frameno,
               alpha_t*180.0/M_PI, delta_alpha_t*180.0/M_PI,
               gamestate.asteroids[target_idx].sf);

            if (gamestate.asteroids[target_idx].sf == 0)
            {
               // large astro
               state = FIRE_AT_LARGE_ASTRO;
               fire  = 1;           // 2 shots remaining
            }
         }
         else if (gamestate.saucer.IsPresent())
         {
            gamestate.saucer.impactCountdown = (int)
               (1.05f * gamestate.saucer.x_rel.norm() / 8.0);
         }
      }

   }
   else                         // FIRE_AT_LARGE_ASTRO
   {
      if (fire_state == OFF)
      {
         fire_state = FIRED;
         keys.fire(true);
         if (--fire < 0)
         {
            state = DEFAULT;    // return to default operation
         }

         printf("frame# %d, fire %d at large Astro at %f +- %f\n",
            gamestate.frameno, fire,
            alpha_t*180.0/M_PI, delta_alpha_t*180.0/M_PI);
      }
   }

   if (fire_state == FIRED_IN_LAST_CYCLE)
   {
      fire_state = OFF;         // we may fire in the next cycle again
   }

}

#endif
