// game.cpp: Beispielspieler fr Asteroids
// Matthias Fuchs
// Original: Harald Bgeholz / c't

#include <stdint.h>             // uint*_t

#include "eigen/vector.h"

#include "output.h"

#include "game.h"               // own interface

GameObject::GameObject(void)
   : dt(-1),
     dist(0),
     radius(0),
     collisionTime(0),
     aimAngle(0),
     deltaAimAngle(0),
     impactTime(0),
     impactCountdown(0)
{
   x.loadZero();
   x0.loadZero();
   v.loadZero();
   x_rel.loadZero();
   v_rel.loadZero();
}

void Asteroid::set(int _x, int _y, int _type, int _sf)
{
   x[0] = _x;
   x[1] = _y;
   type = _type;
   sf   = _sf;

   if (_sf == 0)
   {
      points = 20;      // gro
      radius = 31;
   }
   else if (_sf == 15)
   {
      points = 50;      // mittel
      radius = 15;
   }
   else
   {
      points = 100;     // klein
      radius = 7;
   }

   x0.loadZero();
   dt = -1;
   v.loadZero();
   x_rel.loadZero();
   v_rel.loadZero();

   aimAngle = 0;
}

void Shot::set(int _x, int _y)
{
   x[0] = _x;
   x[1] = _y;
}

bool Saucer::IsPresent (void) const
{
   return size > 0;
}

void Saucer::set(int _x, int _y, int _size)
{
   x[0] = _x;
   x[1] = _y;
   size = _size;

   if (_size == 15)
   {
      points = 200;     // gro
      radius = 12;
   }
   else
   {
      points = 1000;    // klein
      radius = 8;
   }

   x0.loadZero();
   dt = -1;
   v.loadZero();
   x_rel.loadZero();
   v_rel.loadZero();

   aimAngle = 0;
}

bool Ship::IsPresent (void) const
{
   return present;
}

void Ship::set(int _x, int _y, int _viewx, int _viewy)
{
   x[0]  = _x;
   x[1]  = _y;
   view[0] = _viewx;
   view[1] = _viewy;
   present = true;
}


GameStatus::GameStatus(void)
:  nasteroids(0),
   nshots(0),
   t(0),
   frameno(0),
   score(0)
{
   ship.present = false;
   saucer.size  = 0;
}

void GameStatus::clear(void)
{
   ship.present = false;
   saucer.size  = 0;
   nasteroids   = 0;
   nshots       = 0;
}

void GameStatus::RelCoordinates(GameObject& obj) const
{
   obj.x_rel = obj.x - ship.x;

   // x normalisieren auf -512 ... 511
   // y normalisieren auf -384 ... 383

   while (obj.x_rel[0] < -512)
   {
      obj.x_rel[0] += 1024;
   }
   while (obj.x_rel[0] > 511)
   {
      obj.x_rel[0] -= 1024;
   }

   while (obj.x_rel[1] < -384)
   {
      obj.x_rel[1] += 768;
   }
   while (obj.x_rel[1] > 383)
   {
      obj.x_rel[1] -= 768;
   }
}

void Game::Run(void)
{
   FramePacket frame;
   KeysPacket  keys;
   GameStatus  state[2];
   int8_t      expectedFrame = 0;

   for (unsigned int t = 0; /*endlos*/; ++t)
   {
      const int stateIdx    = t%2;
      const int oldStateIdx = (stateIdx+1)%2;

      ++keys.ping;
      connection.SendPacket(keys);
      connection.ReceivePacket(frame);

      // jedes gesendete Paeckchen erhlt eine individuelle Nummer zur
      // Latenzmessung
      state[stateIdx].frameno = state[oldStateIdx].frameno + 1;

      int ping = keys.ping - frame.ping;
      if (ping < 0)
      {
         ping += 256;
      }

      ++expectedFrame;
      int lostFrames = frame.frameno - expectedFrame;
      if (lostFrames < 0)
      {
         lostFrames += 256;
      }

      if (ping != 0 || lostFrames != 0)
      {
         expectedFrame            = frame.frameno;
         state[stateIdx].frameno += lostFrames;

         OutputInfo2("Latenz %d. %d Frames verloren.\n",
            ping, lostFrames);
      }

      InterpretScreen(frame, state[stateIdx]);

      state[stateIdx].t = t;

      keys.clear();   // alle Tasten loslassen
      player.MakeTurn(
         state[stateIdx], state[oldStateIdx],
         keys,
         ping);
   }
}


void Game::InterpretScreen(const FramePacket& packet, GameStatus& game)
{
   const uint8_t* const vector_ram = packet.vectorram;
   int dx, dy, sf, vx, vy, vz, vs;
   int v1x        = 0;
   int v1y        = 0;
   int shipdetect = 0;
   unsigned int pc;

   uint16_t value, value2;

   game.clear();

   value = vector_ram[0] | (vector_ram[1] << 8);
   if (value != 0xe001 && value != 0xe201)
   {
      return; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL
   }

   pc = 2;
   while (pc < sizeof(packet.vectorram))
   {
      value = vector_ram[pc] | (vector_ram[pc+1] << 8);
      const int op = value >> 12; // op code in top 4 bit
      value &= 0x0fff;            // mask out op code

      switch (op)
      {
         default: // op code 0..9: VCTR, draw vector
            value2 = vector_ram[pc+2] | (vector_ram[pc+3] << 8);
            dy = value & 0x3ff;
            if ((value & 0x400) != 0)
            {
               dy = -dy;
            }
            dx = value2 & 0x3ff;
            if ((value2 & 0x400) != 0)
            {
               dx = -dx;
            }
            sf = op;            // scaling factor
            vz = value2 >> 12;  // brightness
            if (dx == 0 && dy == 0 && vz == 15)
            {
               game.shots[game.nshots++].set(vx, vy);
            }
            if (op == 6 && vz == 12 && dx != 0 && dy != 0)
            {
               // ship is drawn
               switch (shipdetect)
               {
                  case 0:
                     v1x = dx;
                     v1y = dy;
                     ++shipdetect;
                     break;
                  case 1:
                     game.ship.set(vx, vy, v1x - dx, v1y - dy);
                     ++shipdetect;
                     break;
                  default:
                     // should never happen
                     break;
               }
            }
            else if (shipdetect != 0)
            {
               shipdetect = 0;
            }
            break;

         case 0xa: // LABS, position beam
            vy     = value & 0x3ff;
            value2 = vector_ram[pc+2] | (vector_ram[pc+3] << 8);
            vx     = value2 & 0x3ff;
            vs     = value2 >> 12; // global scaling factor
            break;

         case 0xb: // HALT
            pc = sizeof(packet.vectorram);
            break;

         case 0xc: // JSRL, subroutine
            switch (value)
            {
               case 0x8f3:      // asteroid type 1
                  game.asteroids[game.nasteroids++].set(vx, vy, 1, vs);
                  break;
               case 0x8ff:      // asteroid type 2
                  game.asteroids[game.nasteroids++].set(vx, vy, 2, vs);
                  break;
               case 0x90d:      // asteroid type 3
                  game.asteroids[game.nasteroids++].set(vx, vy, 3, vs);
                  break;
               case 0x91a:      // asteroid type 4
                  game.asteroids[game.nasteroids++].set(vx, vy, 4, vs);
                  break;
               case 0x929:      // UFO
                  game.saucer.set(vx, vy, vs);
                  break;
               default:
                  // ignore
                  break;
            }
            break;

         case 0xd: // RTSL, return from subroutine
            pc = sizeof(packet.vectorram);
            break;

         case 0xe: // JMPL, unconditional jump
            /*
              pc = (value & 0xfff) << 1;
              break;
            */
            pc = sizeof(packet.vectorram);
            break;

         case 0xf: // SVEC, short vector
            dy = value & 0x300;
            if ((value & 0x400) != 0)
            {
               dy = -dy;
            }
            dx = (value & 3) << 8;
            if ((value & 4) != 0)
            {
               dx = -dx;
            }
            sf = (((value & 8) >> 2) | ((value & 0x800) >> 11)) + 2;
            vz = (value & 0xf0) >> 4;
            break;
      }

      if (op <= 0xa)
      {
         pc += 2U;
      }

      if (op != 0xe) // JMPL
      {
         pc += 2U;
      }
   }

   // add values for the relative coordinates wrt ship
   int i;
   for (i=0; i<game.nasteroids; ++i)
   {
      game.RelCoordinates(game.asteroids[i]);
   }
   for (i=0; i<game.nshots; ++i)
   {
      game.RelCoordinates(game.shots[i]);
   }
   game.RelCoordinates(game.saucer);
}
