// ----------------------------------------------------------------------------
// 'The Sniper'  -  c't Asteroids bot  -  Thorsten Denhard, 2008
// ----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <winsock2.h>

#include "helper.h"

static
double getDistanceS (const int*   xa, 
                    const int*   ya, 
                    const double* ta, 
                    const int*   xb, 
                    const int*   yb, 
                    const double* tb, 
                    int          i, 
                    int          k, 
                    double        knownSpeed)
{
  double xi = double (xa[i]);
  double yi = double (ya[i]);
  double ti = double (ta[i]);

  double xk = double (xb[k]);
  double yk = double (yb[k]);
  double tk = double (tb[k]);

  xk = wrapRel (xk, xi, GAME_WIDTH);
  yk = wrapRel (yk, yi, GAME_HEIGHT);

  double xd = xi - xk;
  double yd = yi - yk;
  double ds = xd*xd + yd*yd;

  double td = fabs (ti - tk);
  double m  = knownSpeed * td;
  double ms = m*m;

  return fabs (ds-ms);
}

void matchObjects (const int*   xa, 
                   const int*   ya, 
                   const double* ta, 
                   const int*   xb, 
                   const int*   yb, 
                   const double* tb, 
                   int          sa, 
                   int          sb, 
                   double        knownSpeed,
                   int*         matchIndexListRet)
{
  // Preliminaries 

  int i = 0;
  int k = 0;
  
  for (i=0; i<sa; ++i)
  {
    matchIndexListRet[i] = -1;
  }

  if (sa > MAX_PER_GROUP)
  {
    return;
  }

  if (sa <= 0 ||
      sb <= 0)
  {
    return;
  }
  
  bool matchFlagList[MAX_PER_GROUP];

  for (i=0; i<MAX_PER_GROUP; ++i)
  {
    matchFlagList[i] = false;
  }

  // Match the two lists
  // This does not check all permutations and thus is 
  // not really correct, but what the hell...

  for (i=0; i<sa; ++i)
  {
    double minDistS = 1e30f;
    int   minIndex = -1;
    
    for (k=0; k<sb; ++k)
    {
      if (true == matchFlagList[k])
      {
        continue;
      }      

      double distS = getDistanceS (xa, ya, ta, xb, yb, tb, i, k, knownSpeed);
      
      // If distance greater than indicated by MAX_SPEED, this cannot be a match
      double td = fabs (ta[i]-tb[k]);
      double m  = MAX_SPEED * td;
      double ms = m*m;
            
      if (distS < minDistS &&
          distS < ms)
      {
        minDistS = distS;
        minIndex = k;
      }
    }

    if (minIndex != -1)
    {
      matchFlagList    [minIndex] = true;
      matchIndexListRet[i]        = minIndex;
    }
  }
}

// ax/ay: Target pos
// vx/vy: Target velocity
// bx/by: Ship position
// ws: Shot speed (scalar)
double getShotAngle (double ax, double ay, double vx, double vy,
                    double bx, double by, double ws,
                    double& tRet)
{
  tRet = 0.0f;
  
  // Needed vars

  const double maxAngleOff = 45.0f * DEG_TO_RAD;
  const int    maxIter     = 8;
        double angleDir    = 1.0f;
        int    i           = 0;

  // Determine vectors and angles
  
  double shipToTargetX = ax - bx;
  double shipToTargetY = ay - by;
  
  double shipToTargetLS = shipToTargetX*shipToTargetX + 
                          shipToTargetY*shipToTargetY;

  double shipToTargetL = mySqrt (shipToTargetLS);

  if (shipToTargetL == 0.0f)
  {
    return 0.0;
  }
  
  double shipToTargetXN = shipToTargetX / shipToTargetL;
  double shipToTargetYN = shipToTargetY / shipToTargetL;

  double initialAngle = myAcos (shipToTargetXN);

  if (shipToTargetYN < 0.0f)
  {
    initialAngle *= -1.0f;
    initialAngle += 2.0f * M_PI;
  }
  
  double vls = vx*vx + vy*vy;
  double vl  = mySqrt (vls);

  if (vl == 0.0f)
  {
    return double (initialAngle);
  }

  double vxN = vx / vl;
  double vyN = vy / vl;

  double vAngle = myAcos (vxN);

  if (vyN < 0.0f)
  {
    vAngle *= -1.0f;
    vAngle += 2.0f * M_PI;
  }

  // Rotate cw or ccw?

  {
    double currentVelXN = cos (initialAngle+0.005f*DEG_TO_RAD);
    double currentVelYN = sin (initialAngle+0.005f*DEG_TO_RAD);

    double currentVelX = ws * currentVelXN;
    double currentVelY = ws * currentVelYN;

    // Adjust source position: shots are not fired directly from ship center!
    double bbx = bx + SHOT_SOURCE_OFF * currentVelXN;
    double bby = by + SHOT_SOURCE_OFF * currentVelYN;
    
    double currentT    = 0.0f;
    double currentS    = 0.0f;
    int    currentTest = testIntersection (ax, ay, vx, vy, bbx, bby, 
                                           currentVelX, currentVelY, currentT, currentS);

    if (currentT < 0.0f)
    {
      angleDir *= -1.0f;
    }
  }
  
  // Check min and max angle
  
  double angleIntervalL = initialAngle;
  double angleIntervalR = initialAngle + angleDir*maxAngleOff;
  double centerInterval = 0.5f * (angleIntervalL + angleIntervalR);
  double tL             = 0.0f;  
  double tR             = 0.0f;  
  
  int testL = -2;
  int testR = -2;
  
  {
    double currentVelXN = cos (angleIntervalL);
    double currentVelYN = sin (angleIntervalL);

    double currentVelX = ws * currentVelXN;
    double currentVelY = ws * currentVelYN;

    // Adjust source position: shots are not fired directly from ship center!
    double bbx = bx + SHOT_SOURCE_OFF * currentVelXN;
    double bby = by + SHOT_SOURCE_OFF * currentVelYN;

    double currentT    = 0.0f;
    double currentS    = 0.0f;
    int    currentTest = testIntersection (ax, ay, vx, vy, bbx, bby, 
                                           currentVelX, currentVelY, currentT, currentS);

    testL = currentTest;
    tL    = currentT;
  }
  
  {
    double currentVelXN = cos (angleIntervalR);
    double currentVelYN = sin (angleIntervalR);

    double currentVelX = ws * currentVelXN;
    double currentVelY = ws * currentVelYN;

    // Adjust source position: shots are not fired directly from ship center!
    double bbx = bx + SHOT_SOURCE_OFF * currentVelXN;
    double bby = by + SHOT_SOURCE_OFF * currentVelYN;

    double currentT    = 0.0f;
    double currentS    = 0.0f;
    int    currentTest = testIntersection (ax, ay, vx, vy, bbx, bby, 
                                           currentVelX, currentVelY, currentT, currentS);

    testR = currentTest;
    tR    = currentT;
  }
  
  if (testL != 1)
  {
    tRet = double (tL);
    return double (angleIntervalL);
  }
  
  if (testR == 0)
  {
    tRet = double (tR);
    return double (angleIntervalR);
  }
  
  if (testR != -1 &&
      testR != -2 &&
      testR != -3)
  {
    tRet = double (tL);
    return double (angleIntervalL);
  }

  // Now, nested intervals to find transition
  
  for (i=0; i<maxIter; ++i)
  {
    double currentVelXN = cos (centerInterval);
    double currentVelYN = sin (centerInterval);

    double currentVelX = ws * currentVelXN;
    double currentVelY = ws * currentVelYN;

    // Adjust source position: shots are not fired directly from ship center!
    double bbx = bx + SHOT_SOURCE_OFF * currentVelXN;
    double bby = by + SHOT_SOURCE_OFF * currentVelYN;

    double currentT    = 0.0f;
    double currentS    = 0.0f;
    int    currentTest = testIntersection (ax, ay, vx, vy, bbx, bby, 
                                           currentVelX, currentVelY, currentT, currentS);
    
    if (currentTest == 0)
    {
      // Exact "hit", no more iters needed
      break;
    }
    else if (currentTest == 1)
    {
      angleIntervalL = centerInterval;
    }
    else
    {
      angleIntervalR = centerInterval;
    }
    
    centerInterval = 0.5f * (angleIntervalL + angleIntervalR);
  }

  // Determine final t and finish
  
  double tC = 0.0f;
  {
    double currentVelXN = cos (centerInterval);
    double currentVelYN = sin (centerInterval);

    double currentVelX = ws * currentVelXN;
    double currentVelY = ws * currentVelYN;

    // Adjust source position: shots are not fired directly from ship center!
    double bbx = bx + SHOT_SOURCE_OFF * currentVelXN;
    double bby = by + SHOT_SOURCE_OFF * currentVelYN;

    double currentT    = 0.0f;
    double currentS    = 0.0f;
    int    currentTest = testIntersection (ax, ay, vx, vy, bbx, bby, 
                                           currentVelX, currentVelY, currentT, currentS);

    tC = currentT;
  }
  
  tRet = double (tC);
  return double (centerInterval);  
}

// Nearest distance from point (bx/by) to line (ax/ay),(vx/vy)
bool trajectoryPointDist (double ax, double ay, double vx, double vy,
                          double bx, double by,
                          double& tRet, double& distSRet)
{
  double wx = vy;
  double wy = -vx;
  double dummy = 0.0f;

  int flag = testIntersection (ax, ay, vx, vy, bx, by, wx, wy, tRet, dummy);
 
  if (flag == -3)
  {
    tRet     = 0.0f;
    distSRet = 0.0f;
    return false;
  }

  double nx = ax + tRet*vx;
  double ny = ay + tRet*vy;
  double dx = nx - bx;
  double dy = ny - by;

  distSRet = dx*dx + dy*dy;
  
  return true;
}

double getCollisionTime (double ax, double ay, double vx, double vy,
                         double bx, double by, double ra, double rb)
{
  double minT = MAX_COLL_T;
  int    i    = 0;

  // Check each of nine mirror images of the object 
  // at wrapped positions and take minimum time

  for (i=0; i<9; ++i)
  {
    double wax = ax;
    double way = ay;

    int mi = i % 3;
    int di = i / 3;

    if (mi == 0)
    {
      wax -= GAME_WIDTH;
    }
    else if (mi == 2)
    {
      wax += GAME_WIDTH;
    }

    if (di == 0)
    {
      way -= GAME_HEIGHT;
    }
    else if (di == 2)
    {
      way += GAME_HEIGHT;
    }

    double t = getCollisionTimeSingle (wax, way, vx, vy, bx, by, ra, rb);

    if (t < minT)
    {
      minT = t;
    }
  }

  return minT;
}

double getCollisionTimeSingle (double ax, double ay, double vx, double vy,
                               double bx, double by, double ra, double rb)
{
  double r  = ra + rb;
  double rs = r*r;

  // Test if we are intersecting already (at t=0)
  
  {
    double dx = bx - ax;
    double dy = by - ay;
    double ds = dx*dx + dy*dy;
    
    if (ds <= r*r)
    {
      return 0.0f;
    }
  }

  // Check nearest time and distance
  
  double t  = 0.0f;
  double ds = 0.0f;
  
  if (true != trajectoryPointDist (ax, ay, vx, vy, bx, by, t, ds))
  {
    return MAX_COLL_T;
  }
 
  if (t < 0.0f)
  {
    return MAX_COLL_T;
  }
  
  if (ds > rs)
  {
    // Object never gets near
    return MAX_COLL_T;
  }
  
  // Actual collision time is smaller, because objects have 
  // a nonzero size. Account for that.

  double offLenS = rs - ds;  
  double vs      = vx*vx + vy*vy;
  double offT    = mySqrt (offLenS / vs);
 
  t -= offT;
  
  if (t < 0.0f)
  {
    t = 0.0f;
  }
  
  if (t > MAX_COLL_T)
  {
    // Should not happen, actually
    t = MAX_COLL_T;
  }

  return t;
}

int testIntersection (double ax, double ay, double vx, double vy,
                      double bx, double by, double wx, double wy,
                      double& tRet, double& sRet)
{
  tRet = 0.0f;
  sRet = 0.0f;
  
  double den = wx*vy - wy*vx;  
  double nom = vx * (by-ay) - vy * (bx-ax);

  if (fabs (den) <= 1e-6f)
  {
    return -3;
  }

  double s = nom / den;
  double t = 0.0f;

  if (fabs (vx) > fabs (vy))
  {
    t = (bx - ax + s*wx) / vx;
  }
  else
  {
    t = (by - ay + s*wy) / vy;
  }

  // Intersection on the wrong side
  
  tRet = t;
  sRet = s;

  if (t <= -1e-3f)
  {
    return -2;
  }
    
  if (s < t)
  {
    return -1;
  }
  else if (s > t)
  {
    return 1;
  }

  return 0;
}

int wrapRel (int x, int xRel, int size)
{
  int ret = x;
  
  if (ret - xRel > size/2)
  {
    ret -= size;
  }
  else if (xRel - ret > size/2)
  {
    ret += size;
  }

  return ret;
}

double wrapRel (double x, double xRel, int size)
{
  double ret = x;
  
  if (ret - xRel > double (size/2))
  {
    ret -= size;
  }
  else if (xRel - ret > double (size/2))
  {
    ret += size;
  }

  return ret;
}

int wrapAbs (int x, int size)
{
  int ret = x;
  
  if (ret < 0)
  {
    ret += size;
  }
  else if (ret >= size)
  {
    ret -= size;
  }

  return ret;
}

double wrapAbs (double x, int size)
{
  double ret = x;
  
  if (ret < 0)
  {
    ret += size;
  }
  else if (ret >= size)
  {
    ret -= size;
  }

  return ret;
}

double angleForVec (double x, double y)
{
  double angle = 0.0f;
  
  double ls = x*x+ y*y;
  double l  = mySqrt (ls);

  if (l <= 0.0f)
  {
    return 0.0f;
  }
  
  double xn = x / l;
  double yn = y / l;

  angle = double (myAcos (xn));

  if (yn < 0.0f)
  {
    angle *= -1.0f;
    angle += 2.0f * M_PI;
  }

  return angle;
}

void vecForAngle (double a, double& xRet, double& yRet)
{
  xRet = cos (a);
  yRet = sin (a);
}

double mySqrt (double x)
{
  double eps = 1e-12;
  
  // With paranoia check
  if (x < eps)
  {
    return 0.0;
  }

  return ::sqrt (x);
}

double myAcos (double x)
{
  double eps = 1e-12;
  
  // With paranoia check
  if (x < -1.0 + eps)
  {
    return M_PI;
  }

  if (x > 1.0 - eps)
  {
    return 0.0;
  }

  return ::acos (x);
}

