
#ifndef VEC_MATRIX22_H
#define VEC_MATRIX22_H

//****************************************************************************//
// Forward declarations
//****************************************************************************//

#include "matrix.h"

//****************************************************************************//
// Class declaration
//****************************************************************************//

/*****************************************************************************/
/** The matrix class for 2x2-matrices. Derived from the general matrix class.
*****************************************************************************/

class Matrix22 : public VecMatrix
{

  // constructors/destructor
public:
  ///Default constructor 
  inline Matrix22();

  ///Construct the matrix instance from the matrix a
  inline Matrix22(const Matrix22& a);

  ///Construct the matrix instance from the DclVectors v1,v2 (viewed as column-vectors).
  inline Matrix22(const Vector2& v1, const Vector2& v2);

  ///extracts a 3x3-matrix from M starting with (0,0) if M is big enough, else nothing is done. 
  inline Matrix22(const VecMatrix& M); 

  ///Destructor of the matrix-instance
  inline ~Matrix22();

  // member functions
public:
  //inline void clear(); //this function is already implemented in the base class

  /// Calculates the determinant of the matrix instance.
  virtual inline VecFloat determinant() const;

  /// Sets the matrix instance to the identity.
  virtual inline void identity();

  /// Inverts the matrix instance.
  virtual inline bool inverse();

  /// Multiplies two matrices into the matrix instance.
  inline void multiply(const Matrix22& a1, const Matrix22& a2);


  ///Multiplies the first matrix and the transposed second matrix into the matrix instance.
  inline void multiplyTranspose(const Matrix22& a1, const Matrix22& a2);

  ///Multiplies the first matrix, the transposed second matrix and the vertex into the matrix instance
  inline void multiplyTranspose(const Matrix22& a1, const Matrix22& a2, const Vector2& v);

  ///Multiplies the first vector and the transposed second vector into the matrix instance.
  inline void multiplyTranspose(const Vector2& v1, const Vector2& v2);

  ///Calculates the norm of the matrix instance: $\sum_{ij} m_{ij}^2$.
  inline VecFloat norm() const;

  //inline void transpose(); //implemented in the base class.

  ///Multiplies the transposed first matrix and the second matrix into the matrix instance.
  inline void transposeMultiply(const Matrix22& a1, const Matrix22& a2);

  ///This operator substracts another matrix from the matrix instance.
  inline void operator-=(const Matrix22& a);

  //virtual inline VecFloat& operator()(int row, int column); //implemented in base-class 
  //virtual inline const VecFloat& operator()(int row, int column) const; //implemented in base-class 

  ///This operator adds another matrix to the matrix instance.
  inline void operator+=(const Matrix22& a);

  inline Matrix22 operator*(const VecFloat d); 

  inline Matrix22 operator*(const Matrix22& ); 

  ///This operator multiplies another matrix to the matrix instance.
  inline void operator*=(const Matrix22& a);

  ///This operator multiplies every entry of the matrix instance with the value d.
  inline void operator*=(const VecFloat d);

  ///This operator divides every entry of the matrix instance by the value d. If d=0 no division takes place.
  inline void operator/=(const VecFloat d);

  ///This operator equates the matrix instance with another matrix. Returns a reference to the matrix-instance for concatentation. 
  inline Matrix22& operator=(const Matrix22& v);

  inline Matrix22& operator=(const VecMatrix& M); 


  ///This operator multiplies the matrix instance with the Vector2 v from right and returns the result. The matrix instance is unchanged.
  inline Vector2 operator*(const Vector2& v); 

  ///Multiplies the matrix instance from left with vector v and returns the result. The matrix instance is unchanged. 
  inline friend Vector2 operator*(const Vector2& v, const Matrix22&);

  inline friend Matrix22 transpose(const Matrix22&);

};

// include the inline functions
#include "vector2.h"
#include "matrix22.inl"
#endif

//****************************************************************************//
