;;; -*- Mode: LISP; Syntax: COMMON-LISP; Package: WELTRAUMPUTZE; Base: 10 -*-

(in-package :weltraumputze)

(defun dump-vram (vram)
  (if (not (and (= (aref vram 0) #x01) (or (= (aref vram 1) #xe0) (= (aref vram 1) #xe2))))
      (error "Unexpected begin of vram sequence."))
  (labels
      ((& (int1 int2)
         (boole boole-and int1 int2))
       (read-word (vram pos)
         (boole boole-ior (aref vram pos) (ash (aref vram (1+ pos)) 8)))
       (print-word (word description &rest args)
         (format t "~4,'0x  ~a~%" word (apply #'format nil description args))))
    (let (vy vx vs v1x v1y (ship-detect nil))
      (do ((pos 0 (+ pos 2))) ((> pos 1023))
        (let* ((word0 (read-word vram pos))
               (word1 (read-word vram (+ pos 2)))
               (op (ash word0 -12)))
          (cond
            ((= op #xa) ; LABS
             (setf vy (& word0 #x3ff)
                   vx (& word1 #x3ff)
                   vs (ash word1 -12)
                   pos (+ pos 2))
             (print-word word0 "LABS  vy: ~A" vy)
             (print-word word1 "      vx: ~A  vs: ~A" vx vs))
            ((= op #xb) ; HALT
             (print-word word0 "HALT")
             (return))
            ((= op #xc) ; JSRL
             (let ((address (& word0 #xfff)))
               (cond
                 ((member address '(#x8f3 #x8ff #x90d #x91a))
                  (print-word word0 "JSRL  a: ~A  (asteroid at (~A, ~A) size ~A)" address vx vy vs))
                 ((= address #x929)
                  (print-word word0 "JSRL  a: ~A  (flying saucer at (~A, ~A) size ~A)" address vx vy vs))
                 (t
                  (print-word word0 "JSRL  a: ~A  (unknown subroutine)" address)))))
            ((= op #xd) ; RTSL
             (print-word word0 "RTSL"))
            ((= op #xe) ; JMPL
             (print-word word0 "JMPL"))
            ((= op #xf) ; SVEC
             (print-word word0 "SVEC"))
            (t ; VCTR
             (let ((dy (if (zerop (& word0 #x400)) (& word0 #x3ff) (- (& word0 #x3ff))))
                   (dx (if (zerop (& word1 #x400)) (& word1 #x3ff) (- (& word1 #x3ff))))
                   (vz (ash word1 -12)))
               (cond
                 ((and (zerop dx) (zerop dy) (= vz 15))
                  (print-word word0 "VCTR  dy: ~A  (shot at (~A, ~A))" dy vx vy)
                  (print-word word1 "      dx: ~A  vz: ~A" dx vz)
                  (setf ship-detect nil))
                 ((and (= op 6) (= vz 12) (/= dx 0) (/= dy 0) ship-detect)
                  (print-word word0 "VCTR  dy: ~A  (ship at (~A, ~A), gun to (~A, ~A))" dy vx vy (- v1x dx) (- v1y dy))
                  (print-word word1 "      dx: ~A  vz: ~A" dx vz)
                  (setf ship-detect nil))
                 ((and (= op 6) (= vz 12) (/= dx 0) (/= dy 0) (not ship-detect))
                  (setf v1x dx
                        v1y dy
                        ship-detect t)
                  (print-word word0 "VCTR  dy: ~A  (not interpreted)" dy)
                  (print-word word1 "      dx: ~A  vz: ~A" dx vz))
                 (t
                  (setf ship-detect nil)
                  (print-word word0 "VCTR  dy: ~A  (not interpreted)" dy)
                  (print-word word1 "      dx: ~A  vz: ~A" dx vz)))
               (setf pos (+ pos 2))))
            ))
        ))))

;;; Mathematical functions

;; Returns coordinates relative to the given ship position such that
;; the ship is placed on (0, 0)
(defun cliprect-correction (x y ship-x ship-y)
  (values (- (mod (+ (- x ship-x) 512) 1024) 512) (- (mod (+ (- y ship-y) 384) 768) 384)))

;; Returns the vector representing the closed distance to (0, 0)
;; for the specified line; this vector is always orthogonal to
;; the line. Note: Don't call this function with dx = dy = 0.
(defun shortest-distance-vector (ax ay dx dy)
  (let ((c (/ (- (+ (* ax dx) (* ay dy))) (+ (* dx dx) (* dy dy)))))
    (values (+ ax (* c dx)) (+ ay (* c dy)))))

;; Returns the distance of the two supplied vectors
(defun distance (x1 y1 x2 y2)
  (magnitude (- x2 x1) (- y2 y1)))

;; Returns the magnitude of the supplied vector
(defun magnitude (x y)
  (sqrt (+ (* x x) (* y y))))

;; Returns the number of frames until a collision occurs;
;; returns nil in case this never happens (r = radius)
(defun collision (x1 y1 dx1 dy1 r1 x2 y2 dx2 dy2 r2)
  (let* ((gx (- x2 x1))
         (gy (- y2 y1))
         (mx (- dx2 dx1))
         (my (- dy2 dy1))
         (r (+ r1 r2))
         (a (+ (* mx mx) (* my my)))
         (b (+ (* 2 gx mx) (* 2 gy my)))
         (c (+ (* gx gx) (* gy gy) (- (* r r))))
         (discriminant (- (* b b) (* 4 a c))))
    (cond
      ((= a 0)
       nil) ; avoid division by zero
      ((= discriminant 0)
       (/ (- b) (* 2 a)))
      ((> discriminant 0)
       (let* ((root (sqrt discriminant))
              (sol1 (/ (- (+ b root)) (* 2 a)))
              (sol2 (/ (+ (- b) root) (* 2 a))))
         (cond
           ((and (< sol1 0) (< sol2 0))
            nil)
           ((< sol1 0)
            sol2)
           ((< sol2 0)
            sol1)
           (t
            (min sol1 sol2)))))
      (t
       nil))))
         

;; Returns the shortest distance of the two supplied vectors
;; taking into account the wrap-boundaries of the screen
(defun flat-world-distance (x1 y1 x2 y2)
  (let ((xdist (abs (- x2 x1)))
        (ydist (abs (- y2 y1))))
    (magnitude
     (if (> xdist 512) (- 1024 xdist) xdist)
     (if (> ydist 384) (- 768 ydist) ydist))))

;; Returns x1 - x2, corrected by screen wrapping for x axis
(defun xwrap- (x1 x2)
  (let ((diff (- x1 x2)))
    (cond
      ((> diff 900)
       (- diff 1024))
      ((< diff -900)
       (+ diff 1024))
      (t
       diff))))

;; Returns y1 - y2, corrected by screen wrapping for y axis
(defun ywrap- (y1 y2)
  (let ((diff (- y1 y2)))
    (cond
      ((> diff 650)
       (- diff 768))
      ((< diff -650)
       (+ diff 768))
      (t
       diff))))

;; Returns the angle of the supplied vector
;; in the range [0, 2*pi[
(defun angle (x y)
  (cond
    ((= 0 x)
     (if (< y 0)
         (* 3/2 pi)
         (/ pi 2)))
    ((< x 0)
     (+ (atan (/ y x)) pi))
    (t
     (if (< y 0)
         (+ (atan (/ y x)) pi pi)
         (atan (/ y x))))))

;;; Main program

(defun dump-targets (targets)
  (dolist (target targets)
    (cond
      ((eq (type-of target) 'asteroid)
       (format t "asteroid at (~3D, ~3D) size ~D type #x~X~%" (x target) (y target) (size target) (atype target)))
      ((eq (type-of target) 'flying-saucer)
       (format t "flying saucer at (~3D, ~3D) size ~D~%" (x target) (y target) (size target)))
      (t
       (format t "something else in target list: ~A~%" target))))
  (format t "~%"))

(defun run-from-command-line ()
  (handler-bind ((error #'(lambda (arg) (format t "~A~%" arg) (sb-ext:quit))))
    (if (< (length sb-ext:*posix-argv*) 2)
        (run)
        (run (cadr sb-ext:*posix-argv*)))))

(defun run (&optional (host "127.0.0.1") (port 1979))
  (let ((s (make-instance 'mame-socket :host (make-inet-address host) :port port)))

    ;; send initial command to start vram transmission
    (send-command s 0 0)

    (defparameter *s* s) ; to be removed later
 
    (labels ((frame-no-diff (old new)
               (let ((diff (- new old)))
                 (if (< diff -127)
                     (+ diff 256)
                     (if (> diff 128)
                         (- diff 256)
                         diff))))
             (frame-no (a)
               (aref a 1024))
             (command-no (a)
               (aref a 1025)))
      (let ((last-command-no 0)
            (last-received-command-no -1)
            (last-commands (make-array 256 :initial-element 0))
            (last-frame-no 0)
            (frame-overflow-counter 0)
            (last-scene nil)
            (up-to-date-scene nil)
            (lost-synchronization nil)
            fire-command-available)
        (do* ((time-begin (get-internal-real-time) (get-internal-real-time))
              (vrams (receive-vrams s) (receive-vrams s)))
             (())

          ;; Make some safety checks to handle lost or out-of-order frames
          (if (or (> (- (get-internal-real-time) time-begin) (* 1.2 internal-time-units-per-second))
                  (> (abs (frame-no-diff (frame-no (car vrams)) last-frame-no)) 32))
              ;; We had to wait too long for a working scene history
              ;; or the oldest new frame is too far away from the next frame expected
              (progn
                (format t "Lost synchronization! Waited ~,3F seconds, frame diff was ~A~%"
                        (/ (- (get-internal-real-time) time-begin) internal-time-units-per-second)
                        (frame-no-diff (frame-no (car vrams)) last-frame-no))
                (setf vrams (list (car vrams))) ; only use the (probably) most current frame
                (setf lost-synchronization t)
                (incf frame-overflow-counter))
              ;; Weed out all frames older than the last one we got
              (progn
                (setf vrams (remove-if
                             #'(lambda (a) (< (frame-no-diff last-frame-no a) 1))
                             vrams :key #'frame-no))))

          ;; Create scene instance for each vram received
          (dolist (vram vrams)
            (when (< (frame-no vram) last-frame-no)
              (incf frame-overflow-counter))
            (let ((scene (create-scene-from-vram
                          vram (+ (* frame-overflow-counter 256) (frame-no vram)))))

              ; Match scene objects with the last scene if reasonable to do so
              (unless (or lost-synchronization (null last-scene))
                (continue-scene-sequence scene last-scene))

              ; Initialize new objects within the scene
              (initialize-new-things scene)

              ; Adjust the angle byte according to the last command(s)
              (if (null last-scene)
                  ; In the very first frame, set the angle byte manually
                  (setf (angle-byte scene) 0)
                  ; Otherwise, the angle byte usually results from past scene and
                  ; second recent command
                  (adjust-angle-byte scene
                                     (if lost-synchronization nil last-scene)
                                     (svref last-commands last-received-command-no)))

              ; Check whether a fire command is available
              (setf fire-command-available
                    (and (not lost-synchronization)
                         last-scene
                         (/= (command-no vram) last-received-command-no)
                         (= 1 (- (slot-value scene 'frame-no) (slot-value last-scene 'frame-no)))
                         (not (fire (svref last-commands (command-no vram))))))

              ; Keep track of the last received command no, last scene and last frame
              (setf last-received-command-no (command-no vram))
              (setf last-scene scene)
              (setf last-frame-no (frame-no vram))))

          ;; In case there is a difference between last command sent and the last
          ;; received command no, apply all commands sent since then to cope with
          ;; network latency
          (if (and (not lost-synchronization)
                   (> (frame-no-diff last-received-command-no last-command-no) 0))
              (let ((scene (copy-scene last-scene)))
                (dotimes (n (frame-no-diff last-received-command-no last-command-no))
                  (update-scene scene
                                (svref last-commands (mod (+ last-received-command-no n 1) 256))
                                (svref last-commands (mod (+ last-received-command-no n) 256))))
                  ; Check and set whether fire command is available
                (setf fire-command-available (not (fire (svref last-commands last-command-no))))
                (setf up-to-date-scene scene))
              (setf up-to-date-scene last-scene)) ; Without latency, work on vram derived scene

          ;; Unless we got no new frames, device a new command and send it
          (unless (null vrams)
            (let ((command (win-strategy up-to-date-scene fire-command-available
                                         (svref last-commands last-command-no))))
              ; Store the command in the command history
              (setf last-command-no (mod (1+ last-command-no) 256))
              (setf (svref last-commands last-command-no) command)

              ; Send the command to the server
              (send-command s command last-command-no))
            )

          ;; Clear the synchronization flag at end of loop
          (setf lost-synchronization nil)
          )))))

