package asteroid.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;

import javax.swing.JComponent;

import asteroid.model.AFlyable;
import asteroid.model.Angle;
import asteroid.model.Asteroid;
import asteroid.model.FlyDetect;
import asteroid.model.GameModel;
import asteroid.model.ICompute;
import asteroid.model.Ship;
import asteroid.model.Shot;
import asteroid.model.Ufo;

class ListView extends JComponent implements ICompute {
	private static final long serialVersionUID = -4448378120664860752L;
	private static final int WIDTH = 640;
	private static final int HEIGHT = 480;
	private static final int FONT_SIZE = 9;
	private static final int WIDTH_CHAR = FONT_SIZE;
	private static final int WIDTH_SHORT = 2 * FONT_SIZE;
	private static final int WIDTH_INT = 4 * FONT_SIZE;
	private static final int WIDTH_DOUBLE = 5 * FONT_SIZE;
	private static final int WIDTH_ARRAY = (8 * WIDTH_SHORT) + WIDTH_CHAR;
	private static final int POS_X = 40;
	private static final int POS_Y = POS_X + WIDTH_INT;
	private static final int POS_Z = POS_Y + WIDTH_INT;
	private static final int POS_MOVE_X = POS_Z + WIDTH_INT;
	private static final int POS_MOVE_Y = POS_MOVE_X + WIDTH_INT;
	private static final int POS_ANGLE = POS_MOVE_Y + WIDTH_INT;
	private static final int POS_VELOCITY = POS_ANGLE + WIDTH_DOUBLE;
	private static final int POS_SHIP_DX = POS_VELOCITY + WIDTH_DOUBLE;
	private static final int POS_SHIP_DY = POS_SHIP_DX + WIDTH_INT;
	private static final int POS_STEPS_X = POS_SHIP_DY + WIDTH_INT;
	private static final int POS_STEPS_Y = POS_STEPS_X + WIDTH_ARRAY;
	private static final boolean DRAW_STEPS = false;
	private static final Color TEXT_COLOR = Color.WHITE;
	private FontMetrics mFM;
	private Font mFont = new Font( "Monospace", Font.PLAIN, FONT_SIZE);
	private Graphics mGraph;
	private int mH;
	private GameModel mModel;
	private int mY;

	public ListView( GameModel model) {
		mModel = model;
		Dimension size = new Dimension( WIDTH, HEIGHT);
		setMinimumSize( size);
		setPreferredSize( size);
		setOpaque( true);
		setDoubleBuffered( true);
	}

	void drawArrayLeft( int[] arr, int x) { // 220 8*25 + 20
		x += WIDTH_CHAR;
		mGraph.drawString( "[", x, mY);
		for (int i = 0; i < AFlyable.STEP_SIZE; ++i) {
			x += WIDTH_SHORT;
			drawShortLeft( arr[i], x);
		}
		mGraph.drawString( "]", x, mY);
	}

	void drawDoubleLeft( double d, int x) {
		StringBuffer sb = new StringBuffer();
		sb.append( Math.round( 100.0 * d) / 100.0);
		int pos = sb.indexOf( ".") + 3;
		while (sb.length() < pos) {
			sb.append( '0');
		}
		String s = sb.toString();
		mGraph.drawString( s, x + WIDTH_DOUBLE - mFM.stringWidth( s), mY);
	}

	void drawFlyable( AFlyable ast) {
		drawIntegerLeft( ast.getX8() >> 3, POS_X);
		drawIntegerLeft( ast.getY8() >> 3, POS_Y);
		drawIntegerLeft( ast.getZ(), POS_Z);
		drawIntegerLeft( ast.getMoveX8(), POS_MOVE_X);
		drawIntegerLeft( ast.getMoveY8(), POS_MOVE_Y);
		drawDoubleLeft( Angle.toRoute( ast.getBetha()), POS_ANGLE);
		drawDoubleLeft( ast.getVelocity8() / 8.0, POS_VELOCITY);
		if (DRAW_STEPS) {
			drawArrayLeft( ast.getStepsX(), POS_STEPS_X);
			drawArrayLeft( ast.getStepsY(), POS_STEPS_Y);
		}
	}

	void drawHead() {
		mGraph.drawString( "Objekt", 0, mY);
		drawStringLeft( "PosX", POS_X, WIDTH_INT);
		drawStringLeft( "PosY", POS_Y, WIDTH_INT);
		drawStringLeft( "Type", POS_Z, WIDTH_INT);
		drawStringLeft( "MoveX", POS_MOVE_X, WIDTH_INT);
		drawStringLeft( "MoveY", POS_MOVE_Y, WIDTH_INT);
		drawStringLeft( "Winkel", POS_ANGLE, WIDTH_DOUBLE);
		drawStringLeft( "Speed", POS_VELOCITY, WIDTH_DOUBLE);
		drawStringLeft( "DX", POS_SHIP_DX, WIDTH_INT);
		drawStringLeft( "DY", POS_SHIP_DY, WIDTH_INT);
		if (DRAW_STEPS) {
			drawStringCenter( "StepsX", POS_STEPS_X, WIDTH_ARRAY);
			drawStringCenter( "StepsY", POS_STEPS_Y, WIDTH_ARRAY);
		}
	}

	void drawIntegerLeft( int i, int x) {
		String s = Integer.toString( i);
		mGraph.drawString( s, x + WIDTH_INT - mFM.stringWidth( s), mY);
	}

	void drawShortLeft( int i, int x) {
		String s = Integer.toString( i);
		mGraph.drawString( s, x - mFM.stringWidth( s), mY);
	}

	void drawStringCenter( String s, int x, int width) {
		mGraph.drawString( s, x + ((width - mFM.stringWidth( s)) / 2), mY);
	}

	void drawStringLeft( String s, int x, int width) {
		mGraph.drawString( s, x + width - mFM.stringWidth( s), mY);
	}

	public void forAsteroid( Asteroid ast) {
		mY += mH;
		mGraph.drawString( (ast.isExplosion() ? "Expl " : "Ast ") + ast.getIndex(), 0, mY);
		if (ast.isVisible() || ast.isExplosion()) {
			drawFlyable( ast);
		}
	}

	public void forAsteroid( FlyDetect detect) {
	}

	public void forShip( Ship ship) {
		mY += mH;
		mGraph.drawString( "Ship", 0, mY);
		if (ship.isVisible()) {
			drawFlyable( ship);
			drawIntegerLeft( ship.getDx(), POS_SHIP_DX);
			drawIntegerLeft( ship.getDy(), POS_SHIP_DY);
		}
	}

	public void forShot( FlyDetect detect) {
	}

	public void forShot( Shot shot) {
		mY += mH;
		mGraph.drawString( "Shot " + shot.getIndex(), 0, mY);
		if (shot.isVisible()) {
			drawFlyable( shot);
			if (!shot.isShip()) {
				mGraph.drawString( "Feind", POS_SHIP_DY, mY);
			}
		}
	}

	public void forUfo( Ufo ufo) {
		mY += mH;
		mGraph.drawString( ufo.isExplosion() ? "Expl" : "Ufo", 0, mY);
		if (ufo.isVisible() || ufo.isExplosion()) {
			drawFlyable( ufo);
		}
	}

	public void paint( Graphics g) {
		g.setFont( mFont);
		g.setColor( Color.black);
		g.fillRect( 0, 0, getWidth(), getHeight());
		g.setColor( TEXT_COLOR);
		mGraph = g;
		mFM = g.getFontMetrics( mFont);
		mH = mFM.getHeight();
		mY = mH;
		drawHead();
		mY += mH;
//		drawTest();
		mModel.forShip( this);
		mModel.forUfo( this);
		mY += mH;
		mModel.forAllAsteroids( this);
		mY += mH;
		mModel.forAllShots( this);
	}
}
