package asteroid.model;

import asteroid.Operation;
import asteroid.SimpleLogging;

public class AsteroidVector {
	public static final int MAX_ASTEROIDS = 27;
	int mVanishs, mObjects;
	private Asteroid[] mAsts = new Asteroid[MAX_ASTEROIDS];
	private FlyDetect[] mDetect = new FlyDetect[MAX_ASTEROIDS + 1];
	private boolean mError;
	private Ship mShip;
	private int mSize;
	private Ufo mUfo;

	AsteroidVector( Ship ship, Ufo ufo) {
		mUfo = ufo;
		mShip = ship;
		for (int i = 0; i < MAX_ASTEROIDS; ++i) {
			mAsts[i] = new Asteroid( i);
			mDetect[i] = new FlyDetect();
		}
		mDetect[MAX_ASTEROIDS] = new FlyDetect();
	}

	void addAsteroid( int x, int y, int z, int type) {
		mDetect[mSize++].set8( x << 3, y << 3, z, type, false);
	}

	void addExplosion( int x, int y, int z, int type) {
		mDetect[mSize++].set8( x << 3, y << 3, z, type, true);
	}

	void clearAll() {
		for (int i = 0; i < MAX_ASTEROIDS; ++i) {
			Asteroid ast = mAsts[i];
			ast.mState = AFlyable.STATE_HIDDEN;
			ast.initStep( 0, 0);
		}
	}

	private void createNewAsteroid( Asteroid ast, FlyDetect det) {
		ast.mCreate = 1;
		ast.initStep( 0, 0);
		ast.mType = det.mType;
		ast.setPos8( det.mX8, det.mY8, det.mZ);
		ast.mState = AFlyable.STATE_VISIBLE;
		ast.init( mShip);
		ast.initFire();
	}

	private void createNewExplosion( Asteroid ast, FlyDetect det) {
		if ((ast.mX8 != det.mX8) || (ast.mY8 != det.mY8)) { // ops, Position stimmt nicht
			mError = true;
			SimpleLogging.addLog( "wrong explosion: " + ast.getIndex());
		}
		++mVanishs;
		ast.mVanish = 1;
		ast.mType = det.mType;
		ast.setPos8( det.mX8, det.mY8, ast.mZ); // z vom Asteroiden !!!
		ast.mState = AFlyable.STATE_EXPLOSION;
		ast.mCountDown = Asteroid.EXPLOSION_COUNTDOWN;
		ast.correctCD();
	}

	void detect() {
		if (GameModel.sFrame >= 439) {
			GameModel.sFrame += 0;
		}
		mVanishs = 0;
		int oldSize = mObjects;
		int start = mUfo.detExplosion( (mSize > 0) ? mDetect[0] : null);
		mObjects = (start == 0) ? mSize : (mSize - 1);
		for (int i = 0, j = 0; i < MAX_ASTEROIDS; ++i) {
			Asteroid ast = mAsts[i];
			ast.mVanish = 0;
			switch (ast.getState()) {
				case AFlyable.STATE_HIDDEN:
					break;
				case AFlyable.STATE_VISIBLE:
					++j;
					break;
				case AFlyable.STATE_EXPLOSION:
					ast.mCountDown -= ast.getFrameDistance();
					if (ast.mCountDown <= 0) {
						ast.mState = (ast.mCountDown == 0) ? AFlyable.STATE_BLOCKED : AFlyable.STATE_HIDDEN;
						--oldSize;
					}
					else {
						++j;
					}
					break;
				case AFlyable.STATE_BLOCKED:
					ast.mState = AFlyable.STATE_HIDDEN;
					break;
			}
		}
		if (oldSize > mObjects) { // ops, Ende von Explosion nicht erkannt
			if (mObjects > 0) {
				SimpleLogging.addLog( "missing vanish: " + oldSize + " > " + mObjects);
			}
			while (oldSize > mObjects) { // kille Explosion
				int index = -1;
				int minCD = Asteroid.EXPLOSION_COUNTDOWN;
				for (int i = 0; i < MAX_ASTEROIDS; ++i) {
					Asteroid ast = mAsts[i];
					if (ast.isExplosion() && (ast.mCountDown <= minCD)) {
						index = i;
					}
				}
				if (index < 0) {
					break;
				}
				mAsts[index].mState = AFlyable.STATE_HIDDEN;
				--oldSize;
			}
			for (int i = MAX_ASTEROIDS - 1; i >= 0; --i) { // kille Asteroid
				Asteroid ast = mAsts[i];
				if (ast.isVisible()) {
					ast.mState = AFlyable.STATE_HIDDEN;
					--oldSize;
					if (oldSize == mObjects) {
						break;
					}
				}
			}
		}
		int oldLast = oldSize;
		int newLast = mObjects;
		for (int i = 0; (i < MAX_ASTEROIDS) && (newLast > 0); ++i) {
			FlyDetect det = mDetect[mSize - newLast];
			Asteroid ast = mAsts[i];
			switch (ast.getState()) {
				case AFlyable.STATE_HIDDEN:
					if (oldLast < newLast) {
						createNewAsteroid( ast, det); // neuer Asteroid
						--newLast;
					}
					break;
				case AFlyable.STATE_VISIBLE:
					if (det.mExplosion) {
						createNewExplosion( ast, det); // neue Explosion
					}
					else {
						setOldAsteroid( ast, det); // alter Asteroid
					}
					--oldLast;
					--newLast;
					break;
				case AFlyable.STATE_EXPLOSION:
					setOldExplosion( ast, det); // alte Explosion
					--oldLast;
					--newLast;
					break;
				case AFlyable.STATE_BLOCKED:
					break;
			}
		}
		if (oldLast > 0) { // ops, zu viele alte Objekte
			SimpleLogging.addLog( "missing old: " + oldLast);
		}
		if (newLast > 0) { // ops, zu viele neue Objekte
			SimpleLogging.addLog( "missing new: " + newLast);
		}
	}

	boolean isError() {
		return mError;
	}

	void forAll( ICompute comp) {
		for (int i = 0; i < MAX_ASTEROIDS; ++i) {
			mAsts[i].doIt( comp);
		}
	}

	void forAllDetect( ICompute comp) {
		for (int i = 0; i < mSize; ++i) {
			mDetect[i].doAsteroid( comp);
		}
	}

	Asteroid get( int index) {
		return mAsts[index];
	}

	private void setOldAsteroid( Asteroid ast, FlyDetect det) {
		if (ast.mType != det.mType) { // ops, alter Asteroid stimmt nicht mehr
			mError = true;
			SimpleLogging.addLog( "wrong asteroid: " + ast.getIndex());
		}
		ast.addStep8( det.mX8, det.mY8);
		ast.setPos8( det.mX8, det.mY8, det.mZ);
		ast.mState = AFlyable.STATE_VISIBLE;
		ast.init( mShip);
	}

	private void setOldExplosion( Asteroid ast, FlyDetect det) {
		if ((ast.mX8 != det.mX8) || (ast.mY8 != det.mY8)) { // ops, Position stimmt nicht
			mError = true;
			SimpleLogging.addLog( "wrong explosion: " + ast.getIndex());
		}
		ast.mType = det.mType;
		ast.setPos8( det.mX8, det.mY8, det.mZ);
		ast.correctCD();
	}

	void prepare() {
		mError = false;
		mSize = 0;
	}

	void set( int index, int x8, int y8, int z, int moveX8, int moveY8, int state) {
		Asteroid ast = mAsts[index];
		ast.set( x8, y8, z, moveX8, moveY8, state);
		switch (state) {
			case AFlyable.STATE_VISIBLE:
				ast.mType = Operation.ASTEROID_1;
				break;
			case AFlyable.STATE_HIDDEN:
			case AFlyable.STATE_EXPLOSION:
			case AFlyable.STATE_BLOCKED:
			default:
				ast.mType = Operation.EXPLOSION_0;
		}
	}

	void testAsteroid( int index, int x, int y, int z) {
		Asteroid ast = mAsts[index];
		ast.mType = Operation.ASTEROID_1;
		ast.initStep( 0, 0);
		ast.setPos8( x << 3, y << 3, z);
		ast.mState = AFlyable.STATE_VISIBLE;
		++mObjects;
	}

	void testExplosion( int index, int x, int y, int z, int countDown) {
		Asteroid ast = mAsts[index];
		ast.mType = Operation.EXPLOSION_0;
		ast.initStep( 0, 0);
		ast.setPos8( x << 3, y << 3, z);
		ast.mState = AFlyable.STATE_EXPLOSION;
		ast.mCountDown = countDown;
		++mObjects;
	}

	public int getVanish() {
		return mVanishs;
	}
}
