/* Game playing logic */
/* $Id: player.c,v 1.16 2008/06/15 16:03:41 raap Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

// 2 Includes fr socket()
#include <sys/types.h>
#include <sys/socket.h>
// 2 Includes fr inet_addr()
#include <netinet/in.h>
#include <arpa/inet.h>
// 2 Includes fr fcntl()
#include <unistd.h>
#include <fcntl.h>
// fr memset()
#define INVALID_SOCKET -1
#define WSAGetLastError() errno

#include <math.h>
#include <time.h>
#include <sys/time.h>

#include "vector.h"
#include "frame.h"

#include "player.h"

#include "objects.h"

#include "dir.h"
#include "statistics.h"
#include "aim.h"

/* TODO

	- handle speed change of UFOs (partly done)
	- speculatively shoot at expected appearance area of UFOs
	- shoot more shots at ufos, assuming they change direction in next frame
	- do not shoot at large asteroids very near to us
	- do hyperjump if nothing else helps
	- handle collision avoidance only when relatively near in time
	- do not turn indiscriminately, check if not turning (for one frame maybe) is better
	- ufo and collision must not block each other
	- maybe sync with own shots ?	to do this we need to track own shots
	- better strategy to hit the last few (1 or 2) asteroids of a level
	- set the inital unused subpixels of object to 0.5
	- UFOs change their direction at most at a 128 frame barrier. Keep track of that.
	- optimize all the apparent sizes of the objects for optimal play
	- killing large asteroids effectively
	- shooting while sniping
	- sniping when 1 asteroid alive
	- do not shoot large or middle asteroids when already 26 objects on screen
	- Error in our model: Large and middle asteroids are marked as zombies, but we know there 
	-	will be new asteroids in their vicinity, so we can turn towards them before they are broken up
	- phantom objects should have a birth time and a death time (should they ?) 
	- turn and shoot at the end of a level
	- a tangential shot at a large asteroid might take longer to hit than a later central shot
	- maybe it is better to clear a level than to shoot the appearing UFO
	- seperate SPEED GOOD from dt
	- prefer slow asteroids to fast asteroids
	
Die Lebenszeit eines Projektils
betrgt 18 und wird jedes 4. Frame um eins runtergezhlt (wenn die
interne Framenummer durch 4 teilbar ist), insgesamt also zwischen 69
und 72 Frames, je nach Abschusszeitpunkt.
*/


time_t  t0, t1; /* time_t  defined in <time.h> and <sys/types.h> as long */
clock_t c0, c1; /* clock_t defined in <time.h> and <sys/types.h> as int */
__int64 i0, i1;	/* number of instructions executed */
struct timeval tv0, tv1; /* time by gettimeofday() */

void start_timing(){
 
  i0 = ReadTSC();
  gettimeofday(&tv0, NULL); 
};


void end_timing(){
	i1 = ReadTSC();
	gettimeofday(&tv1, NULL);
	 
	stat_time_per_frame(tv1.tv_sec*1000000+tv1.tv_usec - (tv0.tv_sec*1000000+tv0.tv_usec), i1 - i0); 
}
;


/* Global unique id generator */
int	id = 0;


/* ========================================= History handling ======================================= */
typedef struct {
	object	ship;
	char	keys;
	int	t;
	HitEvent he;
} hist_dat;

#define HIST_LIM 8	


hist_dat history[8];
int	hist_first;
int 	hist_next;

void history_init(){
	hist_first = hist_next = 0;
};

void history_insert(int time, object *ship, HitEvent *he, char keys){
	/* We store the info in the next slot */	
	history[hist_next].t = time;
	history[hist_next].keys = keys;
	history[hist_next].ship = *ship;
	
	hist_next = (hist_next + 1) % HIST_LIM;
	
	/* Overwrite old slot if necessary */
	if (hist_next == hist_first)
		hist_first = (hist_first + 1) % HIST_LIM;
};

/* We want to know something about the past  */
hist_dat *history_recall(int time){
	int i;
	i = hist_next;
	
	while ( i != hist_first){
		if (history[i].t == time){
			return &(history[i]);
		};	
		i = i - 1;
		if (i < 0) i = HIST_LIM -1;
	};
	return NULL;	
};

/* We can answer some questions about the past */

/* Returns +1 if we turned left, -1 if we turned right and 0 if we did not turn at time t
*/
int turned (int time){
	hist_dat *past;
	past = history_recall(time);
	if (past == NULL){			 /* This can happen if we lose a frame, default is no turn TODO better default may be left turn */ 
		return 0; 
	};

	if ((past->keys & KEY_LEFT) && !(past->keys & KEY_RIGHT)){
		return 1;
	};
		
	if ((past->keys & KEY_RIGHT) && !(past->keys & KEY_LEFT)){
		return -1;
	};
	return 0;
};

/* Returns TRUE, if we wanted to fire (i.e. pressed the fire key) at the given time */
BOOL fired (int time){
	hist_dat *past;
	past = history_recall(time);
	
	if (past == NULL){			/* This can happen if we lose a frame */ 
		return FALSE; 
	};
	return (past->keys & KEY_FIRE);
};

HitEvent *hist_hit_event (int time){
	hist_dat *past;
	past = history_recall(time);
	if (past == NULL){			/* This can happen if we lose a frame */ 
		return NULL; 
	};
	return (&(past->he));
};
	

typedef struct {
	int	curtime;	/* Our globally increasing time base (counts frames). Time of last update of game_state */		
	int 	lives;		/* Number of remaining lives */
	int	score;		/* Current score, adjusted for roll over */
	object	ship;		/* our own ship */
	BOOL	ship_present;	/* is ship visible */
	object	ufo;		/* enemy ufo */
	BOOL	ufo_present;	/* is ufo visible */
	
	ObjTab	shots1;		/* used when updating shots information.  */
	ObjTab	shots2;		/* either shots1 or shots2 is the current list of shots */
	ObjTab	*shots;		/* Points to current list of shots */
	ObjTab	*shots_new;	/* Points to newly built list of shots (only used during updating) */	
	
	ObjTab	asteroids1;	/* see shots1 and shots2 */
	ObjTab	asteroids2;	
	ObjTab	*asteroids;	/* Points to current list of asteroids */
	ObjTab	*asteroids_new;	/* Points to newly built list of asteroids (only used during updating) */	
	
	ObjTab	ufos1;		/* see shots1 and shots2 */
	ObjTab	ufos2;	
	ObjTab	*ufos;		/* Points to current list of asteroids */
	ObjTab	*ufos_new;	/* Points to newly built list of asteroids (only used during updating) */	
	
	int	n_own_shots;	/* number of own shots */	
//	HitEvent	hit_event;	/* If we are shooting, this is filled with valuable information */
	int	dead_band;	/* how many frames before the first own shot vanishes */
} GameState;
	
	
/* We maintain a global (?) game state structure */
/* The function game_state_update fills this structure with all currently available information */

GameState state;

/* Prints the current time */
void prt_time(){
	printf("t=%d: ", state.curtime);
};


/* If a larger asteroid is hit, it will break into two 2 pieces 
	Create ghost asteroids, which can be fired at 
	Their life time is very short, because they will quickly move away from the guessed location
	The time given is the time when the shot is fired
*/
void CreateGhosts(object *tar, ObjTab *tab, int time){
	object ast;
	
	if (tar->size == BIG_AST_SIZE){
	
		/* Simply copy the target */
		ast = *tar;
		ast.size = MID_AST_SIZE;
		ast.aura_time = time + 4;
		ast.death_time = -1;
		ast.id = id++;
		insert(tab, &ast);	
		PrtObj(&ast,"Phantom ");
		
		ast.id = id++;
		insert(tab, &ast);	
		PrtObj(&ast,"Phantom ");
	};
	
	if (tar->size == MID_AST_SIZE){
	
		/* Simply copy the target */
		ast = *tar;
		ast.size = SMALL_AST_SIZE;
		ast.aura_time = time + 4;
		ast.death_time = -1;
		ast.id = id++;
		insert(tab, &ast);	
		PrtObj(&ast,"Phantom ");
		
		ast.id = id++;
		insert(tab, &ast);	
		PrtObj(&ast,"Phantom ");
	};
		
};


/* We press the fire button to shoot at the given target at this time
	We set the object to a zombie for a short time, so that we do not use it as a target again
	Later, when we really see the shot, the correct time of death is stored
	If the shot will be blocked, the zombie resurrects after 2 frames
	If it is a large asteroid and we have enough shot capacity,
	we replace it by two small phantom asteroids at its center
	TODO we need to know the time and place of the killing (maybe not)
	TODO resurrection of falsly identified targets not implemented
*/
void expect_to_die(object *tar, HitEvent *he){
	int n, i;
	object ast;
	
	/* We assume that we will surely hit the target, make it a zombie */
	tar->death_time = state.curtime + 2;

	if (is_ufo(tar))
		return;
	
	/* If we have bigger asteroids, they will not really die but break into 2 pieces.
	Create ghost targets, which are only visible for a short time, so that we can try to hit them
	*/
	CreateGhosts(tar, state.asteroids, state.curtime);
};

/* We are pressing the fire button now. Update all necessary information 
	We do not fire into the blue, so we always have a target.
	We also have an expected time when we hit the target
	We really press the fire button here.
	So we should generate the expected shot.
	TODO send a hit event as parameter 
*/
void fire_now(object *tar, int hit_time, KeysPacket *kp){
	object own_shot;	
	
	key_fire(kp, TRUE);
	

	own_shot.cur_time=state.curtime;
	/* TODO this should really be done with wrap_around (see calc_pos) */
	own_shot.cur_xy.x = state.ship.cur_xy.x + (5 * state.ship.dir.dx) / 2;
	own_shot.cur_xy.y = state.ship.cur_xy.y + (5 * state.ship.dir.dy) / 2;
	init_shot(&own_shot);
	own_shot.own = 1;
	own_shot.id = id++;
	own_shot.he.target_id = tar->id;
	own_shot.he.hit_time = state.curtime + 2 + hit_time;
	own_shot.birth_time = state.curtime + 1;
	own_shot.death_time = state.curtime + 2 + hit_time;
	insert (state.shots, &own_shot);
	PrtObj(&own_shot, "Created new ");
	
#ifdef DEBUG	
	if ( (tar->type == AST) || (tar->type == UFO_S) || (tar->type == UFO_L) ){
		printf("%d: Shooting at target id %d, hitting at time %d\n", state.curtime, tar->id, own_shot.he.hit_time);
	};
#endif
	expect_to_die(tar, &(own_shot.he));
	

};

/* A shot has really been fired at the given object
	Set objects time of death to the time given.
*/
void shooting_at(int tar_id, int time){
	int n, i;
	object *obj;
	
	n = state.asteroids->n;
	for (i=0; i<n; i++){
		obj = &state.asteroids->list[i];
		if ( (obj->id == tar_id)  ) {
			obj->death_time = time;	
#ifdef DEBUG			
			printf("Set death time of id %d to %d\n",obj->id, time);
#endif
			return;
		};
	};	
	n = state.ufos->n;
	for (i=0; i<n; i++){
		obj = &state.ufos->list[i];
		if ( (obj->id == tar_id)  ) {
			obj->death_time = time;	
#ifdef DEBUG	
			printf("Set death time of id %d to %d\n",obj->id, time);
#endif
			return;
		};
	};	
};

/* Predict future direction (and position ( TODO at the moment not implemented) ) of the ship 
	at time t
	Uses dir_idx and history[]
	Stores result in dir_idx and dir and updates cur_time
*/
void predict_pos_ship(object *ship, int t){
	int i;
	
	/* for each future frame */
	for (i= ship->cur_time; i<t; i++){
		/* Lets see if we made a turning movement at this frame 
		and adjust the dir_idx accordingly
		*/
		ship->dir_idx = (ship->dir_idx + 3 * turned(i) ) & 0xff;
	};
	ship->dir = *get_dir(ship->dir_idx);
	ship->cur_time = t;
};

/* We update the global state variable so that it's objects represent the situation after time_diff frames.*/
void predict_future(int time_diff) {
	
	/* Estimate expected position of known asteroids */
	predict_pos_list (state.asteroids->list, state.asteroids->n, state.curtime + time_diff);
	
	/* Estimate expected position of ufos */
	predict_pos_list (state.ufos->list, state.ufos->n, state.curtime + time_diff);
		
	/* Estimate expected position of known shots */
	predict_pos_list (state.shots->list, state.shots->n, state.curtime + time_diff);
	
	/* Update our knowledge about the ships true direction */
	if (state.ship_present == TRUE) predict_pos_ship(&state.ship, state.curtime + time_diff);
					
	state.curtime += time_diff;
};


void game_state_init(){
	state.curtime = 0;		/* Start of our lifetime */
	state.lives = 0;
	state.score = 0;
	state.ship_present = FALSE;
	state.ufo_present = FALSE;
	state.shots = &state.shots1;
	state.asteroids = &state.asteroids1;
	state.ufos = &state.ufos1;
	tab_init(state.shots);
	tab_init(state.asteroids);
	tab_init(state.ufos);
	state.ship.dir_idx = 0;		/* This is the default dir_idx */
	state.dead_band = 0;
};

/* This routine is given a Vector RAM frame and a time difference to the previous seen frame 
   It updates the global GameState variable state to match the seen frame. 
   After its execution we have a representation of the GameState variable
   To recognize certain objects (shots, asteroids) we rely on our ability to predict the future.
   We assume that the current state variable represents those objects as already being in their 
   new locations as calculated 1 frame (the normal time difference) before.
   So we simply check if the current expectation matches the VectorRAM visible objects  
*/

void game_state_update(FrameStatus *frame, int time_diff){
	int i;
	object	shot;
	object	ast;
	object	ufo;
	object  ship;	
	int found;
	HitEvent *he;
	
	/* If we have lost some frames, update our current expectations of the objects locations */ 
	if (time_diff > 1) predict_future (time_diff-1);
		
	/* We record the number of lives */
	state.lives = frame->lives;
	
	/* We should get a monotone non-decreasing score as long as the game is running */

//		printf("Current Asteroids Score: %d\n", frame->score);
	/* Difference between our own score counter and the asteroids counter (only 6 significant digits) */
	int diff = frame->score - (state.score % 100000); 
	if ( diff < 0 ) diff += 100000;
	state.score += diff;
//		printf("Real Score: %d\n", state.score);

		
//	printf("\nNew Frame %d: %d shots, %d asteroids\n", state.curtime, frame->nshots, frame->nasteroids);
	/* We find our ship first because other data is meaningless if ship is not present.
	Some data may depend on ship position
	*/
	if (frame->ship_present){
		int old_dir_idx;
		
		ext2int(frame->ship_x, frame->ship_y, &ship.cur_xy);
		ship.cur_time = state.curtime;
		init_ship(&ship);
		if (state.ship_present == FALSE){
			/* Ship was not there in the last frame, 
			    Create a new ship
			    Some speciality: We assume that direction (dir_idx) has not changed,
				because after a crash or hyperjump it does not change
			*/
			old_dir_idx = state.ship.dir_idx;
			state.ship = ship;
			state.ship.dir_idx = old_dir_idx;
		};		
		state.ship_present = TRUE;
				
		/* Update the current position and time of the ship */
		update_pos (&(state.ship), &ship);

	} else {
		state.ship_present = FALSE;
	};
	
	/* ========================= Processing UFOs ================================== */
	state.ufo_present = (frame->nufos > 0);

	/* Let ufos_new point to the unused list of shots */
	if (state.ufos == &(state.ufos1))
		state.ufos_new = &(state.ufos2);
	else
		state.ufos_new = &(state.ufos1);
	
	/* Clear the new table */
	tab_init(state.ufos_new);
	
	/* for each ufo in vector RAM */
	for (i=0; i<frame->nufos; i++){
		ext2int( frame->ufos[i].x, frame->ufos[i].y, &ufo.cur_xy);
		ufo.cur_time = state.curtime;		
		init_ufo(&ufo);
		
		switch (frame->ufos[i].sf)
		{	 
			case 15: // big UFO
				ufo.type = UFO_L;
				ufo.size = 18*8;
				break;
			case 14: // small ufo
				ufo.type = UFO_S;
				ufo.size = 9*8;
				break;	
			default:
#ifdef DEBUG				
				fprintf(stderr,"Error: Unknown UFO size %d\n", frame->ufos[i].sf);
				exit(20);
#endif				
				break;
		};
		
		/* Check if we already have seen this ufo */
		found = find_obj(state.ufos->list, state.ufos->n, &ufo);
		if (found != -1){
			update_pos (&(state.ufos->list[found]), &ufo);
			insert(state.ufos_new, &(state.ufos->list[found]));
			delete(state.ufos, found);
		} else {
//			PrtObj(&ufo, "New ");
			if ((ufo.type == UFO_S) && (ufo.cur_xy.x != 16) && (ufo.cur_xy.x != 8168)){
#ifdef DEBUG				
				printf("totally unexpected UFO x position at time of appearance\n");
//				exit(20);
#endif
			}; 
			ufo.id = id++;
			insert(state.ufos_new, &ufo);
		};
					
	};	
	/* We have processed all ufos. ufos_new is now the current list */
	state.ufos = state.ufos_new;
	
	/* ============================== Processing shots ==================================== */
	/* Let shots_new point to the unused list of shots */
	if (state.shots == &(state.shots1))
		state.shots_new = &(state.shots2);
	else
		state.shots_new = &(state.shots1);
	
	/* Clear the new table */
	tab_init(state.shots_new);
	state.n_own_shots = 0;
	state.dead_band = 99;
	/* for each shot in vector RAM */
	for (i=0; i<frame->nshots; i++){
		ext2int( frame->shots[i].x, frame->shots[i].y, &shot.cur_xy);
		shot.cur_time = state.curtime;
		init_shot(&shot);
		
		/* Check if we already know this shot */
		found = find_obj(state.shots->list, state.shots->n, &shot);
		if (found != -1){
			/* Is this our own shot ? */
			if (state.shots->list[found].own){
				state.n_own_shots++;
				if (state.shots->list[found].death_time < state.curtime ) {
					/* The shot should be dead, but we still see it.
						So this shot probably missed */
#ifdef DEBUG					
					printf("Own shot missed its target!\n");
#endif					
					/* We hope that it will hit next frame */
					state.shots->list[found].death_time = state.curtime;
				};
				if (state.shots->list[found].birth_time == state.curtime) {
#ifdef DEBUG	
					printf("%d: New Own Shot seen\n", state.curtime);
#endif
					he = &(state.shots->list[found].he);
					if (he->target_id != -1){
#ifdef DEBUG
						printf("Target Id = %d\n", he->target_id);
#endif
						shooting_at(he->target_id, he->hit_time);
						state.shots->list[found].death_time = he->hit_time;
					} else {
#ifdef DEBUG
						printf("Invalid hit event\n");
#endif
					};
				};
				state.dead_band = min (state.dead_band, state.shots->list[found].death_time - state.curtime);
			};	
			update_pos (&(state.shots->list[found]), &shot);
			insert(state.shots_new, &(state.shots->list[found]));
			delete(state.shots, found);

		} else {
//			PrtObj(&shot, "New ");
			shot.id = id++;
			insert(state.shots_new, &shot);
		};
					
	};
		
	/* Insert unborn shots into the new list */
	for (i=0; i < state.shots->n; i++){
		if ( state.shots->list[i].birth_time > state.curtime){
			insert(state.shots_new, &(state.shots->list[i]));	
		};
	};
	
	/* We have processed all shots. shots_new is now the current list */
	state.shots = state.shots_new;
	PrtObjList(state.shots);
//	printf("%d own shots\n", state.n_own_shots);
#ifdef DEBUG
	if (state.n_own_shots == 4) printf("Dead Band = %d\n",state.dead_band);	
#endif
	
	/* ============================================ Processing asteroids ========================== */
	/* Let asteroids_new point to the unused list of asteroids */
	if (state.asteroids == &(state.asteroids1))
		state.asteroids_new = &(state.asteroids2);
	else
		state.asteroids_new = &(state.asteroids1);
	
	/* Clear the new table */
	tab_init(state.asteroids_new);

	/* for each seen asteroid */
	for (i=0; i < frame->nasteroids; i++){
		ext2int(frame->asteroids[i].x, frame->asteroids[i].y, &ast.cur_xy);
		ast.cur_time = state.curtime;		
		init_ast(&ast);
	
		ast.form = frame->asteroids[i].type;
		switch (frame->asteroids[i].sf)
		{	// Ungefhrer Radius des Asteroiden 
			case 0:  // groer Asteroid
				ast.size = BIG_AST_SIZE;
				break;
			case 15: // mittlerer Asteroid
				ast.size = MID_AST_SIZE;
				break;
			case 14: // kleiner Asteroid
				ast.size = SMALL_AST_SIZE;
				break;
			default:
#ifdef DEBUG
				fprintf(stderr,"Error: Unknown Asteroid size %d\n", frame->asteroids[i].sf);
				exit(20);
#endif				
				break;
		};

		
		/* Check if we already have seen this asteroid */
		found = find_obj(state.asteroids->list, state.asteroids->n, &ast);
		if (found != -1){
			update_pos (&(state.asteroids->list[found]), &ast);
			insert(state.asteroids_new, &(state.asteroids->list[found]));
			delete (state.asteroids, found);
		} else {
//			PrtObj(&ast, "New ");
			ast.id = id++;
			insert(state.asteroids_new, &ast);
		};
		
	};
	/* Insert ghost objects into this list */
	for (i=0; i < state.asteroids->n; i++){
		if ( state.asteroids->list[i].aura_time > state.curtime){
			insert(state.asteroids_new, &(state.asteroids->list[i]));
			/* TODO maybe update the position of phantom objects */			
		};
	};

	/* We have processed all asteroids. asteroids_new is now the current list */
	
	state.asteroids = state.asteroids_new;	
	
	/* =============== Update speed and direction information ====================== */
	/* calculate new speed and direction values with current information */
	if (state.ship_present) {
		delta visual_dir;
		/* Special for the ship: We can give an estimate of the direction by visual clues */ 
		ext2int_dir(frame->ship_dx, frame->ship_dy, &visual_dir);
		update_speed_ship(&state.ship, &visual_dir);
	};
			
	/* calculate new speed and direction values with current information */
	update_speed_list_ufos(state.ufos->list, state.ufos->n);	

	/* calculate new speed and direction values with current information */
	update_speed_list(state.shots->list, state.shots->n);	
		
	/* calculate new speed and direction values with current information */
	update_speed_list(state.asteroids->list, state.asteroids->n);
};


/* Given a point in time, how many own shots are visible then ? 
	We assume that the death information is accurate
*/
int own_shots(int t){
	int i;
	int n=0;
	
	/* We use the list of last seen shots */
	for (i=0; i < state.shots->n; i++){
		if ( state.shots->list[i].birth_time > t ) continue;	/* unborns don't count */
		if ( state.shots->list[i].death_time < t ) continue;	/* zombies don't count */
		if ( state.shots->list[i].own )	n++;
	};
	return n;
};


/* Returns time of hitting, if we can hit the given target by firing now.
	The shorts originate from the current ship as is.
	This routine considers that we have to release the fire key after each shot
	Returns -1 if no hit
*/
int sure_shot_now (object *tar){
	HitEvent he;
	
	/* Did we fire in the last frame? Then further shots are blocked */
	if (fired (state.curtime - 1)) {
//		printf("Shooting blocked\n");
		return -1;
	};
	
	/* TODO check that there are less than 4 shots */ 
	
	return ( sure_shot(&state.ship, tar, &he) );
};

/* Try to find the asteroid, which is on fastest collision course with our ship 
	Returns id of colliding asteroid, if any
	else returns -1
*/
int find_collision(ObjList list, int n){
	int i, time_to_collision;
	int shortest, best;
	
	shortest = 99999;
	best = -1;
	
	for (i=0; i < n; i++){
		if (list[i].dt < 1) continue;	/* speed not known */
		if (list[i].death_time > state.curtime) continue; 	/* dead man walking */
		if ( (time_to_collision = collision( &(list[i]), &state.ship)) != -1){
			if (time_to_collision < shortest) {
				shortest = time_to_collision;
				best = i;
			};
		};
	}
	if (shortest < 150) return best;
	else return -1;
};	
	

/* Try to find a target, which we can hit if we shoot now 
	Takes into account if fire button was released last frame
	Excludes dead asteroids
	If target is found, HitEvent he is filled out
	*/
object *FindImmTarget(ObjList list, int n, HitEvent *he){
	int i, res;
	for (i=0; i < n; i++){
		if ( (list[i].dt < 1) && (list[i].type == AST)) continue;	/* speed not known */
		if (list[i].death_time > state.curtime) continue; 	/* zombie */
		if ((list[i].type == AST) && (n >= 25) && (list[i].size != SMALL_AST_SIZE)) continue; /* too many asteroids. kill only the small ones */
		if ( (res=sure_shot_now( &(list[i])) ) >= 0) {
			he->target_id = list[i].id;
			he->hit_time = state.curtime + 2 + res;
#ifdef DEBUG			
			printf("%d: ImmediateTarget found target id %d, hitting at time %d\n", state.curtime, list[i].id, he->hit_time);
#endif			
			return &list[i];
		};
	}
	return NULL;
};


void PrtHitEvent(HitEvent *he){
#ifdef DEBUG
	printf("HitEvent: ");
	printf("Hitting %d at %d, turn_time=%d\n", he->target_id, he->hit_time, he->turn_time);	
#endif
};

/* Try to find the target which we can hit the earliest */
object * FindEarliestTarget(ObjTab *tab){
	object *list = tab->list;
	int n = tab->n;
	int i, res, best, td;
	object *res_tar=NULL;
	HitEvent he;
	PrtObjList(tab);
	best = 99999;
	for (i=0; i < n; i++){
		if ((list[i].type == AST) && (n >= 25) && (list[i].size != SMALL_AST_SIZE)) continue; /* too many asteroids. kill the small ones */
		
		/* Zombie UFOs are no targets */
		if ( is_ufo(&(list[i])) &&   (list[i].death_time > state.curtime) ) continue;

		/* Zombie asteroids are no targets */
		if ( ((list[i].dt < 1) && (list[i].type == AST)) || 	/* speed not known */
				     ((list[i].death_time > state.curtime) && (list[i].type == AST)) )	/* zombie */
			continue;

		res = turn_to_target(&state.ship, &(list[i]), &td, &he );
		if (res >= 0)
			PrtHitEvent(&he);
		
		if ( (res >= 0) && (res < best) ){
			best = res;
			res_tar = &(list[i]);
//			state.target_id = list[i].id;
//			state.hit_time = state.curtime + res;
#ifdef DEBUG
			printf("%d: EarliestTarget found target id %d, turning %d and hitting at time %d\n", state.curtime, list[i].id, td, res);
#endif
		};
	}
	if (best < 99999) {
		return res_tar;
	};
	return NULL;
};


/* This routine is called if we could find no valid target to hit 
	Try to find a target to which we can turn to
	We include targets with unknown speed
	We include large asteroids, even if we have already 26 asteroids
	We include targets which are out of range
	We include zombie targets
*/
object * FindTurnToTarget(ObjTab *tab){
	object *list = tab->list;
	int n = tab->n;
	int i, res, best, td;
	object *res_tar=NULL;
	HitEvent he;
	PrtObjList(tab);
	best = 99999;
	for (i=0; i < n; i++){
		
		res = turn_to_target(&state.ship, &(list[i]), &td, &he );
		if ( (res >= 0) && (list[i].dt < 1) ) 	/* speed not known */
			res = 1000 + res;
		
		else if ( (res >= 0) && (list[i].size != SMALL_AST_SIZE) )
			res = 2000 + res;
		
		else if ( (res < 0) && (list[i].death_time <= state.curtime) )
			res = 3000;
		else res = 4000;

		if ( (res >= 0) && (res < best) ){
		     best = res;
		     res_tar = &(list[i]);
#ifdef DEBUG
		     printf("%d: FindTurnToTarget found target id %d, turning %d and hitting at time %d\n", state.curtime, list[i].id, td, res);
#endif
		};
	}
	if (best < 99999) {
		return res_tar;
	};
	return NULL;
};

/* sets key_left or key_right so that we can hit target in the shortest time */
int target_dir (object *tar, KeysPacket *kp){
	int res, td;
	HitEvent he;
	
	key_right(kp, FALSE);
	key_left(kp, FALSE);
	
	res = turn_to_target(&state.ship, tar, &td, &he );
	if (res >= 0){
		if (td == RIGHT){
#ifdef DEBUG
			printf("Turning right\n");
#endif			
			key_right(kp, TRUE);
			
		} else if (td == LEFT) { 
#ifdef DEBUG
			printf("Turning left\n");
#endif
			key_left(kp, TRUE);
		} else 	{
#ifdef DEBUG
			printf("No turning\n");
#endif
	
		};
	};
	return res;
};

/* We have a specific target, which we want to hit 
	Calculate an appropriate action
	either turn or shoot or wait
	Return TRUE if we have shoot now at the target 
	This routine sets state.hit_event in case that we fire.
*/
BOOL snipe(object *tar, KeysPacket *kp)	{
	object future_ship;
	object future_tar;
	int d, res, t, td;
	
//	PrtObj(&state.ship, "Sniping from ");
//	PrtObj(tar, "to ");
	
	/* Can we simply kill it now ? */
	if ( (res=sure_shot_now(tar)) > 0 ){
		fire_now(tar, res, kp);
		return TRUE; 	
	};			
	/* No, we can not shoot now. Turn towards the target */
	res = target_dir(tar, kp);
#ifdef DEBUG
	printf("%d: Snipe turns to target id %d, hitting at time %d\n", state.curtime, tar->id, state.curtime + res);
#endif	
	return FALSE;
};


/* We have an asteroid on crash course (collision imminent)
	Kill it 
*/
BOOL crash_kill (object *tar, KeysPacket *kp)	{

	return snipe (tar, kp);

};


enum strategies {CT, DO_NOTHING, GATHER_STATISTICS, TRY, HIT}; 
/* Implement a given strategy 
	Sets keys in the keypacket
	TODO get rid of third parameter
*/
void Strategy(enum strategies strat, KeysPacket *kp, int t){
	int dist;
	int min_dist = 0x7fffffff;
	int min_dx = 0;
	int min_dy = 0;
	delta dir;
	int col;
	object middle;
	object *res_tar;
	keys_clear(kp);
	HitEvent he;
	

	switch (strat){

		case HIT:
			/* We try to find a target, which should be destroyed next.
				Later on we decide if we can shoot now at this target or later.
				If we can shoot now, we have to find the next target to turn to.
			*/
			res_tar = NULL;
			
			if  (state.ufos->n > 0)  {
				/* We do have at least one UFO */
				res_tar = FindEarliestTarget(state.ufos);
				if (res_tar != NULL){ 
#ifdef DEBUG
					printf("Sniping UFO\n");
#endif
				};
			};
			
			if (res_tar == NULL){
				col = find_collision(state.asteroids->list, state.asteroids->n);
				if (col != -1){
					res_tar = &(state.asteroids->list[col]);
#ifdef DEBUG
					printf("Crash Kill %d\n", col);
#endif					
				};
			};

			if (res_tar == NULL){
				res_tar = FindEarliestTarget(state.asteroids);
			};
			
			if (res_tar != NULL){
				/* Now we have a target to kill */
				if (snipe(res_tar, kp)){
					/* Snipe wants to shoot, find a new target to turn to */
					res_tar = FindEarliestTarget(state.asteroids);
					if (res_tar != NULL) target_dir(res_tar, kp);
				} else {
					/* Snipe simply turns toward the target. See if we have enough shots
						available and if we can hit another target
					*/
//					if ( ( ((res_tar->type == AST) && (res_tar->size == SMALL_AST_SIZE) ) || 
//								(res_tar->type == UFO_S) || (res_tar->type == UFO_L) )
//						&& (state.n_own_shots < 3) )
					
					if ( NULL != (res_tar = FindImmTarget(state.asteroids->list, state.asteroids->n, &he)) ){
						fire_now(res_tar, he.hit_time - 2 - state.curtime, kp);
					};

				};	
				break;
			} else {
#ifdef DEBUG
				printf("%d: Finding target to turn to\n", state.curtime);
#endif					
				res_tar = FindTurnToTarget(state.asteroids);
				if (res_tar != NULL) { 
					target_dir(res_tar, kp);
					break;
				};			
			};
			
			/* default action */
			key_left(kp,true);

			break;

		case DO_NOTHING:

			break;
		default:
		break;	
	};
};

/* Prepare for a new game. Resets necessary data structures */
void new_game(){
	
	history_init();		
	game_state_init();
};

enum game_status {NOT_STARTED, RUNNING, GAME_OVER, TIME_UP};

/* Global countdown timer */
int countdown;

/* Starting value for countdown timer (number of frames) */
//#define MAX_FRAMES (300 * 60)
//#define MAX_FRAMES (10 * 60)
#define MAX_FRAMES (500 * 60)
	
static void start_countdown(){
	countdown = MAX_FRAMES;
};

static void dec_countdown(int diff){
	countdown -= diff;
};

void LostLife(){
	PrtStatistics();
};

void GameOver(KeysPacket *kp){
	LostLife();
	key_start(kp,true);
};


/* Diff is the time difference between this frame and the last one. Normally 1. */	
void play(FrameStatus *frame_status, KeysPacket *kp, int diff)
{	
#ifdef DEBUG	
	start_timing();
#endif
	/* We have just received a new frame. Update all our information so that it reflects the true state of the game */
	game_state_update(frame_status, diff);
		
	/* All our strategy decisions are based on the state of the game after 1 frame,
		because this is the time when our key presses are accepted by the asteroids engine.
		We therefore predict the future state of the game after 1 frame (and hope that latency is not larger)
		(The respective actions are only visible two frames from the last frame received.)
	*/
	predict_future(1);
	/* Now all further computations are based on the future state of the game */

	/* Decide wich keys (if any) to press */
	Strategy(HIT, kp, state.curtime);
		
	/* We remember some information about current state */
	history_insert( state.curtime, &state.ship, NULL, kp->keys);
#ifdef DEBUG	
	end_timing();
#endif
};

	
void Run(Player *p) {
	FramePacket frame;
	KeysPacket kp;
	FrameStatus frame_status;
	char prevframe = 0;
	int diff, rcv_ok;
	enum game_status status;
	
	new_game();
	InitStatistics();

	init_keys(&kp);
	
	/* We send and receive dummy packets to initialize prevframe and ping. */
	SendPacket(p, &kp);
	ReceivePacket(p, &frame);
	prevframe = frame.frameno;
	
	status = NOT_STARTED;
	for (;;){
		SendPacket(p, &kp);
		rcv_ok = ReceivePacket(p, &frame);
		if (rcv_ok == 0){	// the game is over, time is up
			stat_game_over(state.score);
			PrtStatistics();
			exit(0);
		};
		/* compute the difference between previous received frameno and new frameno */
		diff = (unsigned char) frame.frameno - (unsigned char) prevframe;
		if (diff < 0){
			/* byte must have rolled over */
			diff += 256;
		};
		if (diff == 0){
#ifdef DEBUG			
			printf("WARNING: Frame repeated.\n");
#endif		
		};
		if (diff > 1){
#ifdef DEBUG			
			printf("WARNING: Lost %d Frame(s).\n", diff-1);
#endif			
			stat_frame_loss(diff-1);
		};	
		prevframe = frame.frameno;
		
		if (frame.ping != kp.ping){
#ifdef DEBUG			
			printf("Latenz %d.\n", kp.ping - frame.ping);
#endif		
		};

		InterpretScreen(&frame, &frame_status);

		switch (status){
			case NOT_STARTED:
				if (! frame_status.ship_present) {
					key_start(&kp,TRUE);
					break;
				};
				status = RUNNING;
				start_countdown();
				/* Fall thru ! */
			case RUNNING:
				play(&frame_status, &kp, diff);
				dec_countdown(diff);
				if (countdown <= 0) {
					stat_game_over(state.score);
					PrtStatistics();
					status = TIME_UP;
				} else if (frame_status.lives == 0) {
					stat_game_over(state.score);
					PrtStatistics();
					status = GAME_OVER;
				};
				break;
				
			case TIME_UP:
				if ( (unsigned char)frame.vectorram[1] == 0xe0 )
					key_fire(&kp,TRUE);
				else	
					key_fire(&kp,FALSE);
				key_thrust(&kp,TRUE);
				if (frame_status.lives == 0)
					status = GAME_OVER;
				break;
			
			case GAME_OVER:
				/* Well our game is over */
#ifdef DEBUG
				printf("Game Over\n");
#endif				
				PrtStatistics();
				new_game();
				status = NOT_STARTED;
				break;
				
		};		

	};	
};
	

void init_keys(KeysPacket *kp)
{
	kp->signature[0] = 'c';
	kp->signature[1] = 't';
	kp->signature[2] = 'm';
	kp->signature[3] = 'a';
	kp->signature[4] = 'm';
	kp->signature[5] = 'e';
	kp->keys = '@';
	kp->ping = 0;
}

void keys_clear(KeysPacket *kp)
{
	kp->keys = '@';
}

void key_hyperspace(KeysPacket *kp, BOOL b)
{
	if (b)
		kp->keys |= KEY_HYPERSPACE;
	else
		kp->keys &= ~KEY_HYPERSPACE;
}

void key_fire(KeysPacket *kp, BOOL b)
{
	if (b)
		kp->keys |= KEY_FIRE;
	else
		kp->keys &= ~KEY_FIRE;
}

void key_thrust(KeysPacket *kp, BOOL b)
{
	if (b)
		kp->keys |= KEY_THRUST;
	else
		kp->keys &= ~KEY_THRUST;
}

void key_left(KeysPacket *kp, BOOL b)
{
	if (b){
		kp->keys |= KEY_LEFT;
		kp->keys &= ~KEY_RIGHT;
	}
	else{
		kp->keys &= ~KEY_LEFT;
	}
}

void key_right(KeysPacket *kp, BOOL b)
{
	if (b){
		kp->keys |= KEY_RIGHT;
		kp->keys &= ~KEY_LEFT;
	}
	else {
		kp->keys &= ~KEY_RIGHT;
	}
};

void key_start(KeysPacket *kp, BOOL b)
{
	if (b)
		kp->keys |= KEY_START;
	else
		kp->keys &= ~KEY_START;
}
		
/* Returns 0 iff error occured */
int ReceivePacket(Player *p, FramePacket *packet)
{
//	struct sockaddr_in sender;
//	int sender_size = sizeof (sender);
	fd_set readfds, writefds, exceptfds;

	do
	{
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(p->sock, &readfds);
		FD_SET(p->sock, &exceptfds);
		select(p->sock+1, &readfds, &writefds, &exceptfds, NULL);
		int bytes_received = recv(p->sock, (char *)packet, sizeof(FramePacket), 0);
		if (bytes_received != sizeof(FramePacket))
		{
			int err = WSAGetLastError();
//			fprintf(stderr, "Fehler %d bei recvfrom().\n", err);
			return 0;
		}
		FD_ZERO(&readfds);
		FD_ZERO(&writefds);
		FD_ZERO(&exceptfds);
		FD_SET(p->sock, &readfds);
		struct timeval zero;
		zero.tv_sec = zero.tv_usec = 0;
		select(p->sock+1, &readfds, &writefds, &exceptfds, &zero);
	} while(FD_ISSET(p->sock, &readfds));
	return 1;
}

void SendPacket(Player *p, KeysPacket *packet)
{
	struct sockaddr_in server; /* server address */
	
	/* Construct the server address structure */
	memset(&server, 0, sizeof(server));    /* Zero out structure */
	server.sin_family = AF_INET;                 /* Internet addr family */
	server.sin_addr.s_addr = p->server_ip;  /* Server IP address */
	server.sin_port   = htons(1979);     /* Server port */	
	
	/* Send the string to the server */
	
	packet->ping++; // jedes gesendete Pckchen erhlt eine individuelle Nummer zur Latenzmessung

	if (sizeof(KeysPacket) != sendto(p->sock, (char *)packet, sizeof(KeysPacket), 0, (struct sockaddr*)&server, sizeof server))
	{
		if (errno != EAGAIN)
		{
			perror("Fehler bei sendto()");
			exit(1);
		}

	}
}
