// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2005  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.util;

import java.awt.*;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

/**
 *  A wrapper for the event queue which catches all uncaught exceptions
 *  and distributes them to {@link EventQueueExceptionListener}s.
 *  This should only be used as a singleton, and it is indirectly used by
 *  the {@link Utility#addEventQueueExceptionListener(EventQueueExceptionListener)}
 *  method.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
class EventQueueExceptionWrapper
        extends EventQueue
{
  /** The {@link EventQueueExceptionListener}s. */
  private Collection listeners = new LinkedList();

  /**
   *  Constructor.
   */
  public EventQueueExceptionWrapper()
  {
    Toolkit.getDefaultToolkit().getSystemEventQueue().push(this);
  }

  /**
   * Dispatches an event. The manner in which the event is
   * dispatched depends upon the type of the event and the
   * type of the event's source object:
   * <p> </p>
   * <table border=1 summary="Event types, source types, and dispatch methods">
   * <tr>
   * <th>Event Type</th>
   * <th>Source Type</th>
   * <th>Dispatched To</th>
   * </tr>
   * <tr>
   * <td>ActiveEvent</td>
   * <td>Any</td>
   * <td>event.dispatch()</td>
   * </tr>
   * <tr>
   * <td>Other</td>
   * <td>Component</td>
   * <td>source.dispatchEvent(AWTEvent)</td>
   * </tr>
   * <tr>
   * <td>Other</td>
   * <td>MenuComponent</td>
   * <td>source.dispatchEvent(AWTEvent)</td>
   * </tr>
   * <tr>
   * <td>Other</td>
   * <td>Other</td>
   * <td>No action (ignored)</td>
   * </tr>
   * </table>
   * <p> </p>
   *
   * @param event an instance of <code>java.awt.AWTEvent</code>,
   *              or a subclass of it
   */
  protected void dispatchEvent(AWTEvent event)
  {
    try {
      super.dispatchEvent(event);
    } catch (Throwable t) {
      for (Iterator iterator = listeners.iterator(); iterator.hasNext();) {
        EventQueueExceptionListener listener = (EventQueueExceptionListener)iterator.next();
        listener.exceptionOccured(event, t);
      }
    }
  }

  /**
   *  Add an exception listener which is called when an exception occurs during the
   *  dispatch of an AWT event.
   *  @param listener listener to add
   */
  public void addEventQueueExceptionListener(EventQueueExceptionListener listener)
  {
    listeners.add(listener);
  }

  /**
   *  Remove an exception listener which was called when an exception occurs during the
   *  dispatch of an AWT event.
   *  @param listener listener to remove
   */
  public void removeEventQueueExceptionListener(EventQueueExceptionListener listener)
  {
    listeners.remove(listener);
  }
}
