// strategy.cpp
// ------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#include "stdafx.hpp"

#include "strategy.hpp"

#include "../algo.hpp"
#include "../comm_model.hpp"

#include "ast_cossin.hpp"
#include "ast_random.hpp"
#include "screen_state.hpp"

#define foreach BOOST_FOREACH

namespace intrepid
{
    namespace
    {
        bool can_shoot(game_state const &gs, int frameno, int &dist,
                       int &object)
        {
            if (gs.num_shots < 4)
            {
                game_state temp_gs = gs;
                temp_gs.advance_fire(comm_model::keys::fire);
                temp_gs.advance_ship(comm_model::keys::none);
                temp_gs.advance_move();
                shot *sht = temp_gs.shot_fired;
                if (sht)
                {
                    sht->asteroid_collision = 0;
                    for (int n = 0; n < 62; ++n)
                    {
                        ++ frameno;
                        if (sht->lifetime == 0)
                        {
                            if (sht->asteroid_collision)
                            {
                                object = sht->asteroid_collision
                                         - &temp_gs.asteroids[0];
                                dist = n;
                            }
                            else
                            {
                                object = 27;
                                dist = n;
                            }
                            return true;
                        }
                        temp_gs.advance(comm_model::keys::none);
                    }
                }
            }
            return false;
        }
    }

    uint8_t strategy::advance_gs_until_shot_available(game_state &gs)
    {
        uint8_t steps = 0;
        do
        {
            gs.advance_ship(comm_model::keys::none);
            gs.advance_move();
            gs.advance_collisions();
            gs.ast_rand();
            ++ gs.frame_count;
            ++ steps;
        }
        while (gs.num_shots >= 4 && steps <= 64);
        return steps;
    }

    bool strategy::eval_target(game_state gs, int object, int angle, int dist)
    {
        int size = 0;
        if (object < 27)
        {
            asteroid const &ast = gs.asteroids[object];
            size = ast.size;
        }

        int value = 180 - 2 * angle - dist;
        if (object == 27)
        {
            value = asteroid_mass_ < 3 ? 1 : 1000000;
        }
        if (target_captured_)
        {
            if (!target_.alert_level && value > target_.value)
            {
                target_.type = object;
                target_.action = action_shoot;
                target_.value = value;
                return true;
            }
        }
        else
        {
            target_captured_ = true;
            target_.type = object;
            target_.value = value;
            target_.action = action_shoot;
            target_.alert_level = 0;
            return true;
        }
        return false;
    }

    uint8_t strategy::capture_new_target()
    {
        game_state gs = next_game_state_;
        int count = 0;
        if (gs.num_shots >= 4)
        {
            gs.advance_move();
            gs.advance_collisions();
            gs.ast_rand();
            ++ gs.frame_count;
            ++ count;
        }
        int const frameno = level_frame_count_ + count;
        count = 42;
        uint8_t keys = comm_model::keys::none;
        uint8_t ship_angle = gs.ship.angle;
        for (int n = 0; n <= count; ++n)
        {
            int dist, object;
            gs.ship.angle = uint8_t(ship_angle + 3 * n);
            if (can_shoot(gs, frameno, dist, object))
            {
                if (eval_target(gs, object, n, dist) && n)
                {
                    keys = comm_model::keys::left;
                }
            }
            gs.ship.angle = uint8_t(ship_angle - 3 * n);
            if (can_shoot(gs, frameno, dist, object))
            {
                if (eval_target(gs, object, n, dist) && n)
                {
                    keys = comm_model::keys::right;
                }
            }
        }
        return keys;
    }

    uint8_t strategy::compute_next_keys()
    {
        next_game_state_.advance_collisions();
        next_game_state_.ast_rand();
        ++ next_game_state_.frame_count;
        if (will_get_shot())
        {
            return comm_model::keys::hyperspace;
        }
        uint8_t keys = fire_now() ? comm_model::keys::fire
                                  : comm_model::keys::none;
        will_get_killed();
        next_game_state_.advance_fire(keys);
        keys |= move_ship();

        point const center(4096, 3072);
        point const center_dir = torus_diff(center,
                                            next_game_state_.ship.position);
        point const ship_dir = ast_cossin(next_game_state_.ship.angle);

        bool saucer_escape = false;
        if (next_game_state_.saucer.present)
        {
            point const saucer_dir =
                                torus_diff(next_game_state_.saucer.position,
                                           next_game_state_.ship.position);
            int const dp = dot_product(ship_dir, saucer_dir);
            if (-8000000 < dp && dp < 8000000)
            {
                saucer_escape = true;
            }
        }

        int speed_limit = 2000000;
        int const speed = norm2(next_game_state_.ship.velocity);
        if (asteroid_mass_ == 0 && !next_game_state_.saucer.present)
        {
            point const ship_vel = next_game_state_.ship.velocity >> 8;
            point const dir = (center_dir >> 5) - ship_vel;
            if (dir.x * ship_dir.y - dir.y * ship_dir.x < 0)
            {
                keys = comm_model::keys::left;
            }
            else
            {
                keys = comm_model::keys::right;
            }
            speed_limit = 20000000;
            if (speed < speed_limit
                || dot_product(next_game_state_.ship.velocity, ship_dir) < 0)
            {
                keys |= comm_model::keys::thrust;
            }
        }
        else
        {
            point const ship_pos = next_game_state_.ship.position;
            point barycenter(0, 0);
            int mass = 0;
            foreach (asteroid const &ast, next_game_state_.asteroids)
            {
                if (ast.lifetime == 100)
                {
                    int m;
                    switch (ast.size)
                    {
                    case 2:
                        m = 7;
                        break;
                    case 1:
                        m = 3;
                        break;
                    default:
                        m = 1;
                    }
                    mass += m;
                    barycenter += m * torus_diff(ast.position, ship_pos);
                }
            }
            if (mass > 0)
            {
                barycenter /= mass;
                if (dot_product(ship_dir, barycenter) > 0
                    && speed < speed_limit)
                {
                    keys |= comm_model::keys::thrust;
                }
            }
        }

        if (saucer_escape && speed < 40000000)
        {
            keys |= comm_model::keys::thrust;
        }
        return keys;
    }

    void strategy::compute_next_move(int      thread_nr,
                                     uint8_t &keys,
                                     int     &rating)
    {
        if (thread_nr == 0)
        {
            if (!screen_state_.frame_present())
            {
                keys = 0;
                rating = 1;
                return;
            }

            game_state_.adjust(screen_state_);

            compute_asteroid_mass();
            if (!random_reconstructed_ && level_frame_count_ < 32)
            {
                reconstruct_random();
            }
            if (game_state_.saucer.present)
            {
                random_reconstructed_ = false;
                game_state_.rand_seed = -1;
            }
            if (!game_state_.ship.present)
            {
                random_reconstructed_ = false;
                game_state_.rand_seed = -1;
            }

            next_game_state_ = game_state_;
            comm_model::ping_to_keys_map ping_to_keys
                                    = screen_state_.frame_->get_ping_to_keys();
            uint8_t ping = screen_state_.frame_->get_ping();
            uint8_t next_ping = screen_state_.frame_->get_next_ping();
            for (uint8_t p = ping; p != next_ping; ++p)
            {
                next_game_state_.advance(ping_to_keys[p]);
            }
            keys = compute_next_keys();
            rating = 1;
        }
    }

    void strategy::compute_asteroid_mass()
    {
        int mass = 0;
        foreach (asteroid const &ast, game_state_.asteroids)
        {
            if (ast.lifetime == 100)
            {
                switch (ast.size)
                {
                case 0:
                    mass += 1;
                    break;
                case 1:
                    mass += 3;
                    break;
                case 2:
                    mass += 7;
                    break;
                default:
                    assert(false);
                }
            }
        }
        if (mass > asteroid_mass_)
        {
            // new level
            start_asteroid_mass_ = mass;
            level_frame_count_ = 0;
            random_reconstructed_ = false;
        }
        else
        {
            ++ level_frame_count_;
        }
        asteroid_mass_ = mass;
    }

    bool strategy::fire_now()
    {
        int  dist, object;
        if (can_shoot(next_game_state_, 0, dist, object)
            && (target_captured_
            && object == target_.type
            && target_.action == action_shoot) && level_frame_count_ >= 9)
        {
            return true;
        }
        return false;
    }

    bool strategy::match_random(int n, int &steps)
    {
        int const max_steps = 32;
        game_state::asteroids_array asteroids = game_state_.asteroids;
        bool start_pos_found = false;
        int k;
        for (k = 0; k < max_steps && !start_pos_found; ++k)
        {
            start_pos_found = true;
            foreach (asteroid &ast, asteroids)
            {
                if (ast.lifetime != 100)
                {
                    assert(ast.lifetime == 0);
                    break;
                }
                ast.position -= ast.velocity;
                ast.position.torus_wrap();
                if ((ast.position.x & -ast.precision_x) != 0
                    && (ast.position.y & -ast.precision_y) != 0)
                {
                    start_pos_found = false;
                }
            }
        }
        steps = k;
        if (!start_pos_found)
        {
            return false;
        }

        ast_random::sequence const &rand_sequence = ast_random::get_sequence();
        foreach (asteroid const &ast, asteroids)
        {
            if (ast.lifetime != 100)
            {
                break;
            }
            if (((rand_sequence[n] >> 3) & 0x03) != ast.shape)
            {
                return false;
            }
            point start_pos = ast.position;

            int rnd = rand_sequence[n + 6];
            if ((start_pos.x & -ast.precision_x) == 0)
            {
                if ((start_pos.y & -ast.precision_y) == 0)
                {
                    continue;
                }
                if ((rnd & 1) != 1)
                {
                    return false;
                }
                rnd = (rnd >> 1) & 0x1F;
                if (rnd >= 0x18)
                {
                    rnd &= 0x17;
                }
                if ((start_pos.y >> 8) != rnd)
                {
                    return false;
                }
            }
            else
            {
                if ((rnd & 1) != 0)
                {
                    return false;
                }
                rnd = (rnd >> 1) & 0x1F;
                if ((start_pos.x >> 8) != rnd)
                {
                    return false;
                }
            }

            n += 7;
        }
        return true;
    }

    uint8_t strategy::move_ship()
    {
        uint8_t keys = comm_model::keys::none;
        if (target_captured_)
        {
            if (!target_reachable(keys))
            {
                target_captured_ = false;
            }
        }
        if (!target_captured_)
        {
            keys = capture_new_target();
        }
        return keys;
    }

    void strategy::reconstruct_random()
    {
        if (random_reconstructed_
            || asteroid_mass_ != start_asteroid_mass_
            || asteroid_mass_ < 6 * 7)
        {
            return;
        }

        unsigned int key = 0;
        unsigned int key_ub = 0xFFFF;
        uint8_t  k = 0;
        foreach (asteroid const &ast, game_state_.asteroids)
        {
            if (ast.lifetime != 100)
            {
                break;
            }
            if (!ast.tracked)
            {
                return;
            }
            if (k < 8)
            {
                key |= ast.shape << (7 - k) * 2;
                key_ub >>= 2;
            }
            ++ k;
        }
        key_ub = (key_ub | key) + 1;

        ast_random::index const &rand_index = ast_random::get_index();
        ast_random::index::const_iterator it_begin
                            = lower_bound(rand_index,
                                          ast_random::index_entry(key, 0));
        ast_random::index::const_iterator it_end
                            = upper_bound(rand_index,
                                          ast_random::index_entry(key_ub, 0));

        ast_random::sequence const &rand_sequence = ast_random::get_sequence();
        int seed = -1;
        int steps = 0;
        for (ast_random::index::const_iterator it = it_begin;
             it != it_end;
             ++it)
        {
            int n = it->value;
            int k;
            if (match_random(n, k))
            {
                if (random_reconstructed_)
                {
                    random_reconstructed_ = false;
                    break;
                }
                random_reconstructed_ = true;
                seed = n;
                steps = k;
            }
        }
        game_state_.rand_seed = random_reconstructed_
                                ? seed + asteroid_mass_ + steps - 1 : - 1;
    }

    bool strategy::target_reachable(uint8_t &keys)
    {
        if (target_.type <= 27 && target_.action == action_shoot)
        {
            game_state gs_left = next_game_state_;
            game_state gs_right = next_game_state_;
            for (int n = 0; n < 43; ++n)
            {
                int dist, object;
                gs_left.advance_ship(comm_model::keys::left);
                gs_left.advance_move();
                gs_left.advance_collisions();
                gs_left.ast_rand();
                ++ gs_left.frame_count;
                if (can_shoot(gs_left, 0, dist, object)
                    && object == target_.type)
                {
                    keys = comm_model::keys::left;
                    return true;
                }

                gs_right.advance_ship(comm_model::keys::right);
                gs_right.advance_move();
                gs_right.advance_collisions();
                gs_right.ast_rand();
                ++ gs_right.frame_count;
                if (can_shoot(gs_right, 0, dist, object)
                    && object == target_.type)
                {
                    keys = comm_model::keys::right;
                    return true;
                }
            }
        }
        return false;
    }

    void strategy::will_get_killed()
    {
        game_state gs = next_game_state_;
        if (gs.ship.present)
        {
            gs.advance_ship(comm_model::keys::none);
            gs.advance_move();
            point ship_pos = gs.ship.position;
            for (int n = 0; n < 100; ++n)
            {
                for (int k = 0; k < 27; ++k)
                {
                    if (gs.asteroids[k].is_near_ship(ship_pos))
                    {
                        if (target_captured_)
                        {
                            int alert = 1000 - n;
                            if (target_.alert_level < alert)
                            {
                                target_.alert_level = alert;
                                target_.action = action_shoot;
                                target_.type = k;
                            }
                        }
                        else
                        {
                            target_captured_ = true;
                            target_.action = action_shoot;
                            target_.alert_level = 1000 - n;
                            target_.type = k;
                        }
                    }
                }
                gs.advance(comm_model::keys::none);
            }
        }
    }

    bool strategy::will_get_shot()
    {
        if (next_game_state_.ship.present)
        {
            game_state gs = next_game_state_;
            point const ship_pos = next_game_state_.ship.position;
            gs.advance_move();
            gs.advance_move();
            foreach (shot const &sht, gs.shots)
            {
                if (sht.lifetime)
                {
                    point diff = torus_diff(sht.position, ship_pos);
                    int dist = norm2(diff);
                    if (dist < 100 * 100 && dot_product(sht.velocity,
                                                        diff) < 0)
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

} // end of namespace intrepid
