// screen_state.cpp
// ----------------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#include "stdafx.hpp"

#include "screen_state.hpp"

#include "angles.hpp"
#include "ast_cossin.hpp"

#define foreach BOOST_FOREACH

namespace intrepid
{
    inline int8_t explosion_max_lifetime(int explo_type, int sc)
    {
        static int8_t const life_time_map[4][16] =
            {{15, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 35, 32, 28, 23},
             {12, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 37, 99, 31, 27, 21},
             { 8, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 34, 30, 26, 19},
             { 4, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 36, 33, 29, 24, 17}};
        assert(explo_type >= vram_object_type::explosion1);
        assert(explo_type <= vram_object_type::explosion4);
        assert(sc >= 0 && sc < 16);
        int8_t ret = life_time_map[explo_type
                                   - vram_object_type::explosion1][sc];
        assert(ret >= 0 && ret < 50);
        return ret + 1;
    }

    inline bool is_asteroid(vectorram_object const &vram_obj)
    {
        return vram_obj.obj_type >= vram_object_type::asteroid1 &&
               vram_obj.obj_type <= vram_object_type::asteroid4;
    }

    inline bool is_explosion(vectorram_object const &vram_obj)
    {
        return vram_obj.obj_type >= vram_object_type::explosion1 &&
               vram_obj.obj_type <= vram_object_type::explosion4;
    }

    inline bool is_saucer(vectorram_object const &vram_obj)
    {
        return vram_obj.obj_type == vram_object_type::saucer;
    }

    inline bool is_ship(vectorram_object const &vram_obj)
    {
        return vram_obj.obj_type == vram_object_type::ship;
    }

    inline bool is_shot(vectorram_object const &vram_obj)
    {
        return vram_obj.obj_type == vram_object_type::shot;
    }

    inline int8_t size_from_sf(int sf)
    {
        static int8_t size_tab[16] = {2, 0, 0, 0,
                                      0, 0, 0, 0,
                                      0, 0, 0, 0,
                                      0, 0, 0, 1};
        return size_tab[sf];
    }

// struct screen_asteroids

    void screen_asteroid::reset()
    {
        position = point(0, 0);
        velocity = point(0, 0);
        lifetime = 0;
        precision_x = precision_y = 0;
        shape = 0;
        size = 0;
        tracked = false;
        prev_screen_positions.clear();
    }

// struct screen_ship

    inline void ship_velocity_slow_down(point &velocity)
    {
        if (velocity.x > 0)
        {
            velocity.x -= ((velocity.x >> 7) & -2) + 1;
        }
        else if (velocity.x < 0)
        {
            velocity.x -= ((velocity.x >> 7) & -2);
        }
        if (velocity.y > 0)
        {
            velocity.y -= ((velocity.y >> 7) & -2) + 1;
        }
        else if (velocity.y < 0)
        {
            velocity.y -= ((velocity.y >> 7) & -2);
        }
    }

    void screen_ship::accelerate(uint8_t keys)
    {
        if (keys & comm_model::keys::thrust)
        {
            velocity += 2 * ast_cossin(min_angle);
        }
        else
        {
            ship_velocity_slow_down(velocity);
        }
        if (velocity.x >= 0x4000)
        {
            velocity.x = 0x3FFF;
        }
        else if (velocity.x < -0x4000)
        {
            velocity.x = -0x3FFF;
        }
        if (velocity.y >= 0x4000)
        {
            velocity.y = 0x3FFF;
        }
        else if (velocity.y < -0x4000)
        {
            velocity.y = -0x3FFF;
        }
    }

    void screen_ship::move()
    {
        position += velocity >> 8;
        position.torus_wrap();
    }

    void screen_ship::reset()
    {
        position = point(4192, 3168);
        velocity.x &= 255;
        velocity.y &= 255;
        present = false;
    }

// class screen_state
// public:

    void screen_state::next_frame(parsed_frame const &frame)
    {
        frame_ = &frame;
        uint8_t current_frameno = frame.get_frameno();
        frame_diff_ = first_frame_ ? 1 : byte_diff(current_frameno,
                                                   frameno_byte_);
        if (frame_diff_ > 0)
        {
            frame_skip_ = frame_diff_ - 1;
            if (frame_skip_)
            {
                cout << "WARNING: " << int(frame_skip_)
                     << " frames lost!" << endl;
            }
            latency_ = byte_diff(frame_->get_next_ping(), frame_->get_ping());
            if (latency_ != 2)
            {
                cout << "WARNING: Latency " << latency_ << '!' << endl;
            }

            track_objects();

            frameno_byte_ = current_frameno;
            first_frame_ = false;
        }
    }

    void screen_state::reset()
    {
        first_frame_ = true;
        frameno_ = -1;
        foreach (screen_asteroid &ast, asteroids_)
        {
            ast.reset();
        }
    }

// private:

    void screen_state::adjust_ship(vectorram_object const &vram_obj)
    {
        // Do nothing in the first frame the ship appears.
        if (!frame_->ship_present())
        {
            ship_.present = false;
        }
        if (ship_.present)
        {
            for (uint8_t ping = frame_->get_ping() - frame_diff_;
                 ping != frame_->get_ping();
                 ++ping)
            {
                uint8_t const keys = frame_->get_ping_to_keys()[ping];
                adjust_ship_angle(keys);
                adjust_ship_velocity(keys, vram_obj);
            }
        }
        else
        {
            ship_.reset();
        }
        ship_.present = frame_->ship_present();
    }

    void screen_state::adjust_ship_angle(uint8_t keys)
    {
        // left button has priority
        if (keys & comm_model::keys::left)
        {
            ship_.min_angle += 3;
            ship_.max_angle += 3;
        }
        else if (keys & comm_model::keys::right)
        {
            ship_.min_angle -= 3;
            ship_.max_angle -= 3;
        }
        uint8_t min_a, max_a;
        angles_from_ship_orientation(frame_->get_ship_orientation(),
                                     min_a, max_a);
        ship_.min_angle = max_byte(ship_.min_angle, min_a);
        ship_.max_angle = min_byte(ship_.max_angle, max_a);
        if (byte_less_than(ship_.max_angle, ship_.min_angle))
        {
            ship_.min_angle = min_a;
            ship_.max_angle = max_a;
        }
    }

    void screen_state::adjust_ship_velocity(uint8_t keys,
                                            vectorram_object const &vram_obj)
    {
        if (frame_->even())
        {
            ship_.accelerate(keys);
        }
        ship_.move();
        rect pos = rect::from_vram_pos(vram_obj.pos);
        static double last_x_correct = 0;
        static double last_y_correct = 0;
        bool x_corrected = false;
        bool y_corrected = false;
        if (torus_diff_x(ship_.position.x, pos.tr.x) > 0)
        {
            ship_.position.x = pos.tr.x;
            ship_.velocity.x -= last_x_correct < 0.2 ? 1 : 16;
            x_corrected = true;
        }
        else if (torus_diff_x(pos.bl.x, ship_.position.x) > 0)
        {
            ship_.position.x = pos.bl.x;
            ship_.velocity.x += last_x_correct < 0.2 ? 1 : 16;
            x_corrected = true;
        }
        last_x_correct = (15 * last_x_correct + x_corrected) / 16;
        if (torus_diff_y(ship_.position.y, pos.tr.y) > 0)
        {
            ship_.position.y = pos.tr.y;
            ship_.velocity.y -= last_y_correct < 0.2 ? 1 : 16;
            y_corrected = true;
        }
        else if (torus_diff_y(pos.bl.y, ship_.position.y) > 0)
        {
            ship_.position.y = pos.bl.y;
            ship_.velocity.y += last_y_correct < 0.2 ? 1 : 16;
            y_corrected = true;
        }
        last_y_correct = (15 * last_y_correct + y_corrected) / 16;
    }

    void screen_state::adjust_existing_asteroid(screen_asteroid &ast,
                                              vectorram_object const &vram_obj)
    {
        if (is_explosion(vram_obj))
        {
            ast.position += frame_skip_ * ast.velocity;
            ast.position.torus_wrap();
            ast.lifetime = min(explosion_max_lifetime(vram_obj.obj_type,
                                                      vram_obj.s),
                               ast.lifetime);
        }
        else if (ast.tracked)
        {
            ast.position += frame_diff_ * ast.velocity;
            ast.position.torus_wrap();
        }
        else if (frame_skip_)
        {
            return adjust_new_asteroid(ast, vram_obj);
        }
        else
        {
            ast.prev_screen_positions.push_back(vram_obj.pos);
            assert(ast.prev_screen_positions.size() >= 2);
            assert(ast.prev_screen_positions.size() <= 9);
            ast.velocity = point(0, 0);

            ast.velocity = 8 * vram_torus_diff(
                                    ast.prev_screen_positions.back(),
                                    ast.prev_screen_positions.front())
                           / (ast.prev_screen_positions.size() - 1);
            if (ast.prev_screen_positions.size() == 9)
            {
                ast.position = point::from_vram_pos(
                                            ast.prev_screen_positions.front());
                for (int n = 1; n < 9; ++n)
                {
                    point vpos = ast.prev_screen_positions[n];
                    point next_pos = torus_wrap(ast.position + ast.velocity);
                    ast.position = torus_max(point::from_vram_pos(vpos),
                                             next_pos);
                }
                static int8_t precisions[8] = {8, 1, 2, 1, 4, 1, 2, 1};
                ast.precision_x = precisions[ast.velocity.x & 7];
                ast.precision_y = precisions[ast.velocity.y & 7];
                ast.tracked = true;
            }
            else
            {
                ast.position = point::from_vram_pos(vram_obj.pos);
            }
        }
        if (point::to_vram_pos(ast.position) != vram_obj.pos)
        {
            return adjust_new_asteroid(ast, vram_obj);
        }
    }

    void screen_state::adjust_new_asteroid(screen_asteroid &ast,
                                           vectorram_object const &vram_obj)
    {
        ast.position = point::from_vram_pos(vram_obj.pos);
        ast.velocity = point(0, 0);
        ast.lifetime = is_explosion(vram_obj) ?
                            explosion_max_lifetime(vram_obj.obj_type,
                                                   vram_obj.s)
                            :   100;
        ast.precision_x = 8;
        ast.precision_y = 8;
        ast.shape = static_cast<int8_t>(vram_obj.obj_type
                                        - vram_object_type::asteroid1);
        ast.size = size_from_sf(vram_obj.s);
        ast.tracked = false;
        ast.prev_screen_positions.clear();
        ast.prev_screen_positions.push_back(vram_obj.pos);
    }

    void screen_state::track_objects()
    {
        parsed_frame::iterator const vbegin = frame_->begin();
        parsed_frame::iterator const vend = frame_->end();

        parsed_frame::iterator vobj = vbegin;

        // Shots
        array<screen_shot, 6>::iterator shot = shots_.begin();
        while (vobj != vend && is_shot(*vobj))
        {
            shot->position = point::from_vram_pos(vobj->pos);
            shot->present = true;
            ++ shot;
            ++ vobj;
        }
        while (shot != shots_.end())
        {
            shot->present = false;
            shot ++;
        }

        // Saucer
        if (vobj != vend && (is_saucer(*vobj) || is_explosion(*vobj)))
        {
            saucer_.present = is_saucer(*vobj);
            point pos = point::from_vram_pos(vobj->pos);
            saucer_.velocity = pos - saucer_.position;
            saucer_.velocity.x = min(saucer_.velocity.x, 16);
            saucer_.velocity.y = min(saucer_.velocity.y, 16);
            saucer_.velocity.x = max(saucer_.velocity.x, -16);
            saucer_.velocity.y = max(saucer_.velocity.y, -16);
            saucer_.position = pos;
            saucer_.size = size_from_sf(vobj->s);
            ++ vobj;
        }
        else
        {
            saucer_.present = false;
        }

        if (frameno_ >= 0)
        {
            ++ frameno_;
        }

        // Ship
        if (vobj != vend && is_ship(*vobj))
        {
            adjust_ship(*vobj++);

            if (frameno_ < 0)
            {
                frameno_ = 0;
            }
        }

        // Asteroids
        int const nof_asteroids = static_cast<int>(vend - vobj);
        int nof_current_asteroids = 0;
        foreach (screen_asteroid &ast, asteroids_)
        {
            if (ast.lifetime && ast.lifetime < 100)
            {
                ast.lifetime = static_cast<int8_t>(max(ast.lifetime
                                                       - frame_skip_, 0));
            }
            if (ast.lifetime > 1)
            {
                ++ nof_current_asteroids;
            }
        }
        int nof_new_asteroids = nof_asteroids - nof_current_asteroids;

        foreach (screen_asteroid &ast, asteroids_)
        {
            if (vobj == vend)
            {
                ast.reset();
            }
            else
            {
                if (!ast.lifetime)
                {
                    // Empty slot.
                    // New asteroids always occupy
                    // the first few available slots.
                    if (nof_new_asteroids > 0)
                    {
                        adjust_new_asteroid(ast, *vobj++);
                        -- nof_new_asteroids;
                    }
                }
                else if (ast.lifetime > 1)
                {
                    adjust_existing_asteroid(ast, *vobj++);
                }
            }
            if (ast.lifetime && ast.lifetime < 100)
            {
                -- ast.lifetime;
            }
            assert(ast.lifetime >= 0);
        }
    }
} // end of namespace intrepid
