#ifndef TRACKINGOBJECT_H_
#define TRACKINGOBJECT_H_

#include <list>
#include "Tools.h"
#include "FlyingObjectType.h"

template<class V>
class TrackingObject
{
protected:
	list<V> history;
	wxPoint flightVector;
	int frameID;
	int lifeTime;
	int lastFrameID;
	bool exactFlightVector;
	int minX;
	int minY;
	int maxX;
	int maxY;
	int objectID;
	
	bool estimatedFlightVector;
public:
	TrackingObject()
	{
		frameID = 0;
		exactFlightVector = false;
		flightVector.x = 0;
		flightVector.y =0;
		minX=minY = 0;
		maxX=maxY = 7;
		lifeTime = -1;
		lastFrameID = -1;
		objectID = -1;
		
		estimatedFlightVector = false;
	}
	
	TrackingObject(V& startPoint,int objectID)
	{
		this->frameID = 0;
		exactFlightVector = false;
		flightVector= wxPoint(0,0);
		minX=minY = 0;
		maxX=maxY = 7;
		lifeTime = -1;
		lastFrameID = -1;
		
		estimatedFlightVector = false;
		
		this->objectID = objectID;
	
		insertSnapshot(startPoint);
	}
	
	TrackingObject(const TrackingObject& ref)
	{
		history = ref.history;
		frameID = ref.frameID;
		flightVector = ref.flightVector;
		exactFlightVector = ref.exactFlightVector;
		minX = ref.minX;
		minY = ref.minY;
		maxX = ref.maxX;
		maxY = ref.maxY;
		lifeTime = ref.lifeTime;
		lastFrameID = ref.lastFrameID;
		
		objectID = ref.objectID;
		estimatedFlightVector= ref.estimatedFlightVector;
	}
	
	virtual ~TrackingObject()
	{
		/* alternatives Matching....(nicht eindeutig genug)
		typename list<V*>::iterator iterator = history.begin();
		
		while(iterator != history.end())
		{
			delete *iterator;
			iterator++;
		}*/
	}
	
	virtual void updateFlightVector();
	
	virtual void insertSnapshot(V& snapshot);
	
	void insertSnapshot(V& snapshot,int objectID);
	
	virtual int matchTo(V& snapshot) ;
	
	wxPoint getFlightVector()
	{
		return flightVector;
	}
	
	wxPoint getCurrentPosition()
	{
		return wxPoint(minX,minY);
	}
	
	wxPoint getStart()
	{
		return (--history.end())->getPos();
	}
	
	V& getCurrentSnapshot()
	{
		return *history.begin();
	}
	
	int getFrameID()
	{
		return frameID;
	}
	
	FlyingObjectType getType()
	{
		return (*history.begin()).getType();
	}
	
	void clear()
	{
		/*typename list<V*>::iterator iterator = history.begin();
		
		while(iterator != history.end())
		{
			delete *iterator;
			iterator++;
		}*/
		history.clear();
		exactFlightVector = false;
		frameID = 0;
		flightVector.x = 0;
		flightVector.y = 0;
		minX = 0;
		minY = 0;
		maxX = 7;
		maxY = 7;

		lifeTime = -1;
		lastFrameID = -1;
		objectID = -1;
		estimatedFlightVector =false;
	}
	
	bool isUnused()
	{
		return history.size() == 0;
	}
	
	bool isUsed()
	{
		return history.size() != 0;
	}
	
	bool isExactFlightVector()
	{
		return exactFlightVector;
	}

	int getObjectID()
	{
		return objectID;
	}
	
	void setEstimatedFlightVector(const wxPoint& flightVector)
	{
		this->flightVector = flightVector;
		estimatedFlightVector= true;
	}
	
	int getLifeTime()
	{
		return lifeTime;
	}
};

template<class V>
void TrackingObject<V>::insertSnapshot(V& snapshot,int objectID)
{
	this->objectID = objectID;
	insertSnapshot(snapshot);
}

template<class V>
void TrackingObject<V>::insertSnapshot(V& snapshot)
{
	int framesDiff = 0;
	//FIXME: wenn hier der erste Teil der Bedingung war ist, wäre framesDiff uninitialisiert; stelle erstmal auf 0
	if(history.size() == 0 || (framesDiff = Tools::getFrameDiff(this->frameID,snapshot.getFrameID())) >= 0)
	{
		this->frameID = snapshot.getFrameID();
		
		if(lifeTime == -1)
		{
			lastFrameID = frameID;
			lifeTime = 0;
		}
		else
		{
			lifeTime += Tools::getFrameDiff(lastFrameID,frameID);
			lastFrameID = frameID;
		}
		
		minX += (flightVector.x*framesDiff);
		maxX += (flightVector.x*framesDiff);
		
		minY += (flightVector.y*framesDiff);
		maxY += (flightVector.y*framesDiff);
		
		if(minX - snapshot.getPos().x >= 0 && minX-snapshot.getPos().x < 8)
		{
			if(maxX > snapshot.getPos().x +7)
			{
				maxX = snapshot.getPos().x + 7;
			}
		}
		else if(maxX - snapshot.getPos().x >= 0 && maxX -snapshot.getPos().x < 8)
		{
			if(minX< snapshot.getPos().x)
				minX = snapshot.getPos().x;
		}
		else
		{
			minX = snapshot.getPos().x;
			maxX = minX + 7;
		}
		
		if(minY - snapshot.getPos().y >= 0 && minY-snapshot.getPos().y < 8)
		{
			if(maxY > snapshot.getPos().y +7)
			{
				maxY = snapshot.getPos().y + 7;
			}
		}
		else if(maxY - snapshot.getPos().y >= 0 && maxY -snapshot.getPos().y < 8)
		{
			if(minY< snapshot.getPos().y)
				minY = snapshot.getPos().y;
		}
		else
		{
			minY = snapshot.getPos().y;
			maxY = minY + 7;
		}
		
		
		if(this->flightVector.x % 2 == 1|| flightVector.y %2 == 1)
		{
			//wxLogDebug("min(%d,%d) max(%d,%d) flight(%d,%d)",minX,minY,maxX,maxY,flightVector.x,flightVector.y);
		}
				
		history.push_front(snapshot);
				
		if(!exactFlightVector)
			updateFlightVector();
	}
}

template<class V>
void TrackingObject<V>::updateFlightVector()
{
	int maxDistance=-1;
	typename list<V>::const_iterator start =history.begin();
	typename list<V>::const_iterator end;
	
	while(start != history.end())
	{
		for(end = start,end++;end != history.end();end++)
		{
			if((start->getFrameID()- end->getFrameID())% 8 == 0)
			{
				flightVector = start->getPos() - end->getPos();
				flightVector.x /= Tools::getFrameDiff(end->getFrameID(),start->getFrameID());
				flightVector.y /= Tools::getFrameDiff(end->getFrameID(),start->getFrameID());
				
				exactFlightVector = true;
				return;
			}
			else if(!estimatedFlightVector && (start->getFrameID() - end->getFrameID() > maxDistance))
			{

				flightVector = start->getPos() - end->getPos();
				flightVector.x /= Tools::getFrameDiff(end->getFrameID(),start->getFrameID());
				flightVector.y /= Tools::getFrameDiff(end->getFrameID(),start->getFrameID());
			
				maxDistance = Tools::getFrameDiff(end->getFrameID(),start->getFrameID());
			}
		}
		start++;
	}
}


template<class V>
int TrackingObject<V>::matchTo(V& snapshot)
{
	wxPoint currentPoint = Tools::calcPosition(wxPoint(minX,minY),flightVector,Tools::getFrameDiff(frameID,snapshot.getFrameID()));
	
	currentPoint.x = currentPoint.x&0xfffffff8;
	currentPoint.y = currentPoint.y&0xfffffff8;
	
	currentPoint = Tools::minDistance(currentPoint - snapshot.getPos());
	
	int distance = currentPoint.x*currentPoint.x + currentPoint.y*currentPoint.y;
	
	if(distance <= 8192)
		return distance;
		
	return -1;
}


#endif /*TRACKINGOBJECT_H_*/
