package doertbrexel;

import doertbrexel.spaceobjects.Asteroid;
import doertbrexel.spaceobjects.FlyingSaucer;
import doertbrexel.spaceobjects.Shot;
import doertbrexel.spaceobjects.SpaceObject;
import doertbrexel.spaceobjects.SpaceShip;
import doertbrexel.util.Vector2D;

public class GameStatus
{
	private static final int	TYPE_SPACE_SHIP		= 0;
	private static final int	TYPE_FLYING_SAUCER	= 1;
	private static final int	TYPE_ASTEROID		= 2;
	private static final int	TYPE_SHOT			= 3;

	private static class History
	{
		private static final int	HISTORY_DEPTH	= 10;

		boolean						mFirstRound		= true;
		int							mHistoryPointer	= 0;
		GameStatus[]				mHistory		= new GameStatus[HISTORY_DEPTH];

		public void add(GameStatus pGameStatus)
		{
			mHistory[mHistoryPointer++] = pGameStatus;
			if (mHistoryPointer == HISTORY_DEPTH)
			{
				mHistoryPointer = 0;
				mFirstRound = false;
				//mHistory[mHistoryPointer] = null;
			}
		}

		public GameStatus get(int pIndex)
		{
			int historyPointer = mHistoryPointer - 1 + pIndex;
			while (historyPointer < 0)
				historyPointer += HISTORY_DEPTH;
			while (historyPointer >= HISTORY_DEPTH)
				historyPointer -= HISTORY_DEPTH;
			return mHistory[historyPointer];
		}

	}

	private static History			sHistory				= new History();

	private int[]					mSpaceObjectsPointer	=
															{0, 0, 0, 0};		// Pointer auf die Arrays
	private final SpaceObject[][]	mSpaceObjects			=
															{new SpaceShip[1],
			new FlyingSaucer[1], new SpaceObject[Constants.MAX_ASTEROIDS],
			new Shot[Constants.MAX_SHOTS]					};
	private int						mLatency				= 0;

	public GameStatus()
	{
		sHistory.add(this);
	}

	public GameStatus getPrevious(int pIndex)
	{
		return sHistory.get(pIndex);
	}

	public class SpaceObjectsIterator
	{
		int	mPointer	= 0;

		private SpaceObjectsIterator()
		{

		}

		public boolean hasNext(int i)
		{
			return mPointer < mSpaceObjectsPointer[i];
		}

		public SpaceObject next(int i)
		{
			return (mSpaceObjects[i][mPointer++]);
		}
	}

	public class AsteroidsIterator extends SpaceObjectsIterator
	{
		public boolean hasNext()
		{
			return mPointer < mSpaceObjectsPointer[TYPE_ASTEROID];
		}

		public SpaceObject next()
		{
			return (SpaceObject) super.next(TYPE_ASTEROID);
		}
	}

	public class ShotsIterator extends SpaceObjectsIterator
	{
		public boolean hasNext()
		{
			return mPointer < mSpaceObjectsPointer[TYPE_SHOT];
		}

		public SpaceObject next()
		{
			return (SpaceObject) super.next(TYPE_SHOT);
		}
	}

	public void addShot(final int pX, final int pY)
	{
		addSpaceObject(TYPE_SHOT, new Shot(pX, pY));
	}

	public void addAsteroid(final int pX, final int pY, final int pAsteroidType,
			final int pScalingFactor)
	{
		addSpaceObject(TYPE_ASTEROID, new Asteroid(pX, pY, pAsteroidType, pScalingFactor));
	}

	public void setSaucer(final int pX, final int pY, final int pS)
	{
		addSpaceObject(TYPE_FLYING_SAUCER, new FlyingSaucer(pX, pY, pS));
	}

	public void setSpaceShip(final int pX, final int pY, final int pDx, final int pDy)
	{
		addSpaceObject(TYPE_SPACE_SHIP, new SpaceShip(pX, pY, pDx, pDy));
	}

	private void addSpaceObject(int pType, SpaceObject pSpaceObject)
	{
		if (mSpaceObjectsPointer[pType] < mSpaceObjects[pType].length
				&& mSpaceObjects[pType][mSpaceObjectsPointer[pType]] == null)
		{
			mSpaceObjects[pType][mSpaceObjectsPointer[pType]++] = pSpaceObject;
		}
	}

	public AsteroidsIterator getAsteroidsIterator()
	{
		return new AsteroidsIterator();
	}

	public ShotsIterator getShotsIterator()
	{
		return new ShotsIterator();
	}

	public boolean isShipPresent()
	{
		return mSpaceObjectsPointer[TYPE_SPACE_SHIP] > 0;
	}

	public SpaceShip getShip()
	{
		return (SpaceShip) mSpaceObjects[TYPE_SPACE_SHIP][0];
	}

	public boolean isSaucerPresent()
	{
		return mSpaceObjectsPointer[TYPE_FLYING_SAUCER] > 0;
	}

	public SpaceObject getFlyingSaucer()
	{
		return mSpaceObjects[TYPE_FLYING_SAUCER][0];
	}

	public void evaluate()
	{
		GameStatus previousGameStatus = sHistory.get(-1);
		if (previousGameStatus != null)
		{
			for (int objectTypeIndex = 0; objectTypeIndex < mSpaceObjects.length; objectTypeIndex++)
			{
				for (int previousSpaceObjectsPointer = 0; previousSpaceObjectsPointer < mSpaceObjectsPointer[objectTypeIndex]; previousSpaceObjectsPointer++)
				{
					SpaceObject actualSpaceObject = mSpaceObjects[objectTypeIndex][previousSpaceObjectsPointer];
					switch (objectTypeIndex)
					{
						case TYPE_SPACE_SHIP :
						case TYPE_FLYING_SAUCER :
							if (previousSpaceObjectsPointer < previousGameStatus.mSpaceObjectsPointer[objectTypeIndex])
							{
								actualSpaceObject
										.calculateVelocity(
												previousGameStatus.mSpaceObjects[objectTypeIndex][previousSpaceObjectsPointer],
												mLatency);
							}
							break;

						case TYPE_ASTEROID :
							SpaceObject bestFitPreviousSpaceObject = findBestFitSpaceObject(
									previousGameStatus, actualSpaceObject,
									objectTypeIndex, previousSpaceObjectsPointer);
							if (bestFitPreviousSpaceObject != null)
								actualSpaceObject.calculateVelocity(
										bestFitPreviousSpaceObject, mLatency);
							break;
					}
				}
			}

			if (isSaucerPresent() && previousGameStatus.isSaucerPresent())
			{
				getFlyingSaucer().calculateVelocity(previousGameStatus.getFlyingSaucer(),
						0);

			}
			if (isShipPresent() && previousGameStatus.isSaucerPresent())
			{
				getShip().calculateVelocity(previousGameStatus.getShip(), 0);

			}
		}
	}

	private SpaceObject findBestFitSpaceObject(GameStatus previousGameStatus,
			SpaceObject actualSpaceObject, int pTypeIndex, int pActualSpaceObjectPointer)
	{
		SpaceObject bestFitPreviousSpaceObject = null;
		SpaceObject previousSpaceObject = null;
		int minSquaredDistanceError = Integer.MAX_VALUE;
		int squaredDistanceError = 0;
		// Fr alle Typen von SpaceObjects
		// Nur gleiche Typen vergleichen (gleicher pTypeIndex)
		for (int k = 0; k < previousGameStatus.mSpaceObjectsPointer[pTypeIndex]; k++)
		{
			if (true || k != pActualSpaceObjectPointer)
			{
				previousSpaceObject = previousGameStatus.mSpaceObjects[pTypeIndex][k];
				// nur gleich grosse Objekte kommen in Frage
				if (previousSpaceObject.mType == actualSpaceObject.mType)
				{
					squaredDistanceError = previousSpaceObject.mLocation.add(
							previousSpaceObject.mVelocity).substract(
							actualSpaceObject.mLocation).squaredLength();
					if (squaredDistanceError < minSquaredDistanceError)
					{
						minSquaredDistanceError = squaredDistanceError;
						bestFitPreviousSpaceObject = previousSpaceObject;
					}
				}
			}
		}
		return bestFitPreviousSpaceObject;
	}
}
