/* common.c $Id: common.c,v 1.3 2000/08/31 22:05:14 angus Exp $
 *
 * (C) Angus J. C. Duggan, 1996-2000
 *
 * Common parts for LPR, LPQ, and LPRM client for Windows NT; uses registry
 * and PRINTER environment variable for printer, accepts piped input.
 * PRINTERHOST registry key and environment variable used for LPD host.
 */

#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <assert.h>
#include <malloc.h>

#include "common.h"

void *xalloc(void *ptr, size_t size)
{
  void *result = realloc(ptr, size) ;

  if ( result == NULL )
    error("%s: out of memory!\n", program) ;

  return result ;
}

#define MAX_FORMAT 10 /* Max format specification length */

static void fmtout(char *format, va_list args)
{
  int c ;

  assert(format != NULL) ;
  assert(program != NULL) ;

  while ( (c = (unsigned char)*format++) != 0 ) {
    if ( c == '%' ) {
      int done, index = 0, longform = 0 ;
      char fmtbuf[MAX_FORMAT] ;

      fmtbuf[index++] = c ;
      do {
	done = 1 ;
	fmtbuf[index++] = c = *format++ ;
	fmtbuf[index] = '\0' ;
	switch (c) {
	case '%':
	  fputc(c, stderr) ;
	case '\0':
	  break ;
	case 'e': case 'E': case 'f': case 'g': case 'G':
	  {
	    double d = va_arg(args, double) ;
	    fprintf(stderr, fmtbuf, d) ;
	  }
	  break ;
	case 'c': case 'd': case 'i': case 'o':
	case 'p': case 'u': case 'x': case 'X':
	  if ( longform ) {
	    long l = va_arg(args, long) ;
	    fprintf(stderr, fmtbuf, l) ;
	  } else {
	    int i = va_arg(args, int) ;
	    fprintf(stderr, fmtbuf, i) ;
	  }
	  break ;
	case 's':
	  {
	    char *s = va_arg(args, char *) ;
	    fprintf(stderr, fmtbuf, s) ;
	  }
	  break ;
	case 'l':
	  longform = 1 ;
	  /* FALLTHRU */
	default:
	  done = 0 ;
	}
      } while ( !done ) ;
    } else	/* Not %, write out character */
      fputc(c, stderr) ;
  }
  fflush(stderr) ;
}

void error(char *format, ...)
{
  va_list args ;

  va_start(args, format) ;
  fmtout(format, args) ;
  va_end(args) ;

  exit(1) ;
}

void warning(char *format, ...)
{
  va_list args ;

  va_start(args, format) ;
  fmtout(format, args) ;
  va_end(args) ;
}

/* Communications with LPD are managed through this function.
   Note: return value is exit status. */
static char *lpdhost ;
static char *lpdprint ;
static SOCKET printsock ;

int LPD(char *printer, char *hostname, int (*fn)(char *printer, char *hostname,
						 SOCKET printsock,
						 va_list args), ...)
{
  va_list args ;
  int result = 1 ; /* Fail, unless we get to the bottom */
  BOOL reuse = TRUE, linger = TRUE ;
  WSADATA wsaData;
  struct hostent *hostentry = NULL ;
  struct sockaddr_in slocal ;
  struct sockaddr_in sremote ;

  if ( hostname == NULL )
    error("%s: no host specified, use -H or PRINTHOST environment variable\n",
	  program) ;

  if ( printer == NULL )
    error("%s: no printer specified, use -P or PRINTER environment variable\n",
	  program) ;

  lpdprint = printer ;
  lpdhost = hostname ;

  if ( WSAStartup(MAKEWORD(1,1), &wsaData) != 0)
    error("%s: Unable to start TCP/IP stack\n", program) ;

  /* Make sure winsock version is compatible. */
  if ( LOBYTE(wsaData.wVersion) != 1 || HIBYTE(wsaData.wVersion) != 1 ) {
    warning("%s: wrong version of protocol stack, need 1.1\n", program) ;
    goto cleanup ;
  }

  if ( INVALID_SOCKET == (printsock = socket(PF_INET, SOCK_STREAM, 0)) ) {
    warning("%s: opening lpd socket to %s failed\n", program, lpdhost) ;
    goto cleanup ;
  }

  if ( setsockopt(printsock, SOL_SOCKET, SO_REUSEADDR, (char *)&reuse,
		  sizeof(reuse)) != 0 ||
       setsockopt(printsock, SOL_SOCKET, SO_LINGER, (char *)&linger,
		  sizeof(linger)) != 0 )
    error("%s: can't set socket options, error %d\n", program,
          WSAGetLastError()) ;

  /* Set up local socket address */
  {
    short portno ;
    char buffer[BUFSIZ] ;

    if ( SOCKET_ERROR == gethostname(buffer, BUFSIZ) ) {
      warning("%s: can't get own host name, error %d\n", program,
	      WSAGetLastError()) ;
      goto close_cleanup ;
    }

    if ( (hostentry = gethostbyname(buffer)) == NULL) {
      warning("%s: unknown host %s, error %d\n", program, buffer,
	      WSAGetLastError()) ;
      goto close_cleanup ;
    }

    memset(&slocal, 0, sizeof(slocal)) ;
    slocal.sin_family = AF_INET ;
    memcpy(&slocal.sin_addr, hostentry->h_addr, hostentry->h_length);

    for ( portno = LOCALPORTMIN ; portno <= LOCALPORTMAX ; portno++ ) {
      slocal.sin_port = htons(portno) ;
      if ( bind(printsock, (struct sockaddr *)&slocal, sizeof(slocal)) == 0 )
	break ;	/* Found local port */
    }

    if ( portno > LOCALPORTMAX ) {
      warning("%s: bind to local port failed, error %d\n", program,
	      WSAGetLastError()) ;
      goto close_cleanup ;
    }
  }

  if ( (hostentry = gethostbyname(lpdhost)) == NULL) {
    warning("%s: unknown host %s\n", program, lpdhost) ;
    goto close_cleanup ;
  }

  /* Set up remote socket address */
  memset(&sremote, 0, sizeof(sremote)) ;
  sremote.sin_family = AF_INET ;
  sremote.sin_port = htons(REMOTEPORTNO) ;
  memcpy(&sremote.sin_addr, hostentry->h_addr, hostentry->h_length);

  if ( SOCKET_ERROR == connect(printsock, (struct sockaddr *)&sremote,
			       sizeof(sremote)) ) {
    warning("%s: connect to %s failed, error %d\n", program, lpdhost,
	    WSAGetLastError()) ;
    goto close_cleanup ;
  }

  va_start(args, fn) ;
  result = (*fn)(lpdprint, lpdhost, printsock, args) ;
  va_end(args) ;

  /* If using winsock2.h, can shutdown. Otherwise, should linger */
  /* shutdown(printsock, SD_SEND) ; */

close_cleanup:
  if ( closesocket(printsock) != 0 ) {
    warning("%s: closesocket, error %d\n", program, WSAGetLastError()) ;
    result = 1 ;
  }

cleanup:
  if ( WSACleanup() != 0 ) {
    warning("%s: WSACleanup, error %d\n", program, WSAGetLastError()) ;
    result = 1 ;
  }

  return result ;
}

/* Get and check single byte acknowledgement */
int Acknowledge(int ping)
{
  char buffer[1] ;	/* space for acknowledgement */
  int bytesread ;
  
  buffer[0] = '\0' ;

  if ( ping && 1 != send(printsock, buffer, 1, 0) ) {
    warning("%s: failed sending confirmation to %s, error %d\n", program,
	    lpdhost, WSAGetLastError()) ;
    return 0 ;
  }

  if ( (bytesread = recv(printsock, buffer, 1, 0)) == SOCKET_ERROR ) {
    warning("%s: recv from %s failed, error %d\n", program, lpdhost,
	    WSAGetLastError()) ;
    return 0 ;
  }

  if ( bytesread == 0 ) {
    warning("%s: connection to %s closed from remote end\n", program, lpdhost) ;
    return 0 ;
  }

  if ( buffer[0] != '\0' ) {
    warning("%s: can't print, lpd on %s returned error %d\n", program,
	    lpdhost, (int)buffer[0]) ;
    return 0 ;
  }

  return 1 ;
}

/* Send a formatted command to the LPD */
int Command(char *format, ...)
{
  va_list args ;
  char buffer[BUFSIZ] ;
  int index = 0 ;
  int c ;

  va_start(args, format) ;
  while ( (c = (unsigned char)*format++) != '\0' ) {
    if ( c == '%') {
      int done, fmtindex = 0 ;
      char fmtbuf[MAX_FORMAT] ;

      fmtbuf[fmtindex++] = c ;
      do {
	done = 1 ;
	fmtbuf[fmtindex++] = c = *format++ ;
	fmtbuf[fmtindex] = '\0' ;
	switch ( c ) {
	case 'c':
	  buffer[index++] = va_arg(args, char) ;
	  break ;
	case 's':
	  strcpy(&(buffer[index]), va_arg(args, char *)) ;
	  index += strlen(&(buffer[index])) ;
	  break ;
	case 'd':
	  sprintf(&(buffer[index]), fmtbuf, va_arg(args, int)) ;
	  index += strlen(&(buffer[index])) ;
	  break ;
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	  done = 0 ;
	  break ;
	default:
	  warning("%s: bad command format %s\n", program, format) ;
	  va_end(args) ;
	  return 0 ;
	}
      } while ( !done ) ;
    } else
      buffer[index++] = c ;
  }
  va_end(args) ;

  if ( index > 0 ) {
    int totalsent = 0 ;

    do {
      int bytessent ;

      if ( (bytessent = send(printsock, buffer + totalsent, index - totalsent, 0)) == SOCKET_ERROR ) {
	warning("%s: send to %s, error %d\n", program, lpdhost, WSAGetLastError()) ;
	return 1 ;
      }
      totalsent += bytessent ;
    } while ( totalsent < index ) ;
  }

  return 1 ;
}

/* Get a reply and send to output stream */
int Reply(char *hostname, SOCKET printsock, FILE *ofile)
{
  unsigned char buffer[BUFSIZ] ;
  int len, column = 0 ;

  /* Get reply */
  do {
    int index ;

    if ( SOCKET_ERROR == (len = recv(printsock, buffer, BUFSIZ, 0)) ) {
      warning("%s: recv from %s failed, error %d\n", program, hostname,
              WSAGetLastError()) ;
      return 0 ;
    }

    for ( index = 0 ; index < len ; index++ ) {
      char c = buffer[index] ;

      switch ( c ) {
      case '\t':
        do {
          fputc(' ', ofile) ;
          column++ ;
        } while ( column % 8 != 0 ) ;
        break ;
      case '\n':
        fputc(c, ofile) ;
        column = 0 ;
        break ;
      default:
        fputc(c, ofile) ;
        column++ ;
      }
    }
  } while ( len != 0 ) ;

  if ( column > 0 )
    fputc('\n', ofile) ;

  return 1 ;
}

/* Registry functions: machine:host associations and sequence numbers are kept
   in the registry. Values are stored in the registry key
   HKEY_LOCAL_MACHINE\SOFTWARE\FreeWare\Lpr */
HKEY LPRKey(REGSAM permission)
{
  HKEY key = HKEY_LOCAL_MACHINE ;
  int result, index ;

  static char *keypath[] = {"SOFTWARE", "FreeWare", "Lpr", (char *)NULL} ;

  for ( index = 0 ; keypath[index] != NULL ; index++ ) {
    HKEY subkey ;

    result = RegOpenKeyEx(key, keypath[index], 0, KEY_READ|permission, &subkey) ;

    if ( result != ERROR_SUCCESS ) {
      DWORD disposition ;
      result = RegCreateKeyEx(key, keypath[index], 0, "",
			      REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS,
			      NULL, &subkey, &disposition) ;
    }

    if ( index != 0 )
      RegCloseKey(key) ;

    if ( result != ERROR_SUCCESS ) {
      warning("%s: can't open lpr registry key\n", program) ;
      return (HKEY)NULL ;
    }

    key = subkey ;
  }

  return key ;
}

/* Set job number */
static int SetRegistryInt(char *name, int value)
{
  HKEY key = LPRKey(KEY_SET_VALUE) ;
  LONG result ;

  if ( key == (HKEY)NULL )
    return ERROR_CANTWRITE ;

  result = RegSetValueEx(key, name, 0, REG_DWORD, (BYTE *)&value, 4) ;

  RegCloseKey(key) ;

  return result ;
}

/* Get job number */
static int GetRegistryInt(char *name)
{
  HKEY key = LPRKey(KEY_READ) ;
  DWORD type ;
  unsigned char buffer[4] ;
  DWORD size = 4 ;
  LONG result ;

  if ( key == (HKEY)NULL )
    return 0 ;

  result = RegQueryValueEx(key, name, NULL, &type, buffer, &size) ;

  RegCloseKey(key) ;

  if ( result != ERROR_SUCCESS  || size != 4 ||
       (type != REG_DWORD && type != REG_DWORD_LITTLE_ENDIAN) )
    return 0 ;

  return *((int *)buffer) ;
}

/* Get printer host association */
static char *GetRegistryStr(char *name)
{
  HKEY key = LPRKey(KEY_READ) ;
  DWORD type ;
  unsigned char *buffer = NULL ;
  DWORD size, bufsize = 16 ;
  LONG result ;

  if ( key == (HKEY)NULL )
    return NULL ;

  do {
    if ( buffer == NULL || result == ERROR_MORE_DATA ) {
      bufsize *= 2 ;
      buffer = xalloc(buffer, bufsize) ;
    }

    size = bufsize ;

    result = RegQueryValueEx(key, name, NULL, &type, buffer, &size) ;
  } while ( result == ERROR_MORE_DATA ) ;

  RegCloseKey(key) ;

  if ( result != ERROR_SUCCESS || (type != REG_SZ && type != REG_EXPAND_SZ) ) {
    free(buffer) ;
    return NULL ;
  }

  return (char *)buffer ;
}

/* Get next job number, and update registry key */
int NextJobNumber(void)
{
  int jobno = GetRegistryInt("NextJobNumber") ;

  if ( SetRegistryInt("NextJobNumber", (jobno + 1) % 1000) != ERROR_SUCCESS )
    warning("%s: can't set NextJobNumber registry key\n", program) ;

  return jobno ;
}

char *DefaultHost(char *printer)
{
  char *at = strchr(printer, '@') ;

  if ( at ) {
    *at = '\0' ;
    return at + 1 ;
  }

  if ( printer != NULL ) {
    char *host = GetRegistryStr(printer) ;

    if ( host )
      return host ;
  }

  return getenv(PRINTHOST) ;
}

/*
 * $Log: common.c,v $
 * Revision 1.3  2000/08/31 22:05:14  angus
 * Keep socket open for lpr until reply read; make reply reading routine common
 *
 * Revision 1.2  1999/10/24 05:45:30  angus
 *
 * Add support for -Z options
 * Allow printer@host syntax for all programs
 * Use SO_DONTLINGER to enable socket re-use
 *
 * Revision 1.1.1.1  1999/07/02 06:21:11  angus
 * Initial CVS import
 *
 */
