// $Id: gweMysqlResultSet.java,v 1.8 1997/11/25 21:21:19 xzhu Exp $
//////////////////////////////////////////////////////////////////
// mysql JDBC Driver
//
// Version : 0.9.2
//
// JDBC : 1.22
//
// LastModified: 16.03.1997
//
// Copyright (c) 1997 GWE Technologies Ltd. All rights reserved
// See http://gwe.co.uk/mysql/jdbc/copyright.html for full details

package gwe.sql;

import java.sql.*;
import gwe.sql.gweMysql;
import gwe.sql.gweMysqlColumn;
import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * This class implements the JDBC draft specification for the ResulSet
 * interface for mysql.  The class is designed to hold the results of
 * database queries.
 */
public class gweMysqlResultSet implements java.sql.ResultSet 
{
  private gweMysql result;
  /**
   * The meta data for this result set.
   */
  private gweMysqlMetaResult meta;
  /**
   * Keeps track of the last column read. 
   */
  private int last_read = -1;
  //  We can chain multiple resultSets together - this points to
  // next resultSet in the chain.
  private gweMysqlResultSet next = null;
  private int updateCount;
  private int updateID;

  /**
   * Constructs a new result set object given the gweMysqlResult specified.
   * @param res the gweMysql returned from a previously executed query
   */
  public gweMysqlResultSet(gweMysql res) 
  {
    result = res;
    updateCount = res.getUpdateCount();
    updateID = res.getUpdateID();
  }

  public gweMysqlResultSet(gweMysql res, int count, int ID)
  {
    result = res;
    updateCount = count;
    updateID = ID;
  }

  /**
   * JDBC draft specification method for moving the current row to the
   * next row, returning true if there is another row for processing.
   * @see java.sql.ResultSet#next
   * @exception SQLException thrown if an error occurs during processing
   * @return true if there are more rows to process, otherwise false
   */
  public synchronized boolean next() throws SQLException {
    try {
		last_read=-1;
		return result.nextResult();
    } catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }
    
  /**
   * JDBC specification method to determine if a column is null.
   * @see java.sql.ResultSet#isNull
   * @exception SQLException in the event of an SQLException
   * @param column the column being tested for nullness
   * @return true if the column is null, false otherwise
   */
  public boolean isNull(int column) throws SQLException {
    try {
		return (result.isNull(column));
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC specification method to determine if the last column read was null.
   * @see java.sql.ResultSet#wasNull
   * @exception SQLException in the event of an SQLException
   * @return true if the column was null, false otherwise
   */
  public boolean wasNull() throws SQLException {
    if( last_read == -1 ) return false;
    return isNull(last_read);
  }

  /**
   * JDBC draft specification method for getting a string value from
   * the named column.  Note that the JDBC draft provides that this
   * method gets the value as a char, so you can retrieve int values
   * into String objects.
   * @see java.sql.ResultSet#getString
   * @exception SQLException thrown for invalid columns or bad rows
   * @param column the column being retrieved
   * @return the column as a String
   */
  public String getString(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return null;
    try {
		return result.getString(column);
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Get the value of a column in the current row as a Java byte.
   *
   * @param columnIndex the first column is 1, the second is 2, ...
   * @return the column value; if the value is SQL NULL the result is 0
   */
  public byte getByte(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return 0;
    try {
	    String str = result.getString(column);
	    if( str.equals("") ) return 0;
	    return (byte)(Integer.valueOf(str).intValue());
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }
  
  /**
   * JDBC specification method for retrieving a column as a boolean
   * value.  Interprets "", null, and "0" as false, others as true.
   * @see java.sql.ResultSet#getBoolean
   * @exception SQLException a sure sign of the apocolypse
   * @param column the column for which the value is being retrieved
   * @return false for "", null, or "0"; true otherwise
   */
  public boolean getBoolean(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return false;
    try {
      String str = result.getString(column);
      
      if( str.equals("") ) return false;
      if( str.equals("0") ) return false;
      if( str.equals("N") ) return false;
      if( str.equals("n") ) return false;
      return true;
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC draft specification method to retrieve a short value from
   * the database.
   * @see java.sql.ResultSet#getShort
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a short
   */
  public short getShort(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return 0;
    try {
		String str;
		last_read=column;
		str = result.getString(column);
		if( str == null ) return 0;
	    return (short)(Integer.valueOf(str).intValue());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve an integer value from
   * the database.
   * @see java.sql.ResultSet#getInt
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as an integer
   */
  public int getInt(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return 0;
    try {
		String str;
		last_read=column;
		str = result.getString(column);
		if( str == null ) return 0;
  	    return Integer.valueOf(str).intValue();
    }
    catch( Exception e ) {
      throw new SQLException("ResultSet.getInt " + e.getMessage());
    }
  }  
  
  /**
   * JDBC draft specification method to retrieve a long value from
   * the database.
   * @see java.sql.ResultSet#getLong
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a short
   */
  public long getLong(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return 0;
    try {
		String str;
		last_read=column;
		str = result.getString(column);
		if( str == null ) return 0;
	    return Long.valueOf(str).longValue();
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve a float value from
   * the database.
   * @see java.sql.ResultSet#getFloat
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a float
   */
  public float getFloat(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return 0.0f;
    try {
		String str;
		last_read=column;
		str = result.getString(column);
		if( str == null ) return 0;
      	return Float.valueOf(str).floatValue();
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve a double value from
   * the database.
   * @see java.sql.ResultSet#getDouble
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a double
   */
  public double getDouble(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return 0.0;
    try {
	    String str;
		last_read=column;
    	str = result.getString(column);
    	if( str == null ) return 0;
      	return Double.valueOf(str).doubleValue();
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }  


  public BigDecimal getBigDecimal(int column, int scale) throws SQLException {
    last_read = column;
    if( isNull(column) ) return null;
    try {
		String str;
		last_read=column;
		str = result.getString(column);
		if( str == null ) return new BigDecimal (new BigInteger ("0"), scale);
		else return new BigDecimal (new BigInteger (str), scale);
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }


  public BigDecimal 
  getBigDecimal(String columnName, int scale) throws SQLException 
  {
    try {
        return getBigDecimal(findColumn(columnName), scale);
	}
    catch( SQLException e ) {
      throw e;
    }
  }


  /**
   * JDBC draft specification method to retrieve a Bignum object from
   * the database.
   * @see java.sql.ResultSet#getBignum
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @param scale how many decimal digits after the floating point to maintain
   * @return the named column as a Bignum
  public Bignum getBignum(int column, int scale) throws SQLException {
    try {
		String str;
		str = result.getString(column);
		if( str == null ) return new Bignum(0, scale);
		else return new Bignum(str, scale);
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }
   */


  /**
   * JDBC draft specification method to retrieve a decimal from
   * the database.
   * @see java.sql.ResultSet#getDecimal
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @param scale how many decimal digits after the floating point to maintain
   * @return the named column as a Bignum
  public Bignum getDecimal(int column, int scale) throws SQLException {
    try {
		String str;
		str = result.getString(column);
		if( str == null ) return new Bignum(0, scale);
		else return new Bignum(str, scale);
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }
   */


  /**
   * JDBC draft specification method to return a byte array.
   * @see java.sql.ResultSet#getBytes
   * @exception SQLException thrown if something goes wrong
   * @param column the column being retrieved
   * @return a byte array that is the value of the column
   */

  public byte[] getBytes(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return null;
    try {
		return result.getBytes(column);
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC 1.10 specification for retrieving a date column.
   * Can you say namespace pollution?  I knew you could.
   * @see java.sqlResultSet#getDate
   * @exception SQLException thrown in the event of problems
   * @param column the column being retrieved
   * @return the date value for the column
   */
  public java.sql.Date 
  getDate(int column) throws SQLException 
  {
    last_read = column;
    if( isNull(column) ) return null;
	try {
		String str = result.getString(column);
		if( str.length() != 10) 
		  throw new NumberFormatException("Wrong Length!");
		Integer y = new Integer(str.substring(0,4));
		Integer m = new Integer(str.substring(5,7));
		Integer d = new Integer(str.substring(8,10));
		return new java.sql.Date(y.intValue()-1900, m.intValue()-1, d.intValue());
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC draft specification method for retrieving a time from the database.
   * @see java.sql.ResultSet#getTime
   * @exception SQLException thrown in the event of troubles
   * @param column the column being retrieved
   * @return the column as a java.sql.Time object
   */
  public java.sql.Time 
  getTime(int column) throws SQLException 
  {
    last_read = column;
    if( isNull(column) ) return null;
	try {
            String str = result.getString(column);
	    if (str.length() != 5 && str.length() != 8)
	      throw new NumberFormatException("Wrong Length!");
	    int hr = Integer.parseInt(str.substring(0,2));
	    int min = Integer.parseInt(str.substring(3,5));
	    int sec = (str.length() == 5) ? 0 : Integer.parseInt(str.substring(6));
	    return new Time(hr, min, sec);
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC draft specification method for retrieving a timestamp from
   * the database.
   * @see java.sql.ResultSet#getTimestamp
   * @exception SQLException thrown in the event of troubles
   * @param column the column being retrieved
   * @return the column as a java.sql.Timestamp object
   */
  public java.sql.Timestamp 
  getTimestamp(int column) throws SQLException 
  {
    last_read = column;
    if( isNull(column) ) return null;
	try {
		String str = result.getString(column);
      switch (str.length())
      {
        case 14:
	  {
          Integer y = new Integer(str.substring(0,4));
          Integer m = new Integer(str.substring(4,6));
          Integer d = new Integer(str.substring(6,8));
          Integer hr = new Integer(str.substring(8,10));
          Integer mi = new Integer(str.substring(10,12));
          Integer se = new Integer(str.substring(12,14));
          return new java.sql.Timestamp(y.intValue()-1900, m.intValue()-1, d.intValue(), hr.intValue(), mi.intValue(), se.intValue(), 0);
	  }
	case 12:
	  {
          Integer y = new Integer(str.substring(0,2));
          Integer m = new Integer(str.substring(2,4));
          Integer d = new Integer(str.substring(4,6));
          Integer hr = new Integer(str.substring(6,8));
          Integer mi = new Integer(str.substring(8,10));
          Integer se = new Integer(str.substring(10,12));
          return new java.sql.Timestamp(y.intValue(), m.intValue()-1, d.intValue(), hr.intValue(), mi.intValue(), se.intValue(), 0);
	  }
	case 8:
	  {
          Integer y = new Integer(str.substring(0,4));
          Integer m = new Integer(str.substring(4,6));
          Integer d = new Integer(str.substring(6,8));
          return new java.sql.Timestamp(y.intValue()-1900, m.intValue()-1, d.intValue(), 0, 0, 0, 0);
	  }
	case 6:
	  {
          Integer hr = new Integer(str.substring(6,8));
          Integer mi = new Integer(str.substring(8,10));
          Integer se = new Integer(str.substring(10,12));
          return new java.sql.Timestamp(0, 0, 0, hr.intValue(), mi.intValue(), se.intValue(), 0);
	  }
	default:
	  throw new SQLException("Wrong format for Timestamp: " + str.length());
      }
    }
    catch( Exception e ) 
    {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * This is not currently supported.
   */
  public java.io.InputStream getAsciiStream(int column)
       throws SQLException {
    return null;
  }

  /**
   * This is not currently supported.
   */
  public java.io.InputStream getUnicodeStream(int column)
       throws SQLException {
    return null;
  }

  /**
   * This is not currently supported.
   */
  public java.io.InputStream getBinaryStream(int column)
       throws SQLException {
    return null;
  }
  
  /**
   * JDBC draft specification method for closing a result set.
   * @see java.sql.ResultSet#close
   */
  public void close() throws SQLException 
  {
    try {
      result.closeResultSet(); // for small data set, this is low overhead.  
	// We need to halt the flow of result packets - 
	// for now, close the socket, and re-conect.
	//result.reConnect();
    } catch( Exception e ) 
    { throw new SQLException(e.getMessage()); }
  }

  /**
   * JDBC draft specification method for returning a cursor name.
   * @see java.sql.ResultSet#getCursorName
   * @return ""
   */
  public String getCursorName() throws SQLException {
    return "";
  }

  /**
   * JDBC draft specification method for returning meta-deta on a result
   * set.
   * @see java.sql.ResultSet#getMetaData
   * @exception SQLException thrown on error getting meta-data
   * @return ResultSetMetaData object containing result set info
   */
  public ResultSetMetaData getMetaData()
       throws SQLException {
    if( meta == null ) {
      meta = new gweMysqlMetaResult(result);
    }
    return meta;
  }

  /**
   * JDBC draft specification method for retrieving data as objects.
   * @see java.sql.ResultSet#getObject
   * @exception SQLException in the event of retrieval! error
   * @param column the column desired
   * @param type the SQL data type of the field
   * @scale preceision for Bignums
   * @return the column specified as an Object
   */
  public Object 
  getObject(int column, int type, int scale) throws SQLException 
  {
    last_read = column;
    if( isNull(column) ) return null;
    switch(type) {
    case Types.BIT:
      return new Boolean(getBoolean(column));

    case Types.TINYINT:
      return new Character((char)getInt(column));

    case Types.SMALLINT:
      return new Integer(getShort(column));

    case Types.INTEGER:
      return new Integer(getInt(column));

    case Types.BIGINT:
      return new Long(getLong(column));

    case Types.FLOAT:
      return new Float(getFloat(column));

    case Types.REAL:
      return new Float(getFloat(column));

    case Types.DOUBLE:
      return new Double(getDouble(column));

    case Types.NUMERIC:
      return getBigDecimal(column, scale);

    case Types.DECIMAL:
      return getBigDecimal(column, scale);

    case Types.CHAR:
      return getString(column);

    case Types.VARCHAR:
      return getString(column);

    case Types.LONGVARCHAR:
      return getString(column);

    case Types.DATE:
      return getDate(column);

    case Types.TIME:
      return getTime(column);

    case Types.TIMESTAMP:
      return getTimestamp(column);

    case Types.BINARY:
      return getBytes(column);

    case Types.VARBINARY:
      return getBytes(column);

    case Types.LONGVARBINARY:
      return getBytes(column);

    default:
      return null;
    }
  }

  /**
   * Same as above, except with a default scale to 0.
   */
  public Object getObject(int column, int type)
       throws SQLException {
    return getObject(column, type, 0);
  }

  /**
   * Same as above, except using the column's default SQL type.
   */
  public Object getObject(int column) throws SQLException {
    ResultSetMetaData meta = getMetaData();
    int type = meta.getColumnType(column);

    return getObject(column, type);
  }
    
  /**
   * JDBC 0.65 specification method for retrieving a column value
   * as a char based on the column name.
   * @see java.sql.ResultSet#getString
   * @param name the name of the column desired
   * @return the value of the column as a char
   */
  public String getString(String name) throws SQLException {
    return getString(findColumn(name));
  }

  /**
   * Returns the column as a byte based on column name
   */
  public byte getByte(String columnName) throws SQLException {
    return getByte(findColumn(columnName));
  }
  
  /**
   * Get the value of a BIT column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is false
   */
  public boolean getBoolean(String columnName) throws SQLException {
    return getBoolean(findColumn(columnName));
  }
  
  /**
   * Get the value of a short by column name
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public short getShort(String columnName) throws SQLException {
    return getShort(findColumn(columnName));
  }

  /**
   * Get the value of a INTEGER column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public int getInt(String columnName) throws SQLException {
    return getInt(findColumn(columnName));
  }
  
  /**
   * Get the value of a long column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public long getLong(String columnName) throws SQLException {
    return getLong(findColumn(columnName));
  }

  /**
   * Get the value of a FLOAT column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public float getFloat(String columnName) throws SQLException {
    return getFloat(findColumn(columnName));
  }

  /**
   * Get the value of a DOUBLE column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public double getDouble(String columnName) throws SQLException {
    return getDouble(findColumn(columnName));
  }
  

  /**
   * Get the value of a NUMERIC column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null

  public Bignum getBignum(String columnName, int scale) throws SQLException {
    return getBignum(findColumn(columnName), scale);
  }
   */


  /**
   * Get the value of a BINARY column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public byte[] getBytes(String columnName) throws SQLException {
    return getBytes(findColumn(columnName));
  }
  
  /**
   * Get the value of a DATE column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public java.sql.Date getDate(String columnName) throws SQLException {
    return getDate(findColumn(columnName));
  }
  
  /**
   * Get the value of a TIME column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public java.sql.Time getTime(String columnName) throws SQLException {
    return getTime(findColumn(columnName));
  }
  
  /**
   * Get the value of a TIMESTAMP column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public java.sql.Timestamp getTimestamp(String columnName)
       throws SQLException {
    return getTimestamp(findColumn(columnName));
  }
  
  /**
   * Very large ascii values in a LONGVARCHAR column can
   * be read in small chunks from a java.io.InputStream.
   *
   * <P><B>Note:</B> All the data in the returned stream must
   * be read prior to getting the value of any other column. The
   * next call to a get method implicitly closes the stream.
   *
   * @param columnName is the SQL name of the column
   * @return the java input stream which contains the ascii value 
   */
  public java.io.InputStream getAsciiStream(String columnName)
       throws SQLException {
    return getAsciiStream(findColumn(columnName));
  }

  /**
   * Very large unicode values in a LONGVARCHAR column can
   * be read in small chunks from a java.io.InputStream.
   *
   * <P><B>Note:</B> All the data in the returned stream must
   * be read prior to getting the value of any other column. The
   * next call to a get method implicitly closes the stream.
   *
   * @param columnName is the SQL name of the column
   * @return the java input stream which contains the unicode value 
   */
  public java.io.InputStream getUnicodeStream(String columnName)
       throws SQLException {
    return getUnicodeStream(findColumn(columnName));
  }

  /**
   * Very large binary values in a LONGVARBINARY column can
   * be read in small chunks from a java.io.InputStream.
   *
   * <P><B>Note:</B> All the data in the returned stream must
   * be read prior to getting the value of any other column. The
   * next call to a get method implicitly closes the stream.
   *
   * @param columnName is the SQL name of the column
   * @return the java input stream which contains the binary value */
  public java.io.InputStream getBinaryStream(String columnName)
       throws SQLException {
    return getBinaryStream(findColumn(columnName));
  }

  /**
   * Get the value of a column as an object; an integral value is
   * returned as its java.lang equivalent object; a LONGVARCHAR or
   * LONGVARBINARY value is returned as a java.io.InputStream.
   *
   * @param columnName the SQL column name
   * @param sqlType SQL type code defined by java.sql.Types
   * @return the parameter value; if isNull the value is null 
   */
  public Object getObject(String columnName, int sqlType, int scale)
       throws SQLException {
    return getObject(findColumn(columnName), sqlType, scale);
  }

  /**
   * Same as above, except defaulting scale to 0.
   */
  public Object getObject(String columnName, int type)
       throws SQLException {
    return getObject(findColumn(columnName), type, 0);
  }

  /**
   * Same as above, except returning the default SQL type
   */
  public Object getObject(String columnName) throws SQLException {
    return getObject(findColumn(columnName));
  }
  
  /**
   * column name counter part for other isNull()
   */
  public boolean isNull(String name) throws SQLException {
    return isNull(findColumn(name));
  }

  /**
   * Given a column name, this method returns the column number for that
   * name.  Column name to number mappings are kept inside a Hashtable.
   * Applications that do not need the overhead of this calculation are
   * not penalized since the mapping only occurs on the first attempt to
   * access a column number by name.
   * @exception java.sql.SQLException thrown if a bad name is passed
   * @param name the name of the column desired
   * @return the column number, 1 being the first column
   */
  public int findColumn(String name) throws SQLException {
	try {
		return result.getColumnNum(name);
	}
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }
  /**
   * JDBC draft specification for getting the chain of warnings for this
   * statement.
   * @see java.sql.Statement#getWarnings
   * @return the chain of warnings
   */
  public SQLWarning getWarnings() throws SQLException {
    return null;
  }

  /**
   * JDBC draft specification for clearing the warning chain.
   * @see java.sql.Statement#clearWarnings
   */
  public void clearWarnings() throws SQLException {
  }

  public void modifyColumnHead(String[] colName)
  {
    result.modifyColumnHead(colName);
  }
  
  public void addPseudoColumn(String tabName, int[] realColMap, 
      int[] pseudoColMap, String[] colName, int[] colType, 
      String[] colValue, int[] pseudoColMode)
  {
    result.addPseudoColumn(tabName, realColMap, pseudoColMap, colName, colType,
    	colValue, pseudoColMode);
  }

  public void setRowFilter(int filterColPos, String filterString)
  {
    result.setRowFilter(filterColPos, filterString);
  }

  /**
   * Get updateID for AUTO_INCREASE field which is the current KEY value
   * @return result ID value
   */
  public int getUpdateID()
  {
    return updateID;
  }

  /**
   * Since gweMysqlResultSets can be chained, we need some method of
   * finding the next one in the chain.  The method getNext()
   * returns the next one in the chain.
   *
   * @return the next ResultSet, or null if there are none
   */
  public gweMysqlResultSet getNext()
  {
    return next;
  }

  /**
   * This following method allows us to add a ResultSet object
   * to the end of the current chain.
   *
   * @param r the resultset to add to the end of the chain.
   */
  public void append(gweMysqlResultSet r)
  {
    if (next == null)
      next = r;
    else
      next.append(r);
  }

  /**
   * If we are just a place holder for results, we still need
   * to get an updateCount.  This method returns it.
   *
   * @return the updateCount
   */
  public int getResultCount()
  {
    return updateCount;
  }

  /**
   * We at times need to know if the resultSet we are working
   * with is the result of an UPDATE, DELETE or INSERT (in which
   * case, we only have a row count), or of a SELECT operation
   * (in which case, we have multiple fields) - this routine
   * tells us.
   *
   * @return true if we have tuples available
   */
  public boolean reallyResultSet()
  {
    return result.reallyResultSet();
  }
  
}
