'From VisualWorks, Release 3.0 of February 5, 1998 on October 14, 1998 at 7:31:00 pm'!


Object subclass: #BRProgramNode
	instanceVariableNames: 'parent comments '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRProgramNode comment:
'BRProgramNode is an abstract class that represents an abstract syntax tree node in a Smalltalk program.

Subclasses must implement the following messages:
	accessing
		start
		stop
	visitor
		acceptVisitor:

The #start and #stop methods are used to find the source that corresponds to this node. "source copyFrom: self start to: self stop" should return the source for this node.

The #acceptVisitor: method is used by BRProgramNodeVisitors (the visitor pattern). This will also require updating all the BRProgramNodeVisitors so that they know of the new node.

Subclasses might also want to redefine match:inContext: and copyInContext: to do parse tree searching and replacing.

Subclasses that contain other nodes should override equalTo:withMapping: to compare nodes while ignoring renaming temporary variables, and children that returns a collection of our children nodes.

Instance Variables:
	comments	<Collection of: Interval>	the intervals in the source that have comments for this node
	parent	<BRProgramNode>	the node we''re contained in

'!


!BRProgramNode methodsFor: 'accessing'!

allArgumentVariables
	| children |
	children := self children.
	children isEmpty ifTrue: [^#()].
	^children inject: OrderedCollection new
		into: 
			[:vars :each | 
			vars
				addAll: each allArgumentVariables;
				yourself]!

allDefinedVariables
	| children |
	children := self children.
	children isEmpty ifTrue: [^#()].
	^children inject: OrderedCollection new
		into: 
			[:vars :each | 
			vars addAll: each allDefinedVariables;
				yourself]!

allTemporaryVariables
	| children |
	children := self children.
	children isEmpty ifTrue: [^#()].
	^children inject: OrderedCollection new
		into: 
			[:vars :each | 
			vars
				addAll: each allTemporaryVariables;
				yourself]!

asReturn
	"Change the current node to a return node."

	parent isNil ifTrue: [self error: 'Cannot change to a return without a parent node.'].
	parent isSequence ifFalse: [self error: 'Parent node must be a sequence node.'].
	(parent isLast: self)
		ifFalse: [self error: 'Return node must be last.'].
	^parent addReturn!

blockVariables
	^parent isNil
		ifTrue: [#()]
		ifFalse: [parent blockVariables]!

children
	^#()!

comments
	^comments isNil
		ifTrue: [#()]
		ifFalse: [comments]!

comments: aCollection
	comments := aCollection!

formattedCode
	^self formatterClass new format: self!

formatterClass
	^BRFormatter!

parent
	^parent!

parent: anObject
	parent := anObject!

precedence
	^6!

source
	^parent notNil ifTrue: [parent source] ifFalse: [nil]!

sourceInterval
	^self start to: self stop!

start
	self subclassResponsibility!

stop
	self subclassResponsibility!

temporaryVariables
	^parent isNil
		ifTrue: [#()]
		ifFalse: [parent temporaryVariables]! !

!BRProgramNode methodsFor: 'comparing'!

equalTo: aNode exceptForVariables: variableNameCollection 
	| dictionary |
	dictionary := Dictionary new.
	(self equalTo: aNode withMapping: dictionary) ifFalse: [^false].
	dictionary keysAndValuesDo: 
			[:key :value | 
			(key = value or: [variableNameCollection includes: key]) ifFalse: [^false]].
	^true!

equalTo: aNode withMapping: aDictionary 
	^self = aNode! !

!BRProgramNode methodsFor: 'copying'!

copy
	"This is redefined for IBM Smalltalk which doesn't have postCopy."

	^self shallowCopy postCopy!

copyCommentsFrom: aNode 
	"Add all comments from aNode to us. If we already have the comment, then don't add it."

	| newComments |
	newComments := OrderedCollection new.
	aNode nodesDo: [:each | newComments addAll: each comments].
	self nodesDo: 
			[:each | 
			each comments do: [:comment | newComments remove: comment ifAbsent: []]].
	newComments isEmpty ifTrue: [^self].
	newComments := newComments asSortedCollection: [:a :b | a first < b first].
	self comments: newComments!

postCopy! !

!BRProgramNode methodsFor: 'iterating'!

nodesDo: aBlock 
	aBlock value: self.
	self children do: [:each | each nodesDo: aBlock]! !

!BRProgramNode methodsFor: 'enumeration'!

collect: aBlock 
	"Hacked to fit collection protocols"

	^aBlock value: self!

do: aBlock 
	"Hacked to fit collection protocols"

	aBlock value: self!

size
	"Hacked to fit collection protocols"

	^1! !

!BRProgramNode methodsFor: 'matching'!

copyInContext: aDictionary
	^self copy!

copyList: matchNodes inContext: aDictionary 
	| newNodes |
	newNodes := OrderedCollection new.
	matchNodes do: 
			[:each | 
			| object |
			object := each copyInContext: aDictionary.
			newNodes addAll: object].
	^newNodes!

match: aNode inContext: aDictionary 
	^self = aNode!

matchList: matchNodes against: programNodes inContext: aDictionary 
	^self
		matchList: matchNodes
		index: 1
		against: programNodes
		index: 1
		inContext: aDictionary!

matchList: matchNodes index: matchIndex against: programNodes index: programIndex inContext: aDictionary 
	| node currentIndex currentDictionary nodes |
	matchNodes size < matchIndex ifTrue: [^programNodes size < programIndex].
	node := matchNodes at: matchIndex.
	node isList 
		ifTrue: 
			[currentIndex := programIndex - 1.
			
			[currentDictionary := aDictionary copy.
			programNodes size < currentIndex or: 
					[nodes := programNodes copyFrom: programIndex to: currentIndex.
					(currentDictionary at: node ifAbsentPut: [nodes]) = nodes and: 
							[(self 
								matchList: matchNodes
								index: matchIndex + 1
								against: programNodes
								index: currentIndex + 1
								inContext: currentDictionary) 
									ifTrue: 
										[currentDictionary keysAndValuesDo: [:key :value | aDictionary at: key put: value].
										^true].
							false]]] 
					whileFalse: [currentIndex := currentIndex + 1].
			^false].
	programNodes size < programIndex ifTrue: [^false].
	(node match: (programNodes at: programIndex) inContext: aDictionary) 
		ifFalse: [^false].
	^self 
		matchList: matchNodes
		index: matchIndex + 1
		against: programNodes
		index: programIndex + 1
		inContext: aDictionary! !

!BRProgramNode methodsFor: 'meta variable-accessing'!

cascadeListCharacter
	^$;!

listCharacter
	^$@!

literalCharacter
	^$#!

recurseIntoCharacter
	^$`!

statementCharacter
	^$.! !

!BRProgramNode methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: self class name;
		nextPut: $(;
		nextPutAll: self formattedCode;
		nextPut: $)! !

!BRProgramNode methodsFor: 'querying'!

bestNodeFor: anInterval 
	| selectedChildren |
	(self intersectsInterval: anInterval) ifFalse: [^nil].
	(self containedBy: anInterval) ifTrue: [^self].
	selectedChildren := self children 
				select: [:each | each intersectsInterval: anInterval].
	^selectedChildren size == 1 
		ifTrue: [selectedChildren first bestNodeFor: anInterval]
		ifFalse: [self]!

selfMessages
	| searcher |
	searcher := ParseTreeSearcher new.
	searcher matches: 'self `@msg: ``@args'
		do: 
			[:aNode :answer | 
			answer
				add: aNode selector;
				yourself].
	^searcher executeTree: self initialAnswer: Set new!

statementNode
	"Return your topmost node that is contained by a sequence node."

	(parent isNil or: [parent isSequence]) ifTrue: [^self].
	^parent statementNode!

superMessages
	| searcher |
	searcher := ParseTreeSearcher new.
	searcher matches: 'super `@msg: ``@args'
		do: 
			[:aNode :answer | 
			answer
				add: aNode selector;
				yourself].
	^searcher executeTree: self initialAnswer: Set new!

whichNodeIsContainedBy: anInterval 
	| selectedChildren |
	(self intersectsInterval: anInterval) ifFalse: [^nil].
	(self containedBy: anInterval) ifTrue: [^self].
	selectedChildren := self children 
				select: [:each | each intersectsInterval: anInterval].
	^selectedChildren size == 1 
		ifTrue: [selectedChildren first whichNodeIsContainedBy: anInterval]
		ifFalse: [nil]!

whoDefines: aName 
	^(self defines: aName)
		ifTrue: [self]
		ifFalse: [parent notNil
				ifTrue: [parent whoDefines: aName]
				ifFalse: [nil]]! !

!BRProgramNode methodsFor: 'replacing'!

removeDeadCode
	self children do: [:each | each removeDeadCode]!

replaceNode: aNode withNode: anotherNode 
	self error: 'I don''t store other nodes'!

replaceWith: aNode 
	parent isNil ifTrue: [self error: 'This node doesn''t have a parent'].
	parent replaceNode: self withNode: aNode! !

!BRProgramNode methodsFor: 'testing'!

assigns: aVariableName 
	^(self children detect: [:each | each assigns: aVariableName] ifNone: [nil]) 
		notNil!

containedBy: anInterval 
	^anInterval first <= self start and: [anInterval last >= self stop]!

containsReturn
	^(self children detect: [:each | each containsReturn] ifNone: [nil]) 
		notNil!

defines: aName
	^false!

directlyUses: aNode
	^true!

evaluatedFirst: aNode 
	self children do: 
			[:each | 
			each == aNode ifTrue: [^true].
			each isImmediate ifFalse: [^false]].
	^false!

intersectsInterval: anInterval 
	^(anInterval first between: self start and: self stop) 
		or: [self start between: anInterval first and: anInterval last]!

isAssignment
	^false!

isBlock
	^false!

isCascade
	^false!

isDirectlyUsed
	"This node is directly used as an argument, receiver, or part of an assignment."

	^parent isNil
		ifTrue: [false]
		ifFalse: [parent directlyUses: self]!

isEvaluatedFirst
	"Return true if we are the first thing evaluated in this statement."

	^parent isNil or: [parent isSequence or: [parent evaluatedFirst: self]]!

isImmediate
	^false!

isLast: aNode 
	| children |
	children := self children.
	^children isEmpty not and: [children last == aNode]!

isLiteral
	^false!

isMessage
	^false!

isMethod
	^false!

isReturn
	^false!

isSequence
	^false!

isUsed
	"Answer true if this node could be used as part of another expression. For example, you could use the 
	result of this node as a receiver of a message, an argument, the right part of an assignment, or the 
	return value of a block. This differs from isDirectlyUsed in that it is conservative since it also includes 
	return values of blocks."

	^parent isNil
		ifTrue: [false]
		ifFalse: [parent uses: self]!

isValue
	^false!

isVariable
	^false!

lastIsReturn
	^self isReturn!

references: aVariableName 
	^(self children detect: [:each | each references: aVariableName]
		ifNone: [nil]) notNil!

uses: aNode
	^true! !

!BRProgramNode methodsFor: 'testing-matching'!

isList
	^false!

recurseInto
	^false! !

!BRProgramNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	self subclassResponsibility! !

BRProgramNode class
	instanceVariableNames: ''!


Object subclass: #BRReadBeforeWrittenTester
	instanceVariableNames: 'read checkNewTemps scopeStack searcher '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRReadBeforeWrittenTester methodsFor: 'initialize-release'!

checkNewTemps: aBoolean 
	checkNewTemps := aBoolean!

createSearchTrees
	searcher := ParseTreeSearcher new.

	"Case 1 - Set the values, depending on whether we matched an assignment"
	searcher
		matches: '`var := `@object'
			do: 
				[:aNode :ans | 
				searcher executeTree: aNode value.
				self variableWritten: aNode.
				ans];
		matches: '`var'
			do: 
				[:aNode :ans | 
				self variableRead: aNode.
				ans].

	"Handle the special while* and ifTrue:ifFalse: blocks separately"
	searcher
		matchesAnyOf: 
				#('[| `@temps | ``@.Statements] whileTrue: ``@block' 
				'[| `@temps | ``@.Statements] whileTrue' 
				'[| `@temps | ``@.Statements] whileFalse: ``@block' 
				'[| `@temps | ``@.Statements] whileFalse')
			do: [:aNode :ans | ans];
		matchesAnyOf: 
				#('`@condition ifTrue: [| `@tTemps | `@.trueBlock] ifFalse: [| `@fTemps| `@.falseBlock]' 
				'`@condition ifFalse: [| `@fTemps | `@.falseBlock] ifTrue: [| `@tTemps | `@.trueBlock]')
			do: 
				[:aNode :ans | 
				searcher executeTree: aNode receiver.
				self processIfTrueIfFalse: aNode.
				ans].

	"Case 2 - Recursive call yourself on the body of the block node just matched"
	searcher matches: '[:`@args | | `@temps | `@.Statements]'
		do: 
			[:aNode :ans | 
			self processBlock: aNode.
			ans].
	searcher matches: '| `@temps | `@.Stmts'
		do: 
			[:aNode :ans | 
			self processStatementNode: aNode.
			ans]!

initialize
	scopeStack := OrderedCollection with: Dictionary new.
	read := Set new.
	checkNewTemps := true.
	self createSearchTrees!

initializeVars: varNames 
	varNames do: [:each | self currentScope at: each put: nil]! !

!BRReadBeforeWrittenTester methodsFor: 'accessing'!

executeTree: aParseTree 
	^searcher executeTree: aParseTree!

read
	self currentScope
		keysAndValuesDo: [:key :value | value == true ifTrue: [read add: key]].
	^read! !

!BRReadBeforeWrittenTester methodsFor: 'private'!

copyDictionary: aDictionary 
	"We could send aDictionary the copy message, but that doesn't copy the associations."

	| newDictionary |
	newDictionary := Dictionary new: aDictionary size.
	aDictionary
		keysAndValuesDo: [:key :value | newDictionary at: key put: value].
	^newDictionary!

createScope
	scopeStack add: (self copyDictionary: scopeStack last)!

currentScope
	^scopeStack last!

processBlock: aNode 
	| newScope |
	self createScope.
	self executeTree: aNode body.
	newScope := self removeScope.
	newScope keysAndValuesDo: 
			[:key :value | 
			(value == true and: [(self currentScope at: key) isNil]) 
				ifTrue: [self currentScope at: key put: value]]!

processIfTrueIfFalse: aNode 
	| trueScope falseScope |
	self createScope.
	self executeTree: aNode arguments first body.
	trueScope := self removeScope.
	self createScope.
	self executeTree: aNode arguments last body.
	falseScope := self removeScope.
	self currentScope keysAndValuesDo: 
			[:key :value | 
			value isNil 
				ifTrue: 
					[(trueScope at: key) == (falseScope at: key) 
						ifTrue: [self currentScope at: key put: (trueScope at: key)]
						ifFalse: 
							[((trueScope at: key) == true or: [(falseScope at: key) == true]) 
								ifTrue: [self currentScope at: key put: true]]]]!

processStatementNode: aNode 
	| temps |
	(checkNewTemps not or: [aNode temporaries isEmpty]) 
		ifTrue: 
			[aNode statements do: [:each | self executeTree: each].
			^self].
	self createScope.
	temps := aNode temporaries collect: [:each | each name].
	self initializeVars: temps.
	aNode statements do: [:each | self executeTree: each].
	self removeScope keysAndValuesDo: 
			[:key :value | 
			(temps includes: key) 
				ifTrue: [value == true ifTrue: [read add: key]]
				ifFalse: 
					[(self currentScope at: key) isNil 
						ifTrue: [self currentScope at: key put: value]]]!

removeScope
	^scopeStack removeLast!

variableRead: aNode 
	(self currentScope includesKey: aNode name) ifTrue: 
			[(self currentScope at: aNode name) isNil
				ifTrue: [self currentScope at: aNode name put: true]]!

variableWritten: aNode 
	(self currentScope includesKey: aNode variable name) ifTrue: 
			[(self currentScope at: aNode variable name) isNil
				ifTrue: [self currentScope at: aNode variable name put: false]]! !

BRReadBeforeWrittenTester class
	instanceVariableNames: ''!



!BRReadBeforeWrittenTester class methodsFor: 'instance creation'!

new
	^super new initialize! !

!BRReadBeforeWrittenTester class methodsFor: 'accessing'!

readBeforeWritten: varNames in: aParseTree 
	^(self new)
		checkNewTemps: false;
		initializeVars: varNames;
		executeTree: aParseTree;
		read!

variablesReadBeforeWrittenIn: aParseTree 
	^(self new)
		executeTree: aParseTree;
		read! !

BRProgramNode subclass: #BRSequenceNode
	instanceVariableNames: 'leftBar rightBar statements periods temporaries '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRSequenceNode comment:
'BRSequenceNode is an AST node that represents a sequence of statements. Both BRBlockNodes and BRMethodNodes contain these.

Instance Variables:
	leftBar	<Integer | nil>	the position of the left | in the temporaries definition
	periods	<SequenceableCollection of: Integer>	the positions of all the periods that separate the statements
	rightBar	<Integer | nil>	the position of the right | in the temporaries definition
	statements	<SequenceableCollection of: BRStatementNode>	the statement nodes
	temporaries	<SequenceableCollection of: BRVariableNode>	the temporaries defined

'!


!BRSequenceNode methodsFor: 'accessing'!

addReturn
	| node |
	statements isEmpty ifTrue: [^nil].
	statements last isReturn ifTrue: [^statements last].
	node := BRReturnNode value: statements last.
	statements at: statements size put: node.
	node parent: self.
	^node!

allDefinedVariables
	^(self temporaryNames asOrderedCollection) addAll: super allDefinedVariables;
		yourself!

allTemporaryVariables
	^(self temporaryNames asOrderedCollection)
		addAll: super allTemporaryVariables;
		yourself!

children
	^(OrderedCollection new) addAll: self temporaries;
		addAll: self statements;
		yourself!

periods: anObject
	periods := anObject!

removeTemporaryNamed: aName 
	temporaries := temporaries reject: [:each | each name = aName]!

start
	^leftBar isNil 
		ifTrue: [statements isEmpty ifTrue: [1] ifFalse: [statements first start]]
		ifFalse: [leftBar]!

statements
	^statements!

statements: stmtCollection 
	statements := stmtCollection.
	statements do: [:each | each parent: self]!

stop
	^(periods isEmpty
		ifTrue: [0]
		ifFalse: [periods last])
		max: (statements isEmpty
				ifTrue: [0]
				ifFalse: [statements last stop])!

temporaries
	^temporaries!

temporaries: tempCollection 
	temporaries := tempCollection.
	temporaries do: [:each | each parent: self]!

temporaryNames
	^temporaries collect: [:each | each name]!

temporaryVariables
	^(super temporaryVariables asOrderedCollection) addAll: self temporaryNames;
		yourself! !

!BRSequenceNode methodsFor: 'comparing'!

= anObject 
	"Can't send = to the temporaries and statements collection since they might change from arrays to OCs"

	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	self temporaries size = anObject temporaries size ifFalse: [^false].
	1 to: self temporaries size
		do: 
			[:i | 
			(self temporaries at: i) = (anObject temporaries at: i) ifFalse: [^false]].
	self statements size = anObject statements size ifFalse: [^false].
	1 to: self statements size
		do: [:i | (self statements at: i) = (anObject statements at: i) ifFalse: [^false]].
	^true!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	self statements size == anObject statements size ifFalse: [^false].
	1 to: self statements size
		do: 
			[:i | 
			((self statements at: i) equalTo: (anObject statements at: i)
				withMapping: aDictionary) ifFalse: [^false]].
	aDictionary values asSet size = aDictionary size ifFalse: [^false].	"Not a one-to-one mapping"
	self temporaries
		do: [:each | aDictionary removeKey: each name ifAbsent: []].
	^true!

hash
	^self temporaries hash bitXor: (self statements isEmpty
			ifTrue: [0]
			ifFalse: [self statements first hash])! !

!BRSequenceNode methodsFor: 'copying'!

postCopy
	super postCopy.
	temporaries := temporaries collect: [:each | each copy].
	statements := statements collect: [:each | each copy]! !

!BRSequenceNode methodsFor: 'initialize-release'!

leftBar: leftInteger temporaries: variableNodes rightBar: rightInteger 
	leftBar := leftInteger.
	self temporaries: variableNodes.
	rightBar := rightInteger! !

!BRSequenceNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new)
		temporaries: (self copyList: temporaries inContext: aDictionary);
		statements: (self copyList: statements inContext: aDictionary);
		yourself!

match: aNode inContext: aDictionary 
	self class == aNode class ifFalse: [^false].
	^(self matchList: temporaries
		against: aNode temporaries
		inContext: aDictionary) and: 
				[self matchList: statements
					against: aNode statements
					inContext: aDictionary]! !

!BRSequenceNode methodsFor: 'replacing'!

removeDeadCode
	(self isUsed ifTrue: [statements size - 1] ifFalse: [statements size]) 
		to: 1
		by: -1
		do: [:i | (statements at: i) isImmediate ifTrue: [statements removeAtIndex: i]].
	super removeDeadCode!

removeNode: aNode
	self replaceNode: aNode withNodes: #()!

replaceNode: aNode withNode: anotherNode 
	self statements: (statements 
				collect: [:each | each == aNode ifTrue: [anotherNode] ifFalse: [each]]).
	self temporaries: (temporaries 
				collect: [:each | each == aNode ifTrue: [anotherNode] ifFalse: [each]])!

replaceNode: aNode withNodes: aCollection 
	| index newStatements |
	index := self indexOfNode: aNode.
	newStatements := OrderedCollection new: statements size + aCollection size.
	1 to: index - 1 do: [:i | newStatements add: (statements at: i)].
	newStatements addAll: aCollection.
	index + 1 to: statements size
		do: [:i | newStatements add: (statements at: i)].
	aCollection do: [:each | each parent: self].
	statements := newStatements! !

!BRSequenceNode methodsFor: 'private'!

indexOfNode: aNode 
	"Try to find the node by first looking for ==, and then for ="

	^(1 to: statements size) detect: [:each | each == aNode]
		ifNone: [statements indexOf: aNode]! !

!BRSequenceNode methodsFor: 'testing'!

defines: aName 
	^(temporaries detect: [:each | each name = aName] ifNone: [nil]) notNil!

directlyUses: aNode 
	^false!

isLast: aNode 
	| last |
	statements isEmpty ifTrue: [^false].
	last := statements last.
	^last == aNode or: [last isMessage and: [(#(#ifTrue:ifFalse: #ifFalse:ifTrue:) includes: last selector)
				and: [last arguments inject: false into: [:bool :each | bool or: [each isLast: aNode]]]]]!

isSequence
	^true!

lastIsReturn
	^statements isEmpty not and: [statements last lastIsReturn]!

references: aVariableName 
	^(statements detect: [:each | each references: aVariableName] ifNone: [nil]) 
		notNil!

uses: aNode 
	statements isEmpty ifTrue: [^false].
	aNode == statements last ifFalse: [^false].
	^self isUsed! !

!BRSequenceNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptSequenceNode: self! !

!BRSequenceNode methodsFor: 'adding nodes'!

addNode: aNode 
	aNode parent: self.
	(statements isEmpty not and: [statements last isReturn])
		ifTrue: [self error: 'Cannot add statement after return node'].
	statements := statements asOrderedCollection add: aNode; yourself!

addNode: aNode before: anotherNode 
	| index |
	index := self indexOfNode: anotherNode.
	index = 0 ifTrue: [^self addNode: aNode].
	statements := (statements asOrderedCollection)
				add: aNode beforeIndex: index;
				yourself.
	aNode parent: self!

addNodeFirst: aNode 
	aNode parent: self.
	statements := (statements asOrderedCollection)
				addFirst: aNode;
				yourself!

addNodes: aCollection 
	aCollection do: [:each | each parent: self].
	(statements isEmpty not and: [statements last isReturn]) 
		ifTrue: [self error: 'Cannot add statement after return node'].
	statements := (statements asOrderedCollection)
				addAll: aCollection;
				yourself!

addNodes: aCollection before: anotherNode 
	aCollection do: [:each | self addNode: each before: anotherNode]!

addNodesFirst: aCollection 
	aCollection do: [:each | each parent: self].
	statements := (statements asOrderedCollection)
				addAllFirst: aCollection;
				yourself!

addSelfReturn
	| node |
	self lastIsReturn ifTrue: [^self].
	node := BRReturnNode value: (BRVariableNode named: 'self').
	self addNode: node!

addTemporariesNamed: aCollection 
	aCollection do: [:each | self addTemporaryNamed: each]!

addTemporaryNamed: aString 
	| variableNode |
	variableNode := BRVariableNode named: aString.
	variableNode parent: self.
	temporaries := temporaries copyWith: variableNode! !

!BRSequenceNode methodsFor: 'querying'!

bestNodeFor: anInterval 
	| node |
	node := super bestNodeFor: anInterval.
	node == self 
		ifTrue: 
			[(temporaries isEmpty and: [statements size == 1]) 
				ifTrue: [^statements first]].
	^node!

whichNodeIsContainedBy: anInterval 
	| node |
	node := super whichNodeIsContainedBy: anInterval.
	node == self 
		ifTrue: 
			[(temporaries isEmpty and: [statements size == 1]) 
				ifTrue: [^statements first]].
	^node! !

BRSequenceNode class
	instanceVariableNames: ''!



!BRSequenceNode class methodsFor: 'instance creation'!

leftBar: leftInteger temporaries: variableNodes rightBar: rightInteger 
	^self new
		leftBar: leftInteger
		temporaries: variableNodes
		rightBar: rightInteger!

statements: statementNodes 
	^self temporaries: #() statements: statementNodes!

temporaries: variableNodes statements: statementNodes 
	^(self new)
		temporaries: variableNodes;
		statements: statementNodes;
		yourself! !


!Behavior methodsFor: 'RefactoringBrowser'!

parseTreeFor: aSymbol 
	^BRParser parseMethod: (self sourceCodeAt: aSymbol)
		onError: [:aString :pos | ^nil]! !

Stream subclass: #BRScanner
	instanceVariableNames: 'stream buffer tokenStart currentCharacter characterType classificationTable numberType separatorsInLiterals extendedLiterals saveComments comments extendedLanguage errorBlock nameSpaceCharacter '
	classVariableNames: 'ClassificationTable MetaVariableCharacter '
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRScanner methodsFor: 'accessing'!

classificationTable: anObject
	classificationTable := anObject!

contents
	| contentsStream |
	contentsStream := WriteStream on: (Array new: 50).
	self do: [:each | contentsStream nextPut: each].
	^contentsStream contents!

errorBlock: aBlock 
	errorBlock := aBlock!

extendedLanguage
	^extendedLanguage!

extendedLanguage: aBoolean 
	extendedLanguage := aBoolean!

flush!

getComments
	| oldComments |
	comments isEmpty ifTrue: [^nil].
	oldComments := comments.
	comments := OrderedCollection new: 1.
	^oldComments!

ignoreComments
	saveComments := false!

next
	| token |
	buffer reset.
	tokenStart := stream position.
	characterType == #eof ifTrue: [^BRToken start: tokenStart + 1].	"The EOF token should occur after the end of input"
	token := self scanToken.
	self stripSeparators.
	^token!

nextPut: anObject 
	"Provide an error notification that the receiver does not
	implement this message."

	self shouldNotImplement!

saveComments
	saveComments := true!

scanToken
	"fast-n-ugly. Don't write stuff like this. Has been found to cause cancer in laboratory rats. Basically a 
	case statement. Didn't use Dictionary because lookup is pretty slow."

	characterType == #alphabetic ifTrue: [^self scanIdentifierOrKeyword].
	(characterType == #digit
		or: [currentCharacter == $- and: [(self classify: stream peek) == #digit]])
			ifTrue: [^self scanNumber].
	characterType == #binary ifTrue: [^self scanBinary: BRBinarySelectorToken].
	characterType == #special ifTrue: [^self scanSpecialCharacter].
	currentCharacter == $' ifTrue: [^self scanLiteralString].
	currentCharacter == $# ifTrue: [^self scanLiteral].
	currentCharacter == $$ ifTrue: [^self scanLiteralCharacter].
	(extendedLanguage and: [currentCharacter == MetaVariableCharacter])
		ifTrue: [^self scanMetaVariable].
	^self scannerError: 'Unknown character'! !

!BRScanner methodsFor: 'error handling'!

errorBlock
	^errorBlock isNil
		ifTrue: [[:message :position | ]]
		ifFalse: [errorBlock]!

errorPosition
	^stream position!

scannerError: aString 
	"Evaluate the block. If it returns raise an error"

	self errorBlock value: aString value: self errorPosition.
	self error: aString! !

!BRScanner methodsFor: 'initialize-release'!

initializeForIBM
	numberType := #scanNumberIBM.
	separatorsInLiterals := false.
	extendedLiterals := true.
	nameSpaceCharacter := $:!

initializeForVisualWorks
	numberType := #scanNumberVisualWorks.
	separatorsInLiterals := true.
	extendedLiterals := false.
	(Smalltalk includesKey: #QualifiedName) ifTrue: [nameSpaceCharacter := $.]!

on: aStream 
	buffer := WriteStream on: (String new: 60).
	stream := aStream.
	classificationTable := self class classificationTable.
	saveComments := true.
	extendedLanguage := false.
	comments := OrderedCollection new.
	BRParser isIBM
		ifTrue: [self initializeForIBM]
		ifFalse: [self initializeForVisualWorks]! !

!BRScanner methodsFor: 'private'!

classify: aCharacter 
	| index |
	aCharacter isNil ifTrue: [^nil].
	index := aCharacter asInteger.
	index == 0 ifTrue: [^#separator].
	index > 255 ifTrue: [^nil].
	^classificationTable at: index!

previousStepPosition
	^characterType == #eof 
		ifTrue: [stream position]
		ifFalse: [stream position - 1]!

step
	stream atEnd ifTrue: 
			[characterType := #eof.
			^currentCharacter := nil].
	currentCharacter := stream next.
	characterType := self classify: currentCharacter.
	^currentCharacter! !

!BRScanner methodsFor: 'private-scanning'!

scanAnySymbol
	characterType == #alphabetic ifTrue: [^self scanSymbol].
	characterType == #binary ifTrue: [^self scanBinary: BRLiteralToken].
	^BRToken new!

scanBinary: aClass 
	"This doesn't parse according to the ANSI draft. It only parses 1 or 2 letter binary tokens."

	| val |
	buffer nextPut: currentCharacter.
	self step.
	(characterType == #binary and: [currentCharacter ~~ $-]) ifTrue: 
			[buffer nextPut: currentCharacter.
			self step].
	val := buffer contents.
	val := val asSymbol.
	^aClass value: val start: tokenStart!

scanByteArray
	| byteStream number |
	byteStream := WriteStream on: (ByteArray new: 100).
	self step.
	
	[self stripSeparators.
	characterType == #digit] whileTrue: 
				[number := self scanNumber value.
				(number isInteger and: [number between: 0 and: 255]) 
					ifFalse: [self scannerError: 'Expecting 8-bit integer'].
				byteStream nextPut: number].
	currentCharacter == $] ifFalse: [self scannerError: ''']'' expected'].
	self step.	"]"
	^BRLiteralToken 
		value: byteStream contents
		start: tokenStart
		stop: self previousStepPosition!

scanExponentMultipler
	| exponent isExpNegative position |
	currentCharacter == $e
		ifTrue: 
			[position := stream position.
			self step.
			(isExpNegative := currentCharacter == $-) ifTrue: [self step].
			exponent := self scanNumberOfBase: 10.
			exponent isNil
				ifTrue: 
					["Did not read a valid exponent, e must be start of a message send"

					stream position: position - 1.
					self step.
					exponent := 0]
				ifFalse: [isExpNegative ifTrue: [exponent := exponent negated]]]
		ifFalse: [exponent := 0].
	^10 raisedToInteger: exponent!

scanExtendedLiterals
	| token |
	self step.
	separatorsInLiterals ifTrue: [self stripSeparators].
	token := characterType == #alphabetic 
				ifTrue: [self scanSymbol]
				ifFalse: 
					[characterType == #binary 
						ifTrue: [(self scanBinary: BRLiteralToken) stop: self previousStepPosition]
						ifFalse: 
							[currentCharacter == $' 
								ifTrue: [self scanStringSymbol]
								ifFalse: 
									[currentCharacter == $( 
										ifTrue: 
											[self step.
											^BROptimizedToken start: tokenStart]]]].
	token isNil ifTrue: [self scannerError: 'Expecting a extended literal'].
	token value: ((Smalltalk at: #EsAtom) intern: token value asString).
	^token!

scanIdentifierOrKeyword
	| tokenType token |
	self scanName.
	currentCharacter == nameSpaceCharacter 
		ifTrue: 
			[token := self scanNamespaceName.
			token notNil ifTrue: [^token]].
	(currentCharacter == $: and: [stream peek ~~ $=]) 
		ifTrue: 
			[buffer nextPut: currentCharacter.
			self step.	":"
			tokenType := BRKeywordToken]
		ifFalse: [tokenType := BRIdentifierToken].
	^tokenType value: buffer contents start: tokenStart!

scanLiteral
	self step.
	separatorsInLiterals ifTrue: [self stripSeparators].
	characterType == #alphabetic ifTrue: [^self scanSymbol].
	characterType == #binary 
		ifTrue: [^(self scanBinary: BRLiteralToken) stop: self previousStepPosition].
	currentCharacter == $' ifTrue: [^self scanStringSymbol].
	currentCharacter == $( ifTrue: [^self scanLiteralArray].
	currentCharacter == $[ ifTrue: [^self scanByteArray].
	(separatorsInLiterals and: [currentCharacter == ${]) 
		ifTrue: [^self scanQualifier].
	(extendedLiterals and: [currentCharacter == $#]) 
		ifTrue: [^self scanExtendedLiterals].
	self scannerError: 'Expecting a literal type'!

scanLiteralArray
	| arrayStream |
	arrayStream := WriteStream on: (Array new: 10).
	self step.
	
	[self stripSeparators.
	currentCharacter == $)] whileFalse: 
				[arrayStream nextPut: self scanLiteralArrayParts.
				buffer reset].
	self step.
	^BRLiteralToken 
		value: arrayStream contents
		start: tokenStart
		stop: self previousStepPosition!

scanLiteralArrayParts
	currentCharacter == $# ifTrue: [^self scanLiteral].
	characterType == #alphabetic 
		ifTrue: 
			[| token value |
			token := self scanSymbol.
			value := token value.
			value == #nil ifTrue: [token value: nil].
			value == #true ifTrue: [token value: true].
			value == #false ifTrue: [token value: false].
			^token].
	(characterType == #digit 
		or: [currentCharacter == $- and: [(self classify: stream peek) == #digit]]) 
			ifTrue: [^self scanNumber].
	characterType == #binary 
		ifTrue: [^(self scanBinary: BRLiteralToken) stop: self previousStepPosition].
	currentCharacter == $' ifTrue: [^self scanLiteralString].
	currentCharacter == $$ ifTrue: [^self scanLiteralCharacter].
	currentCharacter == $( ifTrue: [^self scanLiteralArray].
	currentCharacter == $[ ifTrue: [^self scanByteArray].
	^self scannerError: 'Unknown character in literal array'!

scanLiteralCharacter
	| token |
	self step.	"$"
	token := BRLiteralToken value: currentCharacter
				start: tokenStart
				stop: stream position.
	self step.	"char"
	^token!

scanLiteralString
	self step.
	
	[currentCharacter isNil 
		ifTrue: [self scannerError: 'Unmatched '' in string literal.'].
	currentCharacter == $' and: [self step ~~ $']] 
			whileFalse: 
				[buffer nextPut: currentCharacter.
				self step].
	^BRLiteralToken 
		value: buffer contents
		start: tokenStart
		stop: self previousStepPosition!

scanMetaVariable
	[characterType == #alphabetic] whileFalse: 
			[characterType == #eof ifTrue: [self scannerError: 'Meta variable expected'].
			buffer nextPut: currentCharacter.
			self step].
	^self scanIdentifierOrKeyword!

scanName
	[characterType == #alphabetic or: [characterType == #digit]] whileTrue: 
			[buffer nextPut: currentCharacter.
			self step]!

scanNamespaceName
	extendedLiterals 
		ifTrue: 
			[stream peek == $: ifFalse: [^nil].
			buffer next: 2 put: $:.
			self step]
		ifFalse: 
			[(stream atEnd or: [(self classify: stream peek) ~~ #alphabetic]) 
				ifTrue: [^nil].
			buffer nextPut: $.].
	self step.
	self scanName.
	currentCharacter == nameSpaceCharacter ifTrue: [self scanNamespaceName].
	^BRIdentifierToken value: buffer contents start: tokenStart!

scanNumber
	^BRLiteralToken 
		value: (self perform: numberType)
		start: tokenStart
		stop: self previousStepPosition!

scanNumberIBM
	"Did not read a correct number, r must be start of a message send."

	| number isNegative |
	isNegative := false.
	currentCharacter == $- ifTrue: 
			[isNegative := true.
			self step].
	number := self scanNumberWithoutExponent.
	^(isNegative ifTrue: [number negated] ifFalse: [number])
		* self scanExponentMultipler!

scanNumberOfBase: anInteger 
	"Scan a number. Return the number or nil if the current input isn't a valid number."

	| number digits fraction isFloat succeeded |
	digits := '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ' copyFrom: 1 to: anInteger.
	number := 0.
	succeeded := false.
	[digits includes: currentCharacter] whileTrue: 
			[number := number * anInteger + (digits indexOf: currentCharacter) - 1.
			self step.
			succeeded := true].
	succeeded ifFalse: [^nil].
	isFloat := false.
	(currentCharacter == $. and: [digits includes: stream peek]) ifTrue: 
			[self step.
			isFloat := true.
			fraction := 1 / anInteger.
			[digits includes: currentCharacter] whileTrue: 
					[number := number + (((digits indexOf: currentCharacter) - 1) * fraction).
					fraction := fraction / anInteger.
					self step]].
	^isFloat ifTrue: [number asFloat] ifFalse: [number]!

scanNumberVisualWorks
	| number |
	stream skip: -1.
	number := Number readSmalltalkSyntaxFrom: stream.
	self step.
	^number!

scanNumberWithoutExponent
	"Scan an IBM number with the radix -- don't scan the exponent though"

	| number base |
	base := self scanNumberOfBase: 10.
	(currentCharacter == $r and: [base isInteger])
		ifTrue: 
			[| position |
			position := stream position.
			self step.
			number := self scanNumberOfBase: base.
			number isNil ifTrue: 
					["Did not read a correct number, r must be start of a message send."

					stream position: position - 1.
					self step.
					number := base]]
		ifFalse: [number := base].
	^number!

scanQualifier
	| nameStream qualifierClass |
	qualifierClass := Smalltalk at: #QualifiedName ifAbsent: [nil].
	qualifierClass isNil ifTrue: [^self scannerError: 'Unknown character'].
	self step.	"{"
	nameStream := WriteStream on: (String new: 10).
	[currentCharacter == $}] whileFalse: 
			[nameStream nextPut: currentCharacter.
			self step].
	self step.	"}"
	^BRLiteralToken 
		value: (qualifierClass pathString: nameStream contents)
		start: tokenStart
		stop: self previousStepPosition!

scanSpecialCharacter
	| character |
	currentCharacter == $: ifTrue: 
			[self step.
			^currentCharacter == $=
				ifTrue: 
					[self step.
					BRAssignmentToken start: tokenStart]
				ifFalse: [BRSpecialCharacterToken value: $: start: tokenStart]].
	character := currentCharacter.
	self step.
	^BRSpecialCharacterToken value: character start: tokenStart!

scanStringSymbol
	| literalToken |
	literalToken := self scanLiteralString.
	literalToken value: literalToken value asSymbol.
	^literalToken!

scanSymbol
	| lastPosition hasColon value startPosition |
	hasColon := false.
	startPosition := lastPosition := stream position.
	[characterType == #alphabetic] whileTrue: 
			[self scanName.
			currentCharacter == $: 
				ifTrue: 
					[buffer nextPut: $:.
					hasColon := true.
					lastPosition := stream position.
					self step]].
	value := buffer contents.
	(hasColon and: [value last ~~ $:]) 
		ifTrue: 
			[stream position: lastPosition.
			self step.
			value := value copyFrom: 1 to: lastPosition - startPosition + 1].
	^BRLiteralToken 
		value: value asSymbol
		start: tokenStart
		stop: self previousStepPosition!

stripComment
	| start stop |
	start := stream position.
	[self step == $"] whileFalse: 
			[characterType == #eof
				ifTrue: [self scannerError: 'Unmatched " in comment.']].
	stop := stream position.
	self step.
	saveComments ifFalse: [^self].
	comments add: (start to: stop)!

stripSeparators
	
	[[characterType == #separator]
		whileTrue: [self step].
	currentCharacter == $"]
		whileTrue: [self stripComment]! !

!BRScanner methodsFor: 'testing'!

atEnd
	^characterType == #eof!

isReadable
	^true!

isWritable
	^false! !

BRScanner class
	instanceVariableNames: ''!



!BRScanner class methodsFor: 'accessing'!

classificationTable
	ClassificationTable isNil ifTrue: [self initialize].
	^ClassificationTable!

metaVariableCharacter
	^MetaVariableCharacter! !

!BRScanner class methodsFor: 'class initialization'!

initialize
	MetaVariableCharacter := $`.
	ClassificationTable := Array new: 255.
	self initializeChars: 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_' to: #alphabetic.
	self initializeChars: '01234567890' to: #digit.
	self initializeChars: '!!%&*+,-/<=>?@\~|' to: #binary.
	ClassificationTable at: 177 put: #binary.	"plus-or-minus"
	ClassificationTable at: 183 put: #binary.	"centered dot"
	ClassificationTable at: 215 put: #binary.	"times"
	ClassificationTable at: 247 put: #binary.	"divide"
	self initializeChars: '().:;[]^' to: #special.
	#(9 10 12 13 26 32) do: [:i | ClassificationTable at: i put: #separator]!

initializeChars: characters to: aSymbol 
	characters do: [:c | ClassificationTable at: c asInteger put: aSymbol]! !

!BRScanner class methodsFor: 'instance creation'!

on: aStream
	| str |
	str := self basicNew on: aStream.
	str step.
	str stripSeparators.
	^str!

on: aStream errorBlock: aBlock 
	| str |
	str := self basicNew on: aStream.
	str errorBlock: aBlock;
		step;
		stripSeparators.
	^str!

rewriteOn: aStream 
	| str |
	str := self basicNew on: aStream.
	str extendedLanguage: true; ignoreComments.
	str step.
	str stripSeparators.
	^str!

rewriteOn: aStream errorBlock: aBlock
	| str |
	str := self basicNew on: aStream.
	str extendedLanguage: true;
		ignoreComments;
		errorBlock: aBlock;
		step;
		stripSeparators.
	^str! !

!BRScanner class methodsFor: 'testing'!

isSelector: aSymbol 
	| scanner token |
	scanner := self basicNew.
	scanner on: (ReadStream on: aSymbol asString).
	scanner step.
	token := scanner scanAnySymbol.
	token isLiteral ifFalse: [^false].
	token value isEmpty ifTrue: [^false].
	^scanner atEnd!

isVariable: aString 
	| scanner token |
	aString isString ifFalse: [^false].
	aString isEmpty ifTrue: [^false].
	(ClassificationTable at: aString first asInteger) == #alphabetic
		ifFalse: [^false].
	scanner := self basicNew.
	scanner on: (ReadStream on: aString asString).
	scanner errorBlock: [:s :p | ^false].
	scanner step.
	token := scanner scanIdentifierOrKeyword.
	token isKeyword ifTrue: [^false].
	^scanner atEnd! !

KeyedCollection subclass: #BRSmallDictionary
	instanceVariableNames: 'keys values size '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRSmallDictionary methodsFor: 'initialize-release'!

initialize
	keys := Array new: 2.
	values := Array new: 2.
	size := 0! !

!BRSmallDictionary methodsFor: 'accessing'!

at: key ifAbsent: aBlock 
	| index |
	index := self findIndexFor: key.
	^index == 0 ifTrue: [aBlock value] ifFalse: [values at: index]!

at: key ifAbsentPut: aBlock 
	| index |
	index := self findIndexFor: key.
	^index == 0 
		ifTrue: [self privateAt: key put: aBlock value]
		ifFalse: [values at: index]!

empty
	size := 0!

size
	^size! !

!BRSmallDictionary methodsFor: 'testing'!

includesKey: aKey 
	^(self findIndexFor: aKey) ~~ 0! !

!BRSmallDictionary methodsFor: 'adding'!

add: anAssociation 
	self at: anAssociation key put: anAssociation value.
	^anAssociation!

at: key put: value 
	| index |
	index := self findIndexFor: key.
	^index == 0 
		ifTrue: [self privateAt: key put: value]
		ifFalse: [values at: index put: value]! !

!BRSmallDictionary methodsFor: 'removing'!

remove: oldObject ifAbsent: anExceptionBlock 
	self removeKey: oldObject key ifAbsent: anExceptionBlock.
	^oldObject!

removeKey: key ifAbsent: aBlock 
	| index value |
	index := self findIndexFor: key.
	index == 0 ifTrue: [^aBlock value].
	value := values at: index.
	index to: size - 1
		do: 
			[:i | 
			keys at: i put: (keys at: i + 1).
			values at: i put: (values at: i + 1)].
	keys at: size put: nil.
	values at: size put: nil.
	size := size - 1.
	^value! !

!BRSmallDictionary methodsFor: 'enumerating'!

associationsDo: aBlock 
	self keysAndValuesDo: [:key :value | aBlock value: key -> value]!

do: aBlock 
	1 to: size do: [:i | aBlock value: (values at: i)]!

keysAndValuesDo: aBlock 
	1 to: size do: [:i | aBlock value: (keys at: i) value: (values at: i)]!

keysDo: aBlock 
	1 to: size do: [:i | aBlock value: (keys at: i)]! !

!BRSmallDictionary methodsFor: 'copying'!

postCopy
	keys := keys copy.
	values := values copy! !

!BRSmallDictionary methodsFor: 'private'!

findIndexFor: aKey 
	1 to: size do: [:i | (keys at: i) = aKey ifTrue: [^i]].
	^0!

growKeysAndValues
	self growTo: size * 2!

growTo: aSize 
	| newKeys newValues |
	newKeys := Array new: aSize.
	newValues := Array new: aSize.
	1 to: size
		do: 
			[:i | 
			newKeys at: i put: (keys at: i).
			newValues at: i put: (values at: i)].
	keys := newKeys.
	values := newValues!

privateAt: key put: value 
	size == keys size ifTrue: [self growKeysAndValues].
	size := size + 1.
	keys at: size put: key.
	^values at: size put: value! !

BRSmallDictionary class
	instanceVariableNames: ''!



!BRSmallDictionary class methodsFor: 'instance creation'!

new
	^self basicNew initialize!

new: aSize 
	"Ignore the size"

	^self basicNew initialize! !

Object subclass: #BRParseTreeRule
	instanceVariableNames: 'searchTree owner '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRParseTreeRule methodsFor: 'initialize-release'!

initialize!

methodSearchString: aString 
	searchTree := BRParser parseRewriteMethod: aString!

owner: aParseTreeSearcher
	owner := aParseTreeSearcher!

searchString: aString 
	searchTree := BRParser parseRewriteExpression: aString! !

!BRParseTreeRule methodsFor: 'matching'!

canMatch: aProgramNode 
	^true!

foundMatchFor: aProgramNode
	^aProgramNode!

performOn: aProgramNode 
	self context empty.
	^((searchTree match: aProgramNode inContext: self context) 
		and: [self canMatch: aProgramNode]) 
			ifTrue: 
				[owner recusivelySearchInContext.
				self foundMatchFor: aProgramNode]
			ifFalse: [nil]! !

!BRParseTreeRule methodsFor: 'private'!

context
	^owner context! !

BRParseTreeRule class
	instanceVariableNames: ''!



!BRParseTreeRule class methodsFor: 'instance creation'!

methodSearch: aString 
	^(self new)
		methodSearchString: aString;
		yourself!

new
	^(super new)
		initialize;
		yourself!

search: aString 
	^(self new)
		searchString: aString;
		yourself! !

BRParseTreeRule subclass: #BRSearchRule
	instanceVariableNames: 'answerBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRSearchRule methodsFor: 'initialize-release'!

searchFor: aString thenDo: aBlock 
	self searchString: aString.
	answerBlock := aBlock!

searchForMethod: aString thenDo: aBlock 
	self methodSearchString: aString.
	answerBlock := aBlock!

searchForTree: aBRProgramNode thenDo: aBlock 
	searchTree := aBRProgramNode.
	answerBlock := aBlock! !

!BRSearchRule methodsFor: 'testing'!

canMatch: aProgramNode 
	owner answer: (answerBlock value: aProgramNode value: owner answer).
	^true! !

BRSearchRule class
	instanceVariableNames: ''!



!BRSearchRule class methodsFor: 'instance creation'!

searchFor: aString thenDo: aBlock 
	^self new searchFor: aString thenDo: aBlock!

searchForMethod: aString thenDo: aBlock 
	^self new searchForMethod: aString thenDo: aBlock!

searchForTree: aBRProgramNode thenDo: aBlock 
	^self new searchForTree: aBRProgramNode thenDo: aBlock! !

BRProgramNode subclass: #BRStatementNode
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRStatementNode comment:
'BRStatementNode is an abstract class that represents AST nodes that can go in sequence nodes.

'!

BRStatementNode class
	instanceVariableNames: ''!


BRStatementNode subclass: #BRValueNode
	instanceVariableNames: 'parentheses '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRValueNode comment:
'BRValueNode is an abstract class that represents a node that returns some value.

Instance Variables:
	parentheses	<SequenceableCollection of: Inteval>	the positions of the parethesis around this node. We need a collection of intervals for stupid code such as "((3 + 4))" that has multiple parethesis around the same expression.

'!


!BRValueNode methodsFor: 'accessing'!

addParenthesis: anInterval 
	parentheses isNil ifTrue: [parentheses := OrderedCollection new: 1].
	parentheses add: anInterval!

parentheses
	^parentheses isNil
		ifTrue: [#()]
		ifFalse: [parentheses]!

start
	^parentheses isNil 
		ifTrue: [self startWithoutParentheses]
		ifFalse: [parentheses last first]!

startWithoutParentheses
	^self subclassResponsibility!

stop
	^parentheses isNil
		ifTrue: [self stopWithoutParentheses]
		ifFalse: [parentheses last last]!

stopWithoutParentheses
	^self subclassResponsibility! !

!BRValueNode methodsFor: 'testing'!

containedBy: anInterval 
	^anInterval first <= self startWithoutParentheses 
		and: [anInterval last >= self stopWithoutParentheses]!

hasParentheses
	^self parentheses isEmpty not!

isValue
	^true! !

BRValueNode class
	instanceVariableNames: ''!


BRValueNode subclass: #BRBlockNode
	instanceVariableNames: 'left right colons body arguments bar '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRBlockNode comment:
'BRBlockNode is an AST node that represents a block "[...]".

Instance Variables:
	arguments	<SequenceableCollection of: BRVariableNode>	the arguments for the block
	bar	<Integer | nil>	position of the | after the arguments
	body	<BRSequenceNode>	the code inside the block
	colons	<SequenceableCollection of: Integer>	positions of each : before each argument
	left	<Integer>	position of [
	right	<Integer>	position of ]

'!


!BRBlockNode methodsFor: 'accessing'!

allArgumentVariables
	^(self argumentNames asOrderedCollection)
		addAll: super allArgumentVariables;
		yourself!

allDefinedVariables
	^(self argumentNames asOrderedCollection) addAll: super allDefinedVariables;
		yourself!

argumentNames
	^self arguments collect: [:each | each name]!

arguments
	^arguments!

arguments: argCollection 
	arguments := argCollection.
	arguments do: [:each | each parent: self]!

bar
	^bar!

bar: anObject
	bar := anObject!

blockVariables
	| vars |
	vars := super blockVariables asOrderedCollection.
	vars addAll: self argumentNames.
	^vars!

body
	^body!

body: stmtsNode 
	body := stmtsNode.
	body parent: self!

children
	^self arguments copyWith: self body!

colons: anObject
	colons := anObject!

left
	^left!

left: anObject
	left := anObject!

precedence
	^0!

right
	^right!

right: anObject
	right := anObject!

startWithoutParentheses
	^left!

stopWithoutParentheses
	^right! !

!BRBlockNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	self body = anObject body ifFalse: [^false].
	self arguments size = anObject arguments size ifFalse: [^false].
	1 to: self arguments size
		do: [:i | (self arguments at: i) = (anObject arguments at: i) ifFalse: [^false]].
	^true!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	self arguments size = anObject arguments size ifFalse: [^false].
	1 to: self arguments size
		do: 
			[:i | 
			((self arguments at: i) equalTo: (anObject arguments at: i)
				withMapping: aDictionary) ifFalse: [^false]].
	(self body equalTo: anObject body withMapping: aDictionary)
		ifFalse: [^false].
	self arguments do: [:each | aDictionary removeKey: each name].
	^true!

hash
	^self arguments hash bitXor: self body hash! !

!BRBlockNode methodsFor: 'copying'!

postCopy
	super postCopy.
	arguments := arguments collect: [:each | each copy].
	body := body copy! !

!BRBlockNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new)
		arguments: (self copyList: arguments inContext: aDictionary);
		body: (body copyInContext: aDictionary);
		yourself!

match: aNode inContext: aDictionary 
	aNode class == self class ifFalse: [^false].
	^(self matchList: arguments
		against: aNode arguments
		inContext: aDictionary)
			and: [body match: aNode body inContext: aDictionary]! !

!BRBlockNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	body == aNode ifTrue: [self body: anotherNode].
	self arguments: (arguments 
				collect: [:each | each == aNode ifTrue: [anotherNode] ifFalse: [each]])! !

!BRBlockNode methodsFor: 'testing'!

defines: aName 
	^(arguments detect: [:each | each name = aName] ifNone: [nil]) notNil!

directlyUses: aNode 
	^false!

isBlock
	^true!

isImmediate
	^true!

isLast: aNode 
	^body isLast: aNode!

references: aVariableName 
	^body references: aVariableName!

uses: aNode 
	aNode = body ifFalse: [^false].
	^parent isMessage
		ifTrue: [(#(#ifTrue:ifFalse: #ifTrue: #ifFalse: #ifFalse:ifTrue:) includes: parent selector) not or: [parent isUsed]]
		ifFalse: [self isUsed]! !

!BRBlockNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptBlockNode: self! !

BRBlockNode class
	instanceVariableNames: ''!



!BRBlockNode class methodsFor: 'instance creation'!

arguments: argNodes body: sequenceNode 
	^(self new)
		arguments: argNodes;
		body: sequenceNode;
		yourself!

body: sequenceNode 
	^self arguments: #() body: sequenceNode! !

BRValueNode subclass: #BRMessageNode
	instanceVariableNames: 'receiver selector selectorParts arguments '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRMessageNode comment:
'BRMessageNode is an AST node that represents a message send.

Instance Variables:
	arguments	<SequenceableCollection of: BRValueNode>	our argument nodes
	receiver	<BRValueNode>	the receiver''s node
	selector	<Symbol | nil>	the selector we''re sending (cached)
	selectorParts	<SequenceableCollection of: BRValueToken>	the tokens for each keyword

'!


!BRMessageNode methodsFor: 'accessing'!

arguments
	^arguments isNil
		ifTrue: [#()]
		ifFalse: [arguments]!

arguments: argCollection 
	arguments := argCollection.
	arguments do: [:each | each parent: self]!

children
	^(OrderedCollection with: self receiver) addAll: self arguments;
		yourself!

precedence
	^self isUnary ifTrue: [1] ifFalse: [self isKeyword ifTrue: [3] ifFalse: [2]]!

receiver
	^receiver!

receiver: aValueNode 
	receiver := aValueNode.
	receiver parent: self!

selector
	^selector isNil
		ifTrue: [selector := self buildSelector]
		ifFalse: [selector]!

selector: aSelector 
	| keywords numArgs |
	keywords := aSelector keywords.
	numArgs := aSelector numArgs.
	numArgs == arguments size ifFalse: 
			[self error: 'Attempting to assign selector with wrong number of arguments.'].
	selectorParts := numArgs == 0
				ifTrue: [Array with: (BRIdentifierToken value: keywords first start: nil)]
				ifFalse: 
					[keywords first last == $:
						ifTrue: [keywords collect: [:each | BRKeywordToken value: each start: nil]]
						ifFalse: [Array with: (BRBinarySelectorToken value: aSelector start: nil)]].
	selector := aSelector!

startWithoutParentheses
	^receiver start!

stopWithoutParentheses
	^arguments isEmpty 
		ifTrue: [selectorParts first stop]
		ifFalse: [arguments last stop]! !

!BRMessageNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	(self receiver = anObject receiver 
		and: [self selector = anObject selector]) ifFalse: [^false].
	1 to: self arguments size
		do: [:i | (self arguments at: i) = (anObject arguments at: i) ifFalse: [^false]].
	^true!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	((self receiver equalTo: anObject receiver withMapping: aDictionary)
		and: [self selector = anObject selector]) ifFalse: [^false].
	1 to: self arguments size
		do: 
			[:i | 
			((self arguments at: i) equalTo: (anObject arguments at: i)
				withMapping: aDictionary) ifFalse: [^false]].
	^true!

hash
	^(self receiver hash bitXor: self selector hash)
		bitXor: (self arguments isEmpty ifTrue: [0] ifFalse: [self arguments first hash])! !

!BRMessageNode methodsFor: 'copying'!

postCopy
	super postCopy.
	receiver := receiver copy.
	arguments := arguments collect: [:each | each copy]! !

!BRMessageNode methodsFor: 'initialize-release'!

receiver: aValueNode selectorParts: keywordTokens arguments: valueNodes 
	self receiver: aValueNode.
	selectorParts := keywordTokens.
	self arguments: valueNodes! !

!BRMessageNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new) receiver: (receiver copyInContext: aDictionary);
		selectorParts: (selectorParts collect: [:each | each removePositions]);
		arguments: (arguments collect: [:each | each copyInContext: aDictionary]);
		yourself!

match: aNode inContext: aDictionary 
	aNode class == self class ifFalse: [^false].
	self selector == aNode selector ifFalse: [^false].
	(receiver match: aNode receiver inContext: aDictionary) ifFalse: [^false].
	1 to: arguments size
		do: 
			[:i | 
			((arguments at: i) match: (aNode arguments at: i) inContext: aDictionary)
				ifFalse: [^false]].
	^true! !

!BRMessageNode methodsFor: 'private'!

buildSelector
	| selectorStream |
	selectorStream := WriteStream on: (String new: 50).
	selectorParts do: [:each | selectorStream nextPutAll: each value].
	^selectorStream contents asSymbol!

selectorParts
	^selectorParts!

selectorParts: tokenCollection 
	selectorParts := tokenCollection! !

!BRMessageNode methodsFor: 'testing'!

isBinary
	^(self isUnary or: [self isKeyword]) not!

isKeyword
	^selectorParts first value last == $:!

isMessage
	^true!

isUnary
	^arguments isEmpty!

lastIsReturn
	^(#(#ifTrue:ifFalse: #ifFalse:ifTrue:) includes: self selector) and: 
			[arguments first isBlock and: 
					[arguments first body lastIsReturn
						and: [arguments last isBlock and: [arguments last body lastIsReturn]]]]! !

!BRMessageNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptMessageNode: self! !

!BRMessageNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	"If we're inside a cascade node and are changing the receiver, change all the receivers"

	receiver == aNode 
		ifTrue: 
			[self receiver: anotherNode.
			(parent notNil and: [parent isCascade]) 
				ifTrue: [parent messages do: [:each | each receiver: anotherNode]]].
	self arguments: (arguments 
				collect: [:each | each == aNode ifTrue: [anotherNode] ifFalse: [each]])! !

!BRMessageNode methodsFor: 'querying'!

bestNodeFor: anInterval 
	(self intersectsInterval: anInterval) ifFalse: [^nil].
	(self containedBy: anInterval) ifTrue: [^self].
	selectorParts do: 
			[:each | 
			((anInterval first between: each start and: each stop) 
				or: [each start between: anInterval first and: anInterval last]) 
					ifTrue: [^self]].
	self children do: 
			[:each | 
			| node |
			node := each bestNodeFor: anInterval.
			node notNil ifTrue: [^node]]! !

BRMessageNode class
	instanceVariableNames: ''!



!BRMessageNode class methodsFor: 'instance creation'!

receiver: aValueNode selector: aSymbol 
	^self 
		receiver: aValueNode
		selector: aSymbol
		arguments: #()!

receiver: aValueNode selector: aSymbol arguments: valueNodes 
	^(self new)
		receiver: aValueNode;
		arguments: valueNodes;
		selector: aSymbol;
		yourself!

receiver: aValueNode selectorParts: keywordTokens arguments: valueNodes 
	^((keywordTokens detect: [:each | each isMetaVariable] ifNone: [nil])
		notNil ifTrue: [BRMetaMessageNode] ifFalse: [BRMessageNode]) new
			receiver: aValueNode
			selectorParts: keywordTokens
			arguments: valueNodes! !

BRValueNode subclass: #BROptimizedNode
	instanceVariableNames: 'left right body '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BROptimizedNode comment:
'BROptimizedNode is an AST node that represents IBM''s ##(...) expressions. These expressions are evaluated at compile time and directly inserted into the method.

Instance Variables:
	body	<BRSequenceNode>	the body of the expression
	left	<Integer>	position of the ##( characters
	right	<Integer>	position of )

'!


!BROptimizedNode methodsFor: 'accessing'!

body
	^body!

body: stmtsNode 
	body := stmtsNode.
	body parent: self!

children
	^Array with: body!

precedence
	^0!

startWithoutParentheses
	^left!

stopWithoutParentheses
	^right! !

!BROptimizedNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	^self body = anObject body!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	^self body equalTo: anObject body withMapping: aDictionary!

hash
	^self body hash! !

!BROptimizedNode methodsFor: 'initialize-release'!

left: leftInteger body: aSequenceNode right: rightInteger 
	left := leftInteger.
	self body: aSequenceNode.
	right := rightInteger! !

!BROptimizedNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^self class body: (body copyInContext: aDictionary)!

match: aNode inContext: aDictionary 
	aNode class == self class ifFalse: [^false].
	^body match: aNode body inContext: aDictionary! !

!BROptimizedNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptOptimizedNode: self! !

!BROptimizedNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	body == aNode ifTrue: [self body: anotherNode]! !

!BROptimizedNode methodsFor: 'testing'!

isImmediate
	^true! !

BROptimizedNode class
	instanceVariableNames: ''!



!BROptimizedNode class methodsFor: 'instance creation'!

body: aSequenceNode 
	^self new body: aSequenceNode!

left: leftInteger body: aSequenceNode right: rightInteger 
	^self new
		left: leftInteger
		body: aSequenceNode
		right: rightInteger! !

BRProgramNode subclass: #BRMethodNode
	instanceVariableNames: 'selector selectorParts body source arguments tags '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRMethodNode comment:
'BRMethodNode is the AST that represents a Smalltalk method.

Instance Variables:
	arguments	<SequenceableCollection of: BRVariableNode>	the arguments to the method
	body	<BRSequenceNode>	the body/statements of the method
	selector	<Symbol | nil>	the method name (cached)
	selectorParts	<SequenceableCollection of: BRValueToken>	the tokens for the selector keywords
	source	<String>	the source we compiled
	tag	<Interval | nil>	the source location of any resource/primitive tags

'!


!BRMethodNode methodsFor: 'accessing'!

addNode: aNode 
	^body addNode: aNode!

addReturn
	body addReturn!

addSelfReturn
	^body addSelfReturn!

allArgumentVariables
	^(self argumentNames asOrderedCollection)
		addAll: super allArgumentVariables;
		yourself!

allDefinedVariables
	^(self argumentNames asOrderedCollection) addAll: super allDefinedVariables;
		yourself!

argumentNames
	^self arguments collect: [:each | each name]!

arguments
	^arguments!

arguments: variableNodes 
	arguments := variableNodes.
	arguments do: [:each | each parent: self]!

body
	^body!

body: stmtsNode 
	body := stmtsNode.
	body parent: self!

children
	^self arguments copyWith: self body!

primitiveSources
	^self tags 
		collect: [:each | self source copyFrom: each first to: each last]!

selector
	^selector isNil
		ifTrue: [selector := self buildSelector]
		ifFalse: [selector]!

selector: aSelector 
	| keywords numArgs |
	keywords := aSelector keywords.
	numArgs := aSelector numArgs.
	numArgs == arguments size ifFalse: 
			[self error: 'Attempting to assign selector with wrong number of arguments.'].
	selectorParts := numArgs == 0
				ifTrue: [Array with: (BRIdentifierToken value: keywords first start: nil)]
				ifFalse: 
					[keywords first last == $:
						ifTrue: [keywords collect: [:each | BRKeywordToken value: each start: nil]]
						ifFalse: [Array with: (BRBinarySelectorToken value: aSelector start: nil)]].
	selector := aSelector!

source
	^source!

source: anObject
	source := anObject!

start
	^1!

stop
	^source size!

tags
	^tags isNil ifTrue: [#()] ifFalse: [tags]!

tags: aCollectionOfIntervals 
	tags := aCollectionOfIntervals! !

!BRMethodNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	(self selector = anObject selector and: [self body = anObject body]) 
		ifFalse: [^false].
	1 to: self arguments size
		do: [:i | (self arguments at: i) = (anObject arguments at: i) ifFalse: [^false]].
	^true!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	(self selector = anObject selector 
		and: [self body equalTo: anObject body withMapping: aDictionary]) 
			ifFalse: [^false].
	1 to: self arguments size
		do: 
			[:i | 
			((self arguments at: i) equalTo: (anObject arguments at: i)
				withMapping: aDictionary) ifFalse: [^false].
			aDictionary removeKey: (self arguments at: i) name].
	^self primitiveSources = anObject primitiveSources!

hash
	^(self selector hash bitXor: self body hash)
		bitXor: self arguments hash! !

!BRMethodNode methodsFor: 'copying'!

postCopy
	super postCopy.
	body := body copy.
	arguments := arguments collect: [:each | each copy]! !

!BRMethodNode methodsFor: 'initialize-release'!

selectorParts: tokenCollection arguments: variableNodes 
	selectorParts := tokenCollection.
	self arguments: variableNodes! !

!BRMethodNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new)
		selectorParts: (selectorParts collect: [:each | each removePositions]);
		arguments: (arguments collect: [:each | each copyInContext: aDictionary]);
		body: (body copyInContext: aDictionary);
		source: (aDictionary at: '-source-');
		yourself!

match: aNode inContext: aDictionary 
	self class == aNode class ifFalse: [^false].
	aDictionary at: '-source-' put: aNode source.
	self selector == aNode selector ifFalse: [^false].
	^(self matchList: arguments
		against: aNode arguments
		inContext: aDictionary)
			and: [body match: aNode body inContext: aDictionary]! !

!BRMethodNode methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: self formattedCode! !

!BRMethodNode methodsFor: 'private'!

buildSelector
	| selectorStream |
	selectorStream := WriteStream on: (String new: 50).
	selectorParts do: [:each | selectorStream nextPutAll: each value].
	^selectorStream contents asSymbol!

selectorParts
	^selectorParts!

selectorParts: tokenCollection 
	selectorParts := tokenCollection! !

!BRMethodNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	aNode == body ifTrue: [self body: anotherNode].
	self arguments: (arguments 
				collect: [:each | each == aNode ifTrue: [anotherNode] ifFalse: [each]])! !

!BRMethodNode methodsFor: 'testing'!

defines: aName 
	^(arguments detect: [:each | each name = aName] ifNone: [nil]) notNil!

isLast: aNode 
	^body isLast: aNode!

isMethod
	^true!

isPrimitive
	^tags notNil and: [tags isEmpty not]!

lastIsReturn
	^body lastIsReturn!

references: aVariableName 
	^body references: aVariableName!

uses: aNode 
	^body == aNode and: [aNode lastIsReturn]! !

!BRMethodNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptMethodNode: self! !

BRMethodNode class
	instanceVariableNames: ''!



!BRMethodNode class methodsFor: 'instance creation'!

selector: aSymbol arguments: variableNodes body: aSequenceNode 
	^(self new)
		arguments: variableNodes;
		selector: aSymbol;
		body: aSequenceNode;
		yourself!

selector: aSymbol body: aSequenceNode 
	^self 
		selector: aSymbol
		arguments: #()
		body: aSequenceNode!

selectorParts: tokenCollection arguments: variableNodes 
	^((tokenCollection detect: [:each | each isMetaVariable]
		ifNone: [nil]) notNil
		ifTrue: [BRMetaMethodNode]
		ifFalse: [BRMethodNode]) new selectorParts: tokenCollection arguments: variableNodes! !

BRMethodNode subclass: #BRMetaMethodNode
	instanceVariableNames: 'isList '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRMetaMethodNode comment:
'BRMetaMethodNode is a BRMethodNode that will match other method nodes without their selectors being equal. 

Instance Variables:
	isList	<Boolean>	are we matching each keyword or matching all keywords together (e.g., `keyword1: would match a one argument method whereas `@keywords: would match 0 or more arguments)

'!


!BRMetaMethodNode methodsFor: 'initialize-release'!

selectorParts: tokenCollection arguments: variableNodes 
	super selectorParts: tokenCollection arguments: variableNodes.
	isList := (tokenCollection first value at: 2) == self listCharacter! !

!BRMetaMethodNode methodsFor: 'matching'!

copyInContext: aDictionary 
	| selectors |
	selectors := self isSelectorList
				ifTrue: [(aDictionary at: selectorParts first value) keywords]
				ifFalse: [selectorParts collect: [:each | aDictionary at: each value]].
	^(BRMethodNode new)
		selectorParts: (selectors collect: 
						[:each | 
						(each last == $: ifTrue: [BRKeywordToken] ifFalse: [BRIdentifierToken])
							value: each
							start: nil]);
		arguments: (self copyList: arguments inContext: aDictionary);
		body: (body copyInContext: aDictionary);
		source: (aDictionary at: '-source-');
		yourself!

match: aNode inContext: aDictionary 
	aNode class == self matchingClass ifFalse: [^false].
	aDictionary at: '-source-' put: aNode source.
	self isSelectorList ifTrue: 
			[^(aDictionary at: selectorParts first value ifAbsentPut: [aNode selector])
				= aNode selector and: 
						[(aDictionary at: arguments first ifAbsentPut: [aNode arguments])
							= aNode arguments and: [body match: aNode body inContext: aDictionary]]].
	^(self matchArgumentsAgainst: aNode inContext: aDictionary)
		and: [body match: aNode body inContext: aDictionary]!

matchArgumentsAgainst: aNode inContext: aDictionary 
	self arguments size == aNode arguments size ifFalse: [^false].
	(self matchSelectorAgainst: aNode inContext: aDictionary) 
		ifFalse: [^false].
	1 to: arguments size
		do: 
			[:i | 
			((arguments at: i) match: (aNode arguments at: i) inContext: aDictionary) 
				ifFalse: [^false]].
	^true!

matchSelectorAgainst: aNode inContext: aDictionary 
	| keyword |
	1 to: selectorParts size
		do: 
			[:i | 
			keyword := selectorParts at: i.
			(aDictionary at: keyword value
				ifAbsentPut: 
					[keyword isMetaVariable 
						ifTrue: [(aNode selectorParts at: i) value]
						ifFalse: [keyword value]]) 
					= (aNode selectorParts at: i) value ifFalse: [^false]].
	^true! !

!BRMetaMethodNode methodsFor: 'private'!

matchingClass
	^BRMethodNode! !

!BRMetaMethodNode methodsFor: 'testing'!

isSelectorList
	^isList! !

BRMetaMethodNode class
	instanceVariableNames: ''!


Object subclass: #BRParser
	instanceVariableNames: 'scanner currentToken nextToken emptyStatements errorBlock tags source '
	classVariableNames: 'ParserType '
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRParser methodsFor: 'accessing'!

errorBlock: aBlock 
	errorBlock := aBlock.
	scanner notNil ifTrue: [scanner errorBlock: aBlock]!

initializeParserWith: aString type: aSymbol 
	source := aString.
	self scanner: (BRScanner perform: aSymbol
				with: (ReadStream on: aString)
				with: self errorBlock)!

parseExpression
	| node |
	node := self parseStatements: false.
	self atEnd ifFalse: [self parserError: 'Unknown input at end'].
	^node!

parseMethod: aString 
	| node |
	node := self parseMethod.
	self atEnd ifFalse: [self parserError: 'Unknown input at end'].
	node source: aString.
	^node! !

!BRParser methodsFor: 'error handling'!

errorBlock
	^errorBlock isNil
		ifTrue: [[:message :position | ]]
		ifFalse: [errorBlock]!

errorPosition
	^currentToken start!

parserError: aString 
	"Evaluate the block. If it returns raise an error"

	self errorBlock value: aString value: self errorPosition.
	self error: aString! !

!BRParser methodsFor: 'initialize-release'!

initializeForIBM
	emptyStatements := true.
	scanner notNil ifTrue: [scanner initializeForIBM]!

initializeForVisualWorks
	emptyStatements := false.
	scanner notNil ifTrue: [scanner initializeForVisualWorks]!

scanner: aScanner 
	scanner := aScanner.
	tags := nil.
	self class isIBM 
		ifTrue: [self initializeForIBM]
		ifFalse: [self initializeForVisualWorks].
	self step! !

!BRParser methodsFor: 'private'!

addCommentsTo: aNode
	aNode comments: scanner getComments!

nextToken
	^nextToken isNil
		ifTrue: [nextToken := scanner next]
		ifFalse: [nextToken]!

patchNegativeLiteral
	"Handle the special negative number case for binary message sends."

	(currentToken value respondsTo: #negated) ifFalse: [^self].
	currentToken value <= 0 ifFalse: [^self].
	currentToken value = 0 
		ifTrue: 
			[(source notNil and: 
					[source isEmpty not 
						and: [(source at: (currentToken start min: source size)) == $-]]) 
				ifFalse: [^self]].
	nextToken := currentToken.
	currentToken := BRBinarySelectorToken value: #- start: nextToken start.
	nextToken value: nextToken value negated.
	nextToken start: nextToken start + 1!

step
	nextToken notNil ifTrue: 
			[currentToken := nextToken.
			nextToken := nil.
			^currentToken].
	currentToken := scanner next! !

!BRParser methodsFor: 'private-parsing'!

parseArgs
	| args |
	args := OrderedCollection new.
	[currentToken isIdentifier]
		whileTrue: [args add: self parseVariableNode].
	^args!

parseAssignment
	"Need one token lookahead to see if we have a ':='. This method could 
	make it possible to assign the literals true, false and nil."

	| node position |
	(currentToken isIdentifier and: [self nextToken isAssignment])
		ifFalse: [^self parseCascadeMessage].
	node := self parseVariableNode.
	position := currentToken start.
	self step.
	^BRAssignmentNode variable: node
		value: self parseAssignment
		position: position!

parseBinaryMessage
	| node |
	node := self parseUnaryMessage.
	
	[currentToken isLiteral ifTrue: [self patchNegativeLiteral].
	currentToken isBinary]
			whileTrue: [node := self parseBinaryMessageWith: node].
	^node!

parseBinaryMessageWith: aNode 
	| binaryToken |
	binaryToken := currentToken.
	self step.
	^BRMessageNode receiver: aNode
		selectorParts: (Array with: binaryToken)
		arguments: (Array with: self parseUnaryMessage)!

parseBinaryPattern
	| binaryToken |
	currentToken isBinary
		ifFalse: [self parserError: 'Message pattern expected'].
	binaryToken := currentToken.
	self step.
	^BRMethodNode selectorParts: (Array with: binaryToken)
		arguments: (Array with: self parseVariableNode)!

parseBlock
	| position node |
	position := currentToken start.
	self step.
	node := self parseBlockArgs.
	node left: position.
	node body: (self parseStatements: false).
	(currentToken isSpecial and: [currentToken value == $]])
		ifFalse: [self parserError: ''']'' expected'].
	node right: currentToken start.
	self step.
	^node!

parseBlockArgs
	| verticalBar args colons node |
	node := BRBlockNode new.
	args := OrderedCollection new: 2.
	colons := OrderedCollection new: 2.
	verticalBar := false.
	[currentToken isSpecial and: [currentToken value == $:]] whileTrue: 
			[colons add: currentToken start.
			self step.	":"
			verticalBar := true.
			args add: self parseVariableNode].
	verticalBar ifTrue: 
			[currentToken isBinary
				ifTrue: 
					[node bar: currentToken start.
					currentToken value == #|
						ifTrue: [self step]
						ifFalse: 
							[currentToken value == #'||'
								ifTrue: 
									["Hack the current token to be the start 
									of temps bar"

									currentToken value: #|;
										start: currentToken start + 1]
								ifFalse: [self parserError: '''|'' expected']]]
				ifFalse: 
					[(currentToken isSpecial and: [currentToken value == $]])
						ifFalse: [self parserError: '''|'' expected']]].
	node arguments: args;
		colons: colons.
	^node!

parseCascadeMessage
	| node receiver messages semicolons |
	node := self parseKeywordMessage.
	(currentToken isSpecial 
		and: [currentToken value == $; and: [node isMessage]]) ifFalse: [^node].
	receiver := node receiver.
	messages := OrderedCollection new: 3.
	semicolons := OrderedCollection new: 3.
	messages add: node.
	[currentToken isSpecial and: [currentToken value == $;]] whileTrue: 
			[semicolons add: currentToken start.
			self step.
			messages add: (currentToken isIdentifier 
						ifTrue: [self parseUnaryMessageWith: receiver]
						ifFalse: 
							[currentToken isKeyword 
								ifTrue: [self parseKeywordMessageWith: receiver]
								ifFalse: 
									[| temp |
									currentToken isLiteral ifTrue: [self patchNegativeLiteral].
									currentToken isBinary ifFalse: [self parserError: 'Message expected'].
									temp := self parseBinaryMessageWith: receiver.
									temp == receiver ifTrue: [self parserError: 'Message expected'].
									temp]])].
	^BRCascadeNode messages: messages semicolons: semicolons!

parseKeywordMessage
	^self parseKeywordMessageWith: self parseBinaryMessage!

parseKeywordMessageWith: node 
	| args isKeyword keywords |
	args := OrderedCollection new: 3.
	keywords := OrderedCollection new: 3.
	isKeyword := false.
	[currentToken isKeyword] whileTrue: 
			[keywords add: currentToken.
			self step.
			args add: self parseBinaryMessage.
			isKeyword := true].
	^isKeyword
		ifTrue: 
			[BRMessageNode receiver: node
				selectorParts: keywords
				arguments: args]
		ifFalse: [node]!

parseKeywordPattern
	| keywords args |
	keywords := OrderedCollection new: 2.
	args := OrderedCollection new: 2.
	[currentToken isKeyword] whileTrue: 
			[keywords add: currentToken.
			self step.
			args add: self parseVariableNode].
	^BRMethodNode selectorParts: keywords arguments: args!

parseMessagePattern
	^currentToken isIdentifier
		ifTrue: [self parseUnaryPattern]
		ifFalse: 
			[currentToken isKeyword
				ifTrue: [self parseKeywordPattern]
				ifFalse: [self parseBinaryPattern]]!

parseMethod
	| methodNode |
	methodNode := self parseMessagePattern.
	self parseResourceTag.
	self addCommentsTo: methodNode.
	methodNode body: (self parseStatements: true).
	methodNode tags: tags.
	^methodNode!

parseOptimizedExpression
	| position node |
	position := currentToken start.
	self step.
	node := BROptimizedNode left: position
				body: (self parseStatements: false)
				right: currentToken start.
	(currentToken isSpecial and: [currentToken value == $)])
		ifFalse: [self parserError: ''')'' expected'].
	self step.
	^node!

parseParenthesizedExpression
	| leftParen node |
	leftParen := currentToken start.
	self step.
	node := self parseAssignment.
	^(currentToken isSpecial and: [currentToken value == $)])
		ifTrue: 
			[node addParenthesis: (leftParen to: currentToken start).
			self step.
			node]
		ifFalse: [self parserError: ''')'' expected']!

parsePrimitiveIdentifier
	| value token |
	token := currentToken.
	value := currentToken value.
	self step.
	value = 'true' ifTrue: 
			[^BRLiteralNode literalToken: (BRLiteralToken value: true
						start: token start
						stop: token start + 3)].
	value = 'false' ifTrue: 
			[^BRLiteralNode literalToken: (BRLiteralToken value: false
						start: token start
						stop: token start + 4)].
	value = 'nil' ifTrue: 
			[^BRLiteralNode literalToken: (BRLiteralToken value: nil
						start: token start
						stop: token start + 2)].
	^BRVariableNode identifierToken: token!

parsePrimitiveLiteral
	| token |
	token := currentToken.
	self step.
	^BRLiteralNode literalToken: token!

parsePrimitiveObject
	currentToken isIdentifier ifTrue: [^self parsePrimitiveIdentifier].
	currentToken isLiteral ifTrue: [^self parsePrimitiveLiteral].
	currentToken isSpecial ifTrue: 
			[currentToken value == $[ ifTrue: [^self parseBlock].
			currentToken value == $( ifTrue: [^self parseParenthesizedExpression]].
	currentToken isOptimized ifTrue: [^self parseOptimizedExpression].
	self parserError: 'Variable expected'!

parseResourceTag
	| start |
	[currentToken isBinary and: [currentToken value == #<]] whileTrue: 
			[start := currentToken start.
			self step.
			[scanner atEnd or: [currentToken isBinary and: [currentToken value == #>]]] 
				whileFalse: [self step].
			(currentToken isBinary and: [currentToken value == #>]) 
				ifFalse: [self parserError: '''>'' expected'].
			tags isNil 
				ifTrue: [tags := OrderedCollection with: (start to: currentToken stop)]
				ifFalse: [tags add: (start to: currentToken stop)].
			self step]!

parseStatementList: tagBoolean into: sequenceNode 
	| statements return periods returnPosition node |
	return := false.
	statements := OrderedCollection new.
	periods := OrderedCollection new.
	self addCommentsTo: sequenceNode.
	tagBoolean ifTrue: [self parseResourceTag].
	
	[self atEnd 
		or: [currentToken isSpecial and: ['])' includes: currentToken value]]] 
			whileFalse: 
				[return ifTrue: [self parserError: 'End of statement list encounted'].
				(currentToken isSpecial and: [currentToken value == $^]) 
					ifTrue: 
						[returnPosition := currentToken start.
						self step.
						node := BRReturnNode return: returnPosition value: self parseAssignment.
						self addCommentsTo: node.
						statements add: node.
						return := true]
					ifFalse: 
						[node := self parseAssignment.
						self addCommentsTo: node.
						statements add: node].
				(currentToken isSpecial and: [currentToken value == $.]) 
					ifTrue: 
						[periods add: currentToken start.
						self step]
					ifFalse: [return := true].
				emptyStatements 
					ifTrue: 
						[[currentToken isSpecial and: [currentToken value == $.]] whileTrue: 
								[periods add: currentToken start.
								self step]]].
	sequenceNode
		statements: statements;
		periods: periods.
	^sequenceNode!

parseStatements: tagBoolean 
	| args leftBar rightBar |
	args := #().
	leftBar := rightBar := nil.
	currentToken isBinary ifTrue: [currentToken value == #|
			ifTrue: 
				[leftBar := currentToken start.
				self step.
				args := self parseArgs.
				(currentToken isBinary and: [currentToken value = #|])
					ifFalse: [self parserError: '''|'' expected'].
				rightBar := currentToken start.
				self step]
			ifFalse: [currentToken value == #'||' 
					ifTrue: 
						[rightBar := (leftBar := currentToken start) + 1.
						self step]]].
	^self parseStatementList: tagBoolean into: (BRSequenceNode
			leftBar: leftBar
			temporaries: args
			rightBar: rightBar)!

parseUnaryMessage
	| node |
	node := self parsePrimitiveObject.
	[currentToken isIdentifier]
		whileTrue: [node := self parseUnaryMessageWith: node].
	^node!

parseUnaryMessageWith: aNode 
	| selector |
	selector := currentToken.
	self step.
	^BRMessageNode receiver: aNode
		selectorParts: (Array with: selector)
		arguments: #()!

parseUnaryPattern
	| selector |
	selector := currentToken.
	self step.
	^BRMethodNode selectorParts: (Array with: selector) arguments: #()!

parseVariableNode
	| node |
	currentToken isIdentifier
		ifFalse: [self parserError: 'Variable name expected'].
	node := BRVariableNode identifierToken: currentToken.
	self step.
	^node! !

!BRParser methodsFor: 'testing'!

atEnd
	^currentToken class == BRToken! !

BRParser class
	instanceVariableNames: ''!



!BRParser class methodsFor: 'accessing'!

parseExpression: aString 
	^self parseExpression: aString onError: nil!

parseExpression: aString onError: aBlock 
	| node parser |
	parser := self new.
	parser errorBlock: aBlock.
	parser initializeParserWith: aString type: #on:errorBlock:.
	node := parser parseExpression.
	^(node statements size == 1 and: [node temporaries isEmpty])
		ifTrue: [node statements first]
		ifFalse: [node]!

parseMethod: aString 
	^self parseMethod: aString onError: nil!

parseMethod: aString onError: aBlock 
	| parser |
	parser := self new.
	parser errorBlock: aBlock.
	parser initializeParserWith: aString type: #on:errorBlock:.
	^parser parseMethod: aString!

parseRewriteExpression: aString 
	^self parseRewriteExpression: aString onError: nil!

parseRewriteExpression: aString onError: aBlock 
	| node parser |
	parser := self new.
	parser errorBlock: aBlock.
	parser initializeParserWith: aString type: #rewriteOn:errorBlock:.
	node := parser parseExpression.
	^(node statements size == 1 and: [node temporaries isEmpty])
		ifTrue: [node statements first]
		ifFalse: [node]!

parseRewriteMethod: aString 
	^self parseRewriteMethod: aString onError: nil!

parseRewriteMethod: aString onError: aBlock 
	| parser |
	parser := self new.
	parser errorBlock: aBlock.
	parser initializeParserWith: aString type: #rewriteOn:errorBlock:.
	^parser parseMethod: aString! !

!BRParser class methodsFor: 'class initialization'!

initialize
	"Try to determine which image we're running in"

	ParserType := (Smalltalk includesKey: #VisualComponent)
				ifTrue: [#VisualWorks]
				ifFalse: [#IBM]! !

!BRParser class methodsFor: 'parsing'!

parseMethodPattern: aString 
	| parser |
	parser := self new.
	parser errorBlock: [:error :position | ^nil].
	parser initializeParserWith: aString type: #on:errorBlock:.
	^parser parseMessagePattern selector! !

!BRParser class methodsFor: 'testing'!

isIBM
	^ParserType == #IBM!

isVisualWorks
	^ParserType == #VisualWorks! !

BRValueNode subclass: #BRCascadeNode
	instanceVariableNames: 'messages semicolons '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRCascadeNode comment:
'BRCascadeNode is an AST node for cascaded messages (e.g., "self print1 ; print2").

Instance Variables:
	messages	<SequenceableCollection of: BRMessageNode>	the messages 
	semicolons	<SequenceableCollection of: Integer>	positions of the ; between messages

'!


!BRCascadeNode methodsFor: 'accessing'!

children
	^self messages!

messages
	^messages!

messages: messageNodeCollection 
	messages := messageNodeCollection.
	messages do: [:each | each parent: self]!

precedence
	^4!

startWithoutParentheses
	^messages first start!

stopWithoutParentheses
	^messages last stop! !

!BRCascadeNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	self messages size = anObject messages size ifFalse: [^false].
	1 to: self messages size
		do: [:i | (self messages at: i) = (anObject messages at: i) ifFalse: [^false]].
	^true!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	self messages size == anObject messages size ifFalse: [^false].
	1 to: self messages size
		do: 
			[:i | 
			((self messages at: i) equalTo: (anObject messages at: i)
				withMapping: aDictionary) ifFalse: [^false]].
	^true!

hash
	^self messages hash! !

!BRCascadeNode methodsFor: 'copying'!

postCopy
	super postCopy.
	messages := messages collect: [:each | each copy]! !

!BRCascadeNode methodsFor: 'initialize-release'!

messages: messageNodes semicolons: integerCollection 
	self messages: messageNodes.
	semicolons := integerCollection! !

!BRCascadeNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new) messages: (self copyList: messages inContext: aDictionary);
		yourself!

match: aNode inContext: aDictionary 
	aNode class == self class ifFalse: [^false].
	^self matchList: messages
		against: aNode messages
		inContext: aDictionary! !

!BRCascadeNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	self messages: (messages 
				collect: [:each | each == aNode ifTrue: [anotherNode] ifFalse: [each]])! !

!BRCascadeNode methodsFor: 'testing'!

directlyUses: aNode 
	^messages last = aNode and: [self isDirectlyUsed]!

isCascade
	^true!

uses: aNode 
	^messages last = aNode and: [self isUsed]! !

!BRCascadeNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptCascadeNode: self! !

!BRCascadeNode methodsFor: 'querying'!

bestNodeFor: anInterval 
	| selectedChildren |
	(self intersectsInterval: anInterval) ifFalse: [^nil].
	(self containedBy: anInterval) ifTrue: [^self].
	messages 
		reverseDo: [:each | (each containedBy: anInterval) ifTrue: [^each]].
	selectedChildren := (messages 
				collect: [:each | each bestNodeFor: anInterval]) 
					reject: [:each | each isNil].
	^selectedChildren detect: [:each | true] ifNone: [nil]!

whichNodeIsContainedBy: anInterval 
	| selectedChildren |
	(self intersectsInterval: anInterval) ifFalse: [^nil].
	(self containedBy: anInterval) ifTrue: [^self].
	messages 
		reverseDo: [:each | (each containedBy: anInterval) ifTrue: [^each]].
	selectedChildren := (messages 
				collect: [:each | each whichNodeIsContainedBy: anInterval]) 
					reject: [:each | each isNil].
	^selectedChildren detect: [:each | true] ifNone: [nil]! !

BRCascadeNode class
	instanceVariableNames: ''!



!BRCascadeNode class methodsFor: 'instance creation'!

messages: messageNodes 
	^self new messages: messageNodes!

messages: messageNodes semicolons: integerCollection 
	^self new messages: messageNodes semicolons: integerCollection! !

BRValueNode subclass: #BRAssignmentNode
	instanceVariableNames: 'variable assignment value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRAssignmentNode comment:
'BRAssignmentNode is an AST node for assignment statements

Instance Variables:
	assignment	<Integer>	position of the :=
	value	<BRValueNode>	the value that we''re assigning
	variable	<BRVariableNode>	the variable being assigned

'!


!BRAssignmentNode methodsFor: 'accessing'!

children
	^Array with: value with: variable!

precedence
	^5!

startWithoutParentheses
	^variable start!

stopWithoutParentheses
	^value stop!

value
	^value!

value: aValueNode 
	value := aValueNode.
	value parent: self!

variable
	^variable!

variable: varNode 
	variable := varNode.
	variable parent: self! !

!BRAssignmentNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	^self variable = anObject variable and: [self value = anObject value]!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	^(self variable equalTo: anObject variable withMapping: aDictionary)
		and: [self value equalTo: anObject value withMapping: aDictionary]!

hash
	^self variable hash bitXor: self value hash! !

!BRAssignmentNode methodsFor: 'copying'!

postCopy
	super postCopy.
	variable := variable postCopy.
	value := value postCopy! !

!BRAssignmentNode methodsFor: 'initialize-release'!

variable: aVariableNode value: aValueNode position: anInteger 
	self variable: aVariableNode.
	self value: aValueNode.
	assignment := anInteger! !

!BRAssignmentNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new) variable: (variable copyInContext: aDictionary);
		value: (value copyInContext: aDictionary);
		yourself!

match: aNode inContext: aDictionary 
	aNode class == self class ifFalse: [^false].
	^(variable match: aNode variable inContext: aDictionary)
		and: [value match: aNode value inContext: aDictionary]! !

!BRAssignmentNode methodsFor: 'testing'!

assigns: aVariableName 
	^variable name = aVariableName or: [value assigns: aVariableName]!

directlyUses: aNode 
	^aNode = value ifTrue: [true] ifFalse: [self isDirectlyUsed]!

isAssignment
	^true!

uses: aNode 
	^aNode = value ifTrue: [true] ifFalse: [self isUsed]! !

!BRAssignmentNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	value == aNode ifTrue: [self value: anotherNode].
	variable == aNode ifTrue: [self variable: anotherNode]! !

!BRAssignmentNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptAssignmentNode: self! !

!BRAssignmentNode methodsFor: 'querying'!

bestNodeFor: anInterval 
	(self intersectsInterval: anInterval) ifFalse: [^nil].
	(self containedBy: anInterval) ifTrue: [^self].
	assignment isNil ifTrue: [^super bestNodeFor: anInterval].
	((anInterval first between: assignment and: assignment + 1) 
		or: [assignment between: anInterval first and: anInterval last]) 
			ifTrue: [^self].
	self children do: 
			[:each | 
			| node |
			node := each bestNodeFor: anInterval.
			node notNil ifTrue: [^node]]! !

BRAssignmentNode class
	instanceVariableNames: ''!



!BRAssignmentNode class methodsFor: 'instance creation'!

variable: aVariableNode value: aValueNode 
	^self 
		variable: aVariableNode
		value: aValueNode
		position: nil!

variable: aVariableNode value: aValueNode position: anInteger 
	^self new
		variable: aVariableNode
		value: aValueNode
		position: anInteger! !

BRValueNode subclass: #BRVariableNode
	instanceVariableNames: 'token '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRVariableNode comment:
'BRVariableNode is an AST node that represent a variable (global, inst var, temp, etc.).

Instance Variables:
	token	<BRValueToken>	the token that contains our name and position

'!


!BRVariableNode methodsFor: 'accessing'!

name
	^token value!

precedence
	^0!

startWithoutParentheses
	^token start!

stopWithoutParentheses
	^token stop! !

!BRVariableNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	^self name = anObject name!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	^(aDictionary at: self name ifAbsentPut: [anObject name]) = anObject name!

hash
	^self name hash! !

!BRVariableNode methodsFor: 'initialize-release'!

identifierToken: anIdentifierToken 
	token := anIdentifierToken! !

!BRVariableNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^self class identifierToken: token removePositions! !

!BRVariableNode methodsFor: 'testing'!

isImmediate
	^true!

isVariable
	^true!

references: aVariableName 
	^self name = aVariableName! !

!BRVariableNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptVariableNode: self! !

BRVariableNode class
	instanceVariableNames: ''!



!BRVariableNode class methodsFor: 'instance creation'!

identifierToken: anIdentifierToken 
	^(anIdentifierToken isMetaVariable
		ifTrue: [BRMetaVariableNode]
		ifFalse: [BRVariableNode]) new identifierToken: anIdentifierToken!

named: aString 
	^self identifierToken: (BRIdentifierToken value: aString start: 0)! !

BRStatementNode subclass: #BRReturnNode
	instanceVariableNames: 'return value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRReturnNode comment:
'BRReturnNode is an AST node that represents a return expression.

Instance Variables:
	return	<Integer>	the position of the ^ character
	value	<BRValueNode>	the value that is being returned

'!


!BRReturnNode methodsFor: 'accessing'!

children
	^Array with: value!

start
	^return!

stop
	^value stop!

value
	^value!

value: valueNode 
	value := valueNode.
	value parent: self! !

!BRReturnNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class = anObject class ifFalse: [^false].
	^self value = anObject value!

equalTo: anObject withMapping: aDictionary 
	self class = anObject class ifFalse: [^false].
	^self value equalTo: anObject value withMapping: aDictionary!

hash
	^self value hash! !

!BRReturnNode methodsFor: 'copying'!

postCopy
	super postCopy.
	value := value copy! !

!BRReturnNode methodsFor: 'initialize-release'!

return: returnInteger value: aValueNode 
	return := returnInteger.
	self value: aValueNode! !

!BRReturnNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^(self class new) value: (value copyInContext: aDictionary); yourself!

match: aNode inContext: aDictionary 
	aNode class == self class ifFalse: [^false].
	^value match: aNode value inContext: aDictionary! !

!BRReturnNode methodsFor: 'testing'!

containsReturn
	^true!

isReturn
	^true! !

!BRReturnNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptReturnNode: self! !

!BRReturnNode methodsFor: 'replacing'!

replaceNode: aNode withNode: anotherNode 
	value == aNode ifTrue: [self value: anotherNode]! !

BRReturnNode class
	instanceVariableNames: ''!



!BRReturnNode class methodsFor: 'instance creation'!

return: returnInteger value: aValueNode 
	^self new return: returnInteger value: aValueNode!

value: aNode
	^self return: nil value: aNode! !

BRVariableNode subclass: #BRMetaVariableNode
	instanceVariableNames: 'recurseInto isList isLiteral isStatement isAnything '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRMetaVariableNode comment:
'BRMetaVariableNode is an AST node that is used to match several other types of nodes (literals, variables, value nodes, statement nodes, and sequences of statement nodes).

The different types of matches are determined by the name of the node. If the name contains a # character, then it will match a literal. If it contains, a . then it matches statements. If it contains no extra characters, then it matches only variables. These options are mutually exclusive.

The @ character can be combined with the name to match lists of items. If combined with the . character, then it will match a list of statement nodes (0 or more). If used without the . or # character, then it matches anything except for list of statements. Combining the @ with the # is not supported.

Adding another ` in the name will cause the search/replace to look for more matches inside the node that this node matched. This option should not be used for top level expressions since that would cause infinite recursion (e.g., searching only for "``@anything").

Instance Variables:
	isList	<Boolean>	can we match a list of items (@)
	isLiteral	<Boolean>	only match a literal node (#)
	isStatement	<Boolean>	only match statements (.)
	recurseInto	<Boolean>	search for more matches in the node we match (`)

'!


!BRMetaVariableNode methodsFor: 'initialize-release'!

identifierToken: anIdentifierToken 
	super identifierToken: anIdentifierToken.
	self initializeMetaVariables!

initializeMetaVariables
	| name |
	name := self name.
	isAnything := isList := isLiteral := isStatement := recurseInto := false.
	2 to: name size
		do: 
			[:i | 
			| character |
			character := name at: i.
			character == self listCharacter 
				ifTrue: [isAnything := isList := true]
				ifFalse: 
					[character == self literalCharacter 
						ifTrue: [isLiteral := true]
						ifFalse: 
							[character == self statementCharacter 
								ifTrue: [isStatement := true]
								ifFalse: 
									[character == self recurseIntoCharacter 
										ifTrue: [recurseInto := true]
										ifFalse: [^self]]]]]! !

!BRMetaVariableNode methodsFor: 'accessing'!

parent: aBRProgramNode 
	"Fix the case where '``@node' should match a single node, not a sequence node."

	super parent: aBRProgramNode.
	parent isSequence 
		ifTrue: 
			[(self isStatement or: [parent temporaries includes: self]) 
				ifFalse: [isList := false]]! !

!BRMetaVariableNode methodsFor: 'matching'!

copyInContext: aDictionary 
	^aDictionary at: self!

match: aNode inContext: aDictionary 
	self isAnything ifTrue: [^(aDictionary at: self ifAbsentPut: [aNode]) = aNode].
	self isLiteral ifTrue: [^self matchLiteral: aNode inContext: aDictionary].
	self isStatement
		ifTrue: [^self matchStatement: aNode inContext: aDictionary].
	aNode class == self matchingClass ifFalse: [^false].
	^(aDictionary at: self ifAbsentPut: [aNode]) = aNode!

matchLiteral: aNode inContext: aDictionary 
	^aNode class == BRLiteralNode 
		and: [(aDictionary at: self ifAbsentPut: [aNode]) = aNode]!

matchStatement: aNode inContext: aDictionary 
	(aNode parent notNil and: [aNode parent isSequence]) ifFalse: [^false].
	^(aDictionary at: self ifAbsentPut: [aNode]) = aNode! !

!BRMetaVariableNode methodsFor: 'private'!

matchingClass
	^BRVariableNode! !

!BRMetaVariableNode methodsFor: 'testing-matching'!

isAnything
	^isAnything!

isList
	^isList!

isLiteral
	^isLiteral!

isStatement
	^isStatement!

recurseInto
	^recurseInto! !

BRMetaVariableNode class
	instanceVariableNames: ''!


Object subclass: #BRProgramNodeVisitor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRProgramNodeVisitor comment:
'BRProgramNodeVisitor is an abstract visitor for the BRProgramNodes.

'!


!BRProgramNodeVisitor methodsFor: 'copying'!

copy
	"Here since IBM doesn't do postCopy's"

	^self shallowCopy postCopy!

postCopy! !

!BRProgramNodeVisitor methodsFor: 'initialize-release'!

initialize! !

!BRProgramNodeVisitor methodsFor: 'visiting'!

visitArgument: each 
	"Here to allow subclasses to detect arguments or temporaries."

	^self visitNode: each!

visitArguments: aNodeCollection 
	^aNodeCollection do: [:each | self visitArgument: each]!

visitNode: aNode 
	^aNode acceptVisitor: self! !

!BRProgramNodeVisitor methodsFor: 'visitor-double dispatching'!

acceptAssignmentNode: anAssignmentNode 
	self visitNode: anAssignmentNode variable.
	self visitNode: anAssignmentNode value!

acceptBlockNode: aBlockNode 
	self visitArguments: aBlockNode arguments.
	self visitNode: aBlockNode body!

acceptCascadeNode: aCascadeNode 
	aCascadeNode messages do: [:each | self visitNode: each]!

acceptLiteralNode: aLiteralNode!

acceptMessageNode: aMessageNode 
	self visitNode: aMessageNode receiver.
	aMessageNode arguments do: [:each | self visitNode: each]!

acceptMethodNode: aMethodNode 
	self visitArguments: aMethodNode arguments.
	self visitNode: aMethodNode body!

acceptOptimizedNode: anOptimizedNode 
	self visitNode: anOptimizedNode body!

acceptReturnNode: aReturnNode 
	self visitNode: aReturnNode value!

acceptSequenceNode: aSequenceNode 
	self visitArguments: aSequenceNode temporaries.
	aSequenceNode statements do: [:each | self visitNode: each]!

acceptVariableNode: aVariableNode! !

BRProgramNodeVisitor class
	instanceVariableNames: ''!



!BRProgramNodeVisitor class methodsFor: 'instance creation'!

new
	^super new initialize! !

BRProgramNodeVisitor subclass: #ParseTreeSearcher
	instanceVariableNames: 'searches answer argumentSearches context '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

ParseTreeSearcher comment:
'ParseTreeSearcher walks over a normal source code parse tree using the visitor pattern, and then matches these nodes against the meta-nodes using the match:inContext: methods defined for the meta-nodes.

Instance Variables:
	answer	<Object>	the "answer" that is propagated between matches
	argumentSearches	<Collection of: (Association key: BRProgramNode value: BlockClosure)>	argument searches (search for the BRProgramNode and perform the BlockClosure when its found)
	context	<BRSmallDictionary>	a dictionary that contains what each meta-node matches against. This could be a normal Dictionary that is created for each search, but is created once and reused (efficiency).
	searches	<Collection of: (Association key: BRProgramNode value: BlockClosure)>	non-argument searches (search for the BRProgramNode and perform the BlockClosure when its found)'!


!ParseTreeSearcher methodsFor: 'accessing'!

addArgumentRule: aParseTreeRule 
	argumentSearches add: aParseTreeRule.
	aParseTreeRule owner: self!

addArgumentRules: ruleCollection 
	ruleCollection do: [:each | self addArgumentRule: each]!

addRule: aParseTreeRule 
	searches add: aParseTreeRule.
	aParseTreeRule owner: self!

addRules: ruleCollection 
	ruleCollection do: [:each | self addRule: each]!

answer
	^answer!

context
	^context!

executeMethod: aParseTree initialAnswer: anObject 
	answer := anObject.
	searches detect: [:each | (each performOn: aParseTree) notNil] ifNone: [].
	^answer!

executeTree: aParseTree 
	"Save our current context, in case someone is performing another search inside a match."

	| oldContext |
	oldContext := context.
	context := BRSmallDictionary new.
	self visitNode: aParseTree.
	context := oldContext.
	^answer!

executeTree: aParseTree initialAnswer: aValue 
	answer := aValue.
	^self executeTree: aParseTree! !

!ParseTreeSearcher methodsFor: 'initialize-release'!

answer: anObject
	answer := anObject!

initialize
	super initialize.
	context := BRSmallDictionary new.
	searches := OrderedCollection new.
	argumentSearches := OrderedCollection new: 0.
	answer := nil! !

!ParseTreeSearcher methodsFor: 'searching'!

matches: aString do: aBlock 
	self addRule: (BRSearchRule searchFor: aString thenDo: aBlock)!

matchesAnyArgumentOf: stringCollection do: aBlock 
	stringCollection do: [:each | self matchesArgument: each do: aBlock]!

matchesAnyMethodOf: aStringCollection do: aBlock 
	aStringCollection do: [:each | self matchesMethod: each do: aBlock]!

matchesAnyOf: aStringCollection do: aBlock 
	aStringCollection do: [:each | self matches: each do: aBlock]!

matchesAnyTreeOf: treeCollection do: aBlock 
	treeCollection do: [:each | self matchesTree: each do: aBlock]!

matchesArgument: aString do: aBlock 
	self addArgumentRule: (BRSearchRule searchFor: aString thenDo: aBlock)!

matchesArgumentTree: aBRProgramNode do: aBlock 
	self 
		addArgumentRule: (BRSearchRule searchForTree: aBRProgramNode thenDo: aBlock)!

matchesMethod: aString do: aBlock 
	self addRule: (BRSearchRule searchForMethod: aString thenDo: aBlock)!

matchesTree: aBRProgramNode do: aBlock 
	self addRule: (BRSearchRule searchForTree: aBRProgramNode thenDo: aBlock)! !

!ParseTreeSearcher methodsFor: 'private'!

foundMatch!

lookForMoreMatchesInContext: oldContext 
	oldContext keysAndValuesDo: 
			[:key :value | 
			(key isString not and: [key recurseInto]) 
				ifTrue: [value do: [:each | self visitNode: each]]]!

performSearches: aSearchCollection on: aNode 
	| value |
	1 to: aSearchCollection size
		do: 
			[:i | 
			value := (aSearchCollection at: i) performOn: aNode.
			value notNil 
				ifTrue: 
					[self foundMatch.
					^value]].
	^nil!

recusivelySearchInContext
	"We need to save the matched context since the other searches might overwrite it."

	| oldContext |
	oldContext := context.
	context := BRSmallDictionary new.
	self lookForMoreMatchesInContext: oldContext.
	context := oldContext! !

!ParseTreeSearcher methodsFor: 'visiting'!

visitArgument: aNode 
	| value |
	value := self performSearches: argumentSearches on: aNode.
	^value isNil 
		ifTrue: 
			[aNode acceptVisitor: self.
			aNode]
		ifFalse: [value]!

visitNode: aNode 
	| value |
	value := self performSearches: searches on: aNode.
	^value isNil 
		ifTrue: 
			[aNode acceptVisitor: self.
			aNode]
		ifFalse: [value]! !

!ParseTreeSearcher methodsFor: 'obsolete'!

addArgumentSearch: aSearchCondition 
	self addArgumentRule: (self buildParseTreeRuleFor: aSearchCondition)!

addArgumentSearches: aSearchCondition 
	aSearchCondition key do: [:each | self addArgumentSearch: each -> aSearchCondition value]!

addMethodSearch: aSearchCondition 
	self addRule: (self buildMethodParseTreeRuleFor: aSearchCondition)!

addMethodSearches: aSearchCondition 
	aSearchCondition key do: [:each | self addMethodSearch: each -> aSearchCondition value]!

addSearch: aSearchCondition 
	self addRule: (self buildParseTreeRuleFor: aSearchCondition)!

addSearches: aSearchCondition 
	aSearchCondition key do: [:each | self addSearch: each -> aSearchCondition value]!

buildMethodParseTreeRuleFor: aSearchCondition 
	^(aSearchCondition key isKindOf: BRProgramNode) 
		ifTrue: 
			[BRSearchRule searchForTree: aSearchCondition key
				thenDo: aSearchCondition value]
		ifFalse: 
			[BRSearchRule searchForMethod: aSearchCondition key
				thenDo: aSearchCondition value]!

buildParseTreeRuleFor: aSearchCondition 
	^(aSearchCondition key isKindOf: BRProgramNode) 
		ifTrue: 
			[BRSearchRule searchForTree: aSearchCondition key
				thenDo: aSearchCondition value]
		ifFalse: 
			[BRSearchRule searchFor: aSearchCondition key thenDo: aSearchCondition value]! !

ParseTreeSearcher class
	instanceVariableNames: ''!



!ParseTreeSearcher class methodsFor: 'accessing'!

treeMatching: aString in: aParseTree 
	(self new)
		matches: aString do: [:aNode :answer | ^aNode];
		executeTree: aParseTree.
	^nil!

treeMatchingStatements: aString in: aParseTree 
	| notifier tree lastIsReturn |
	notifier := self new.
	tree := BRParser parseExpression: aString.
	lastIsReturn := tree lastIsReturn.
	notifier matches: (lastIsReturn 
				ifTrue: ['| `@temps | `@.S1. ' , tree formattedCode]
				ifFalse: ['| `@temps | `@.S1. ' , tree formattedCode , '. `@.S2'])
		do: [:aNode :answer | ^tree].
	notifier executeTree: aParseTree.
	^nil! !

!ParseTreeSearcher class methodsFor: 'instance creation'!

getterMethod: aVarName 
	^(self new)
		matchesMethod: '`method ^' , aVarName do: [:aNode :ans | aNode selector];
		yourself!

justSendsSuper
	^(self new)
		matchesAnyMethodOf: 
				#('`@method: `@Args ^super `@method: `@Args' 
				'`@method: `@Args super `@method: `@Args')
			do: [:aNode :ans | true];
		yourself!

returnSetterMethod: aVarName 
	^(self new)
		matchesMethod: '`method: `Arg ^' , aVarName , ' := `Arg'
			do: [:aNode :ans | aNode selector];
		yourself!

setterMethod: aVarName 
	^(self new)
		matchesAnyMethodOf: (Array with: '`method: `Arg ' , aVarName , ' := `Arg'
					with: '`method: `Arg ^' , aVarName , ' := `Arg')
			do: [:aNode :ans | aNode selector];
		yourself! !

!ParseTreeSearcher class methodsFor: 'private'!

buildSelectorString: aSelector 
	| stream keywords |
	aSelector numArgs = 0 ifTrue: [^aSelector].
	stream := WriteStream on: String new.
	keywords := aSelector keywords.
	1 to: keywords size
		do: 
			[:i | 
			stream nextPutAll: (keywords at: i);
				nextPutAll: ' ``@arg';
				nextPutAll: i printString;
				nextPut: $ ].
	^stream contents!

buildSelectorTree: aSelector 
	^BRParser parseRewriteExpression: '``@receiver ' 
				, (self buildSelectorString: aSelector)
		onError: [:err :pos | ^nil]!

buildTree: aString method: aBoolean 
	^aBoolean
		ifTrue: [BRParser parseRewriteMethod: aString]
		ifFalse: [BRParser parseRewriteExpression: aString]! !

ParseTreeSearcher subclass: #ParseTreeRewriter
	instanceVariableNames: 'tree '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

ParseTreeRewriter comment:
'ParseTreeRewriter walks over and transforms its BRProgramNode (tree). If the tree is modified, then answer is set to true, and the modified tree can be retrieved by the #tree method.

Instance Variables:
	tree	<BRProgramNode>	the parse tree we''re transforming'!


!ParseTreeRewriter methodsFor: 'accessing'!

executeTree: aParseTree 
	| oldContext |
	oldContext := context.
	context := BRSmallDictionary new.
	answer := false.
	tree := self visitNode: aParseTree.
	context := oldContext.
	^answer!

tree
	^tree! !

!ParseTreeRewriter methodsFor: 'replacing'!

replace: searchString with: replaceString 
	self addRule: (BRStringReplaceRule searchFor: searchString
				replaceWith: replaceString)!

replace: searchString with: replaceString when: aBlock 
	self addRule: (BRStringReplaceRule 
				searchFor: searchString
				replaceWith: replaceString
				when: aBlock)!

replace: searchString withValueFrom: replaceBlock 
	self addRule: (BRBlockReplaceRule searchFor: searchString
				replaceWith: replaceBlock)!

replace: searchString withValueFrom: replaceBlock when: conditionBlock 
	self addRule: (BRBlockReplaceRule 
				searchFor: searchString
				replaceWith: replaceBlock
				when: conditionBlock)!

replaceArgument: searchString with: replaceString 
	self addArgumentRule: (BRStringReplaceRule searchFor: searchString
				replaceWith: replaceString)!

replaceArgument: searchString with: replaceString when: aBlock 
	self addArgumentRule: (BRStringReplaceRule 
				searchFor: searchString
				replaceWith: replaceString
				when: aBlock)!

replaceArgument: searchString withValueFrom: replaceBlock 
	self addArgumentRule: (BRBlockReplaceRule searchFor: searchString
				replaceWith: replaceBlock)!

replaceArgument: searchString withValueFrom: replaceBlock when: conditionBlock 
	self addArgumentRule: (BRBlockReplaceRule 
				searchFor: searchString
				replaceWith: replaceBlock
				when: conditionBlock)!

replaceMethod: searchString with: replaceString 
	self addRule: (BRStringReplaceRule searchForMethod: searchString
				replaceWith: replaceString)!

replaceMethod: searchString with: replaceString when: aBlock 
	self addRule: (BRStringReplaceRule 
				searchForMethod: searchString
				replaceWith: replaceString
				when: aBlock)!

replaceMethod: searchString withValueFrom: replaceBlock 
	self addRule: (BRBlockReplaceRule searchForMethod: searchString
				replaceWith: replaceBlock)!

replaceMethod: searchString withValueFrom: replaceBlock when: conditionBlock 
	self addRule: (BRBlockReplaceRule 
				searchForMethod: searchString
				replaceWith: replaceBlock
				when: conditionBlock)!

replaceTree: searchTree withTree: replaceTree 
	self addRule: (BRStringReplaceRule searchForTree: searchTree
				replaceWith: replaceTree)!

replaceTree: searchTree withTree: replaceTree when: aBlock 
	self addRule: (BRStringReplaceRule 
				searchForTree: searchTree
				replaceWith: replaceTree
				when: aBlock)! !

!ParseTreeRewriter methodsFor: 'private'!

foundMatch
	answer := true!

lookForMoreMatchesInContext: oldContext 
	oldContext keysAndValuesDo: 
			[:key :value | 
			(key isString not and: [key recurseInto]) 
				ifTrue: 
					[oldContext at: key put: (value collect: [:each | self visitNode: each])]]! !

!ParseTreeRewriter methodsFor: 'visiting'!

visitArguments: aNodeCollection 
	^aNodeCollection collect: [:each | self visitArgument: each]! !

!ParseTreeRewriter methodsFor: 'visitor-double dispatching'!

acceptAssignmentNode: anAssignmentNode 
	anAssignmentNode variable: (self visitNode: anAssignmentNode variable).
	anAssignmentNode value: (self visitNode: anAssignmentNode value)!

acceptBlockNode: aBlockNode 
	aBlockNode arguments: (self visitArguments: aBlockNode arguments).
	aBlockNode body: (self visitNode: aBlockNode body)!

acceptCascadeNode: aCascadeNode 
	| newMessages notFound |
	newMessages := OrderedCollection new: aCascadeNode messages size.
	notFound := OrderedCollection new: aCascadeNode messages size.
	aCascadeNode messages do: 
			[:each | 
			| newNode |
			newNode := self performSearches: searches on: each.
			newNode isNil 
				ifTrue: 
					[newNode := each.
					notFound add: newNode].
			newNode isMessage 
				ifTrue: [newMessages add: newNode]
				ifFalse: 
					[newNode isCascade 
						ifTrue: [newMessages addAll: newNode messages]
						ifFalse: 
							[Transcript
								show: 'Cannot replace message node inside of cascaded node with non-message node.';
								cr.
							newMessages add: each]]].
	notFound size == aCascadeNode messages size 
		ifTrue: 
			[| receiver |
			receiver := self visitNode: aCascadeNode messages first receiver.
			newMessages do: [:each | each receiver: receiver]].
	notFound 
		do: [:each | each arguments: (each arguments collect: [:arg | self visitNode: arg])].
	aCascadeNode messages: newMessages!

acceptMessageNode: aMessageNode 
	aMessageNode receiver: (self visitNode: aMessageNode receiver).
	aMessageNode 
		arguments: (aMessageNode arguments collect: [:each | self visitNode: each])!

acceptMethodNode: aMethodNode 
	aMethodNode arguments: (self visitArguments: aMethodNode arguments).
	aMethodNode body: (self visitNode: aMethodNode body)!

acceptOptimizedNode: anOptimizedNode 
	anOptimizedNode body: (self visitNode: anOptimizedNode body)!

acceptReturnNode: aReturnNode 
	aReturnNode value: (self visitNode: aReturnNode value)!

acceptSequenceNode: aSequenceNode 
	aSequenceNode temporaries: (self visitArguments: aSequenceNode temporaries).
	aSequenceNode statements: (aSequenceNode statements collect: [:each | self visitNode: each])! !

ParseTreeRewriter class
	instanceVariableNames: ''!



!ParseTreeRewriter class methodsFor: 'accessing'!

replace: code with: newCode in: aParseTree 
	^(self 
		replace: code
		with: newCode
		method: false)
		executeTree: aParseTree;
		tree!

replace: code with: newCode in: aParseTree onInterval: anInterval 
	| rewriteRule |
	rewriteRule := self new.
	^rewriteRule
		replace: code
			with: newCode
			when: [:aNode | aNode intersectsInterval: anInterval];
		executeTree: aParseTree;
		tree!

replaceStatements: code with: newCode in: aParseTree onInterval: anInterval 
	| tree searchStmt replaceStmt |
	tree := self buildTree: code method: false.
	tree lastIsReturn
		ifTrue: 
			[searchStmt := '| `@temps | `@.Statements. ' , code.
			replaceStmt := '| `@temps | `@.Statements. ^' , newCode]
		ifFalse: 
			[searchStmt := '| `@temps | `@.Statements1. ' , code , '.  `@.Statements2'.
			replaceStmt := '| `@temps | `@.Statements1. ' , newCode , '.  `@.Statements2'].
	^self
		replace: searchStmt
		with: replaceStmt
		in: aParseTree
		onInterval: anInterval! !

!ParseTreeRewriter class methodsFor: 'instance creation'!

classVariable: aVarName getter: getMethod setter: setMethod 
	| rewriteRule |
	rewriteRule := self new.
	rewriteRule
		replace: aVarName , ' := ``@object'
			with: 'self class ' , setMethod , ' ``@object';
		replace: aVarName with: 'self class ' , getMethod.
	^rewriteRule!

removeTemporaryNamed: aName 
	| rewriteRule |
	rewriteRule := self new.
	rewriteRule replace: '| `@temps1 ' , aName , ' `@temps2 | ``@.Statements'
		with: '| `@temps1  `@temps2 | ``@.Statements'.
	^rewriteRule!

rename: varName to: newVarName 
	| rewriteRule |
	rewriteRule := self new.
	rewriteRule
		replace: varName with: newVarName;
		replaceArgument: varName with: newVarName.
	^rewriteRule!

rename: varName to: newVarName handler: aBlock 
	"Rename varName to newVarName, evaluating aBlock if there is a 
	temporary variable with the same name as newVarName. This 
	does not change temporary variables with varName."

	| rewriteRule |
	rewriteRule := self new.
	rewriteRule
		replace: varName with: newVarName;
		replaceArgument: newVarName
			withValueFrom: 
				[:aNode | 
				aBlock value.
				aNode].
	^rewriteRule!

replace: code with: newCode method: aBoolean 
	| rewriteRule |
	rewriteRule := self new.
	aBoolean 
		ifTrue: [rewriteRule replaceMethod: code with: newCode]
		ifFalse: [rewriteRule replace: code with: newCode].
	^rewriteRule!

replaceLiteral: literal with: newLiteral 
	| rewriteRule |
	rewriteRule := self new.
	rewriteRule 
		replace: '`#literal'
		withValueFrom: [:aNode | aNode]
		when: 
			[:aNode | 
			self 
				replaceLiteral: literal
				with: newLiteral
				inToken: aNode token].
	^rewriteRule!

variable: aVarName getter: getMethod setter: setMethod 
	| rewriteRule |
	rewriteRule := self new.
	rewriteRule
		replace: aVarName , ' := ``@object'
			with: 'self ' , setMethod , ' ``@object';
		replace: aVarName with: 'self ' , getMethod.
	^rewriteRule! !

!ParseTreeRewriter class methodsFor: 'private'!

replaceLiteral: literal with: newLiteral inToken: literalToken 
	| value |
	value := literalToken realValue.
	value == literal ifTrue: 
			[literalToken value: newLiteral
				start: nil
				stop: nil.
			^true].
	^value class == Array and: 
			[literalToken value inject: false
				into: 
					[:bool :each | 
					bool | (self replaceLiteral: literal
								with: newLiteral
								inToken: each)]]! !

BRMessageNode subclass: #BRMetaMessageNode
	instanceVariableNames: 'isList isCascadeList '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRMetaMessageNode comment:
'BRMetaMessageNode is a BRMessageNode that will match other message nodes without their selectors being equal. 

Instance Variables:
	isList	<Boolean>	are we matching each keyword or matching all keywords together (e.g., `keyword1: would match a one argument method whereas `@keywords: would match 0 or more arguments)'!


!BRMetaMessageNode methodsFor: 'initialize-release'!

receiver: aValueNode selectorParts: keywordTokens arguments: valueNodes 
	| message |
	super 
		receiver: aValueNode
		selectorParts: keywordTokens
		arguments: valueNodes.
	isCascadeList := isList := false.
	message := keywordTokens first value.
	2 to: message size
		do: 
			[:i | 
			| character |
			character := message at: i.
			character == self listCharacter 
				ifTrue: [isList := true]
				ifFalse: 
					[character == self cascadeListCharacter 
						ifTrue: [isCascadeList := true]
						ifFalse: [^self]]]! !

!BRMetaMessageNode methodsFor: 'matching'!

copyInContext: aDictionary 
	| selectors |
	self isList ifTrue: [^aDictionary at: self].
	selectors := self isSelectorList 
				ifTrue: [(aDictionary at: selectorParts first value) keywords]
				ifFalse: [selectorParts collect: [:each | aDictionary at: each value]].
	^(BRMessageNode new)
		receiver: (receiver copyInContext: aDictionary);
		selectorParts: (selectors collect: 
						[:each | 
						(each last == $: ifTrue: [BRKeywordToken] ifFalse: [BRIdentifierToken]) 
							value: each
							start: nil]);
		arguments: (self copyList: arguments inContext: aDictionary);
		yourself!

match: aNode inContext: aDictionary 
	aNode class == self matchingClass ifFalse: [^false].
	(receiver match: aNode receiver inContext: aDictionary) ifFalse: [^false].
	self isSelectorList ifTrue: 
			[^(aDictionary at: selectorParts first value ifAbsentPut: [aNode selector])
				== aNode selector and: 
						[(aDictionary at: arguments first ifAbsentPut: [aNode arguments])
							= aNode arguments]].
	^self matchArgumentsAgainst: aNode inContext: aDictionary!

matchArgumentsAgainst: aNode inContext: aDictionary 
	self arguments size == aNode arguments size ifFalse: [^false].
	(self matchSelectorAgainst: aNode inContext: aDictionary) 
		ifFalse: [^false].
	1 to: arguments size
		do: 
			[:i | 
			((arguments at: i) match: (aNode arguments at: i) inContext: aDictionary) 
				ifFalse: [^false]].
	^true!

matchSelectorAgainst: aNode inContext: aDictionary 
	| keyword |
	1 to: selectorParts size
		do: 
			[:i | 
			keyword := selectorParts at: i.
			(aDictionary at: keyword value
				ifAbsentPut: 
					[keyword isMetaVariable 
						ifTrue: [(aNode selectorParts at: i) value]
						ifFalse: [keyword value]]) 
					= (aNode selectorParts at: i) value ifFalse: [^false]].
	^true! !

!BRMetaMessageNode methodsFor: 'private'!

matchingClass
	^BRMessageNode! !

!BRMetaMessageNode methodsFor: 'testing-matching'!

isList
	^isCascadeList and: [parent notNil and: [parent isCascade]]!

isSelectorList
	^isList! !

BRMetaMessageNode class
	instanceVariableNames: ''!


Object subclass: #BRToken
	instanceVariableNames: 'sourcePointer '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRToken methodsFor: 'accessing'!

length
	^self subclassResponsibility!

removePositions
	sourcePointer := nil!

start
	^sourcePointer!

stop
	^self start + self length - 1! !

!BRToken methodsFor: 'initialize-release'!

start: anInteger 
	sourcePointer := anInteger! !

!BRToken methodsFor: 'printing'!

printOn: aStream 
	aStream
		nextPut: $ ;
		nextPutAll: self class name! !

!BRToken methodsFor: 'testing'!

isAssignment
	^false!

isBinary
	^false!

isIdentifier
	^false!

isKeyword
	^false!

isLiteral
	^false!

isMetaVariable
	^false!

isOptimized
	^false!

isSpecial
	^false! !

BRToken class
	instanceVariableNames: ''!



!BRToken class methodsFor: 'instance creation'!

start: anInterval 
	^self new start: anInterval! !

BRToken subclass: #BRValueToken
	instanceVariableNames: 'value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRValueToken methodsFor: 'accessing'!

value
	^value!

value: anObject
	value := anObject! !

!BRValueToken methodsFor: 'initialize-release'!

value: aString start: anInteger 
	value := aString.
	sourcePointer := anInteger! !

!BRValueToken methodsFor: 'printing'!

printOn: aStream
	super printOn: aStream.
	aStream nextPut: $(.
	value printOn: aStream.
	aStream nextPutAll: ')'! !

!BRValueToken methodsFor: 'private'!

length
	^value size! !

BRValueToken class
	instanceVariableNames: ''!



!BRValueToken class methodsFor: 'instance creation'!

value: aString start: anInteger 
	^self new value: aString start: anInteger! !

BRValueToken subclass: #BRLiteralToken
	instanceVariableNames: 'stopPosition '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRLiteralToken methodsFor: 'accessing'!

realValue
	^value class == Array
		ifTrue: [value collect: [:each | each realValue]]
		ifFalse: [value]!

stop: anObject 
	stopPosition := anObject! !

!BRLiteralToken methodsFor: 'initialize-release'!

value: aString start: anInteger stop: stopInteger 
	value := aString.
	sourcePointer := anInteger.
	stopPosition := stopInteger! !

!BRLiteralToken methodsFor: 'private'!

length
	^stopPosition - self start + 1! !

!BRLiteralToken methodsFor: 'testing'!

isLiteral
	^true! !

BRLiteralToken class
	instanceVariableNames: ''!



!BRLiteralToken class methodsFor: 'instance creation'!

value: anObject 
	| literal |
	literal := anObject class == Array 
				ifTrue: [anObject collect: [:each | self value: each]]
				ifFalse: [anObject].
	^self 
		value: literal
		start: nil
		stop: nil!

value: aString start: anInteger stop: stopInteger 
	^self new
		value: aString
		start: anInteger
		stop: stopInteger! !

BRValueToken subclass: #BRBinarySelectorToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRBinarySelectorToken methodsFor: 'testing'!

isBinary
	^true! !

BRBinarySelectorToken class
	instanceVariableNames: ''!


BRValueToken subclass: #BRSpecialCharacterToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRSpecialCharacterToken methodsFor: 'private'!

length
	^1! !

!BRSpecialCharacterToken methodsFor: 'testing'!

isSpecial
	^true! !

BRSpecialCharacterToken class
	instanceVariableNames: ''!


BRToken subclass: #BROptimizedToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BROptimizedToken methodsFor: 'private'!

length
	^3! !

!BROptimizedToken methodsFor: 'testing'!

isOptimized
	^true! !

BROptimizedToken class
	instanceVariableNames: ''!


BRValueToken subclass: #BRIdentifierToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRIdentifierToken methodsFor: 'testing'!

isIdentifier
	^true!

isMetaVariable
	^value first == BRScanner metaVariableCharacter! !

BRIdentifierToken class
	instanceVariableNames: ''!


BRToken subclass: #BRAssignmentToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRAssignmentToken methodsFor: 'private'!

length
	^2! !

!BRAssignmentToken methodsFor: 'testing'!

isAssignment
	^true! !

BRAssignmentToken class
	instanceVariableNames: ''!


BRValueToken subclass: #BRKeywordToken
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRKeywordToken methodsFor: 'testing'!

isKeyword
	^true!

isMetaVariable
	^value first == BRScanner metaVariableCharacter! !

BRKeywordToken class
	instanceVariableNames: ''!


BRProgramNodeVisitor subclass: #BRFormatter
	instanceVariableNames: 'codeStream lineStart firstLineLength tabs '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRFormatter methodsFor: 'accessing'!

firstLineLength
	^firstLineLength isNil
		ifTrue: [codeStream position]
		ifFalse: [firstLineLength]!

format: aNode 
	self visitNode: aNode.
	^codeStream contents!

isMultiLine
	^firstLineLength notNil!

lastLineLength
	^codeStream position - (lineStart max: 0)! !

!BRFormatter methodsFor: 'copying'!

postCopy
	super postCopy.
	lineStart := self lineLength negated.
	codeStream := WriteStream on: (String new: 60).
	firstLineLength := nil! !

!BRFormatter methodsFor: 'initialize-release'!

initialize
	super initialize.
	codeStream := WriteStream on: (String new: 60).
	tabs := 0.
	lineStart := 0! !

!BRFormatter methodsFor: 'private'!

for: aValue do: doBlock separatedBy: separatorBlock 
	"This is implemented here since IBM Smalltalk doesn't implement a do:separatedBy: method"

	aValue isEmpty ifTrue: [^self].
	1 to: aValue size - 1
		do: 
			[:i | 
			doBlock value: (aValue at: i).
			separatorBlock value].
	doBlock value: aValue last!

indent
	firstLineLength isNil ifTrue: [firstLineLength := codeStream position].
	codeStream cr.
	tabs timesRepeat: [codeStream tab].
	lineStart := codeStream position!

indent: anInteger while: aBlock 
	tabs := tabs + anInteger.
	aBlock value.
	tabs := tabs - anInteger!

indentWhile: aBlock 
	self indent: 1 while: aBlock!

lineLength
	^codeStream position - lineStart!

lineStart: aPosition 
	lineStart := aPosition!

maximumArgumentsPerLine
	^2!

maxLineSize
	^75!

needsParenthesisFor: aNode 
	| parent grandparent |
	aNode isValue ifFalse: [^false].
	parent := aNode parent.
	parent isNil ifTrue: [^false].
	(aNode isMessage and: [parent isMessage and: [parent receiver == aNode]])
		ifTrue: 
			[grandparent := parent parent.
			(grandparent notNil and: [grandparent isCascade]) ifTrue: [^true]].
	aNode precedence < parent precedence ifTrue: [^false].
	aNode isAssignment & parent isAssignment ifTrue: [^false].
	aNode isAssignment | aNode isCascade ifTrue: [^true].
	aNode precedence == 0 ifTrue: [^false].
	aNode isMessage ifFalse: [^true].
	aNode precedence = parent precedence ifFalse: [^true].
	aNode isUnary ifTrue: [^false].
	aNode isKeyword ifTrue: [^true].
	parent receiver == aNode ifFalse: [^true].
	^self precedenceOf: parent selector greaterThan: aNode selector!

precedenceOf: parentSelector greaterThan: childSelector 
	"Put parenthesis around things that are preceived to have 'lower' precedence. For example, 'a + b * c' 
	-> '(a + b) * c' but 'a * b + c' -> 'a * b + c'"

	| childIndex parentIndex operators |
	operators := #(#($| $& $?) #($= $~ $< $>) #($- $+) #($* $/ $% $\) #($@)).
	childIndex := 0.
	parentIndex := 0.
	1 to: operators size
		do: 
			[:i | 
			((operators at: i) includes: parentSelector first) ifTrue: [parentIndex := i].
			((operators at: i) includes: childSelector first) ifTrue: [childIndex := i]].
	^childIndex < parentIndex!

selectorsToLeaveOnLine
	^#(#to:do: #to:by: #to:by:do:)!

selectorsToStartOnNewLine
	^#(#ifTrue:ifFalse: #ifFalse:ifTrue: #ifTrue: #ifFalse:)! !

!BRFormatter methodsFor: 'private-formatting'!

formatLiteral: aValue 
	| isArray |
	(isArray := aValue class == Array) | (aValue class == ByteArray) ifTrue: 
			[codeStream nextPutAll: (isArray ifTrue: ['#('] ifFalse: ['#[']).
			self for: aValue
				do: [:each | self formatLiteral: each]
				separatedBy: [codeStream nextPut: $ ].
			codeStream nextPut: (isArray ifTrue: [$)] ifFalse: [$]]).
			^self].
	aValue isSymbol ifTrue: 
			[self formatSymbol: aValue.
			^self].
	aValue class == Character ifTrue: 
			[codeStream nextPut: $$;
				nextPut: aValue.
			^self].
	aValue storeOn: codeStream!

formatMessage: aMessageNode cascade: cascadeBoolean 
	| selectorParts arguments multiLine formattedArgs indentFirst firstArgLength length |
	selectorParts := aMessageNode selectorParts.
	arguments := aMessageNode arguments.
	formattedArgs := OrderedCollection new.
	multiLine := aMessageNode selector numArgs > self maximumArgumentsPerLine.
	length := aMessageNode selector size + arguments size + 1.
	firstArgLength := 0.
	self indentWhile: 
			[1 to: arguments size
				do: 
					[:i | 
					| formatter string |
					formatter := (self copy) 
								lineStart: (selectorParts at: i) length negated;
								yourself.
					string := formatter format: (arguments at: i).
					formattedArgs add: string.
					i == 1 ifTrue: [firstArgLength := formatter firstLineLength].
					length := length + string size.
					multiLine := multiLine or: [formatter isMultiLine]]].
	multiLine := multiLine or: [length + self lineLength > self maxLineSize].
	indentFirst := cascadeBoolean not and: 
					[multiLine and: 
							[(self startMessageSendOnNewLine: aMessageNode) or: 
									[self lineLength + selectorParts first length + 2 + firstArgLength 
										> self maxLineSize]]].
	indentFirst ifTrue: [self indent].
	self 
		formatMessageSelector: selectorParts
		withArguments: formattedArgs
		multiline: multiLine!

formatMessageSelector: selectorParts withArguments: formattedArgs multiline: multiLine 
	formattedArgs isEmpty 
		ifTrue: [codeStream nextPutAll: selectorParts first value]
		ifFalse: 
			[1 to: formattedArgs size
				do: 
					[:i | 
					i ~~ 1 & multiLine not ifTrue: [codeStream nextPut: $ ].
					codeStream 
						nextPutAll: (selectorParts at: i) value;
						nextPut: $ ;
						nextPutAll: (formattedArgs at: i).
					(multiLine and: [i < formattedArgs size]) ifTrue: [self indent]]]!

formatMethodCommentFor: aNode indentBefore: aBoolean 
	| source |
	source := aNode source.
	source isNil ifTrue: [^self].
	aNode comments do: 
			[:each | 
			aBoolean ifTrue: [self indent].
			codeStream nextPutAll: (aNode source copyFrom: each first to: each last);
				cr.
			aBoolean ifFalse: [self indent]]!

formatMethodPatternFor: aMethodNode 
	| selectorParts arguments |
	selectorParts := aMethodNode selectorParts.
	arguments := aMethodNode arguments.
	arguments isEmpty
		ifTrue: [codeStream nextPutAll: selectorParts first value]
		ifFalse: 
			[selectorParts with: arguments
				do: 
					[:selector :arg | 
					codeStream nextPutAll: selector value;
						nextPut: $ .
					self visitArgument: arg.
					codeStream nextPut: $ ]]!

formatStatementCommentFor: aNode 
	| source |
	source := aNode source.
	source isNil ifTrue: [^self].
	aNode comments do: 
			[:each | 
			| crs |
			crs := self newLinesFor: source startingAt: each first.
			(crs - 1 max: 0) timesRepeat: [codeStream cr].
			crs == 0 ifTrue: [codeStream tab] ifFalse: [self indent].
			codeStream nextPutAll: (source copyFrom: each first to: each last)]!

formatStatementsFor: aSequenceNode 
	| statements |
	statements := aSequenceNode statements.
	statements isEmpty ifTrue: [^self].
	1 to: statements size - 1
		do: 
			[:i | 
			self visitNode: (statements at: i).
			codeStream nextPut: $..
			self formatStatementCommentFor: (statements at: i).
			self indent].
	self visitNode: statements last.
	self formatStatementCommentFor: statements last!

formatSymbol: aSymbol 
	"Format the symbol, if its not a selector then we must put quotes around it. The and: case below, 
	handles the VisualWorks problem of not accepting two bars as a symbol."

	codeStream nextPut: $#.
	((BRScanner isSelector: aSymbol) and: [aSymbol ~~ #'||'])
		ifTrue: [codeStream nextPutAll: aSymbol]
		ifFalse: [aSymbol asString printOn: codeStream]!

formatTagFor: aMethodNode 
	| primitiveSources |
	primitiveSources := aMethodNode primitiveSources.
	primitiveSources do: 
			[:each | 
			codeStream nextPutAll: each.
			self indent]!

formatTemporariesFor: aSequenceNode 
	| temps |
	temps := aSequenceNode temporaries.
	temps isEmpty ifTrue: [^self].
	codeStream nextPutAll: '| '.
	temps do: 
			[:each | 
			self visitArgument: each.
			codeStream nextPut: $ ].
	codeStream nextPut: $|.
	self indent!

newLinesFor: aString startingAt: anIndex 
	| count cr lf index char |
	cr := Character value: 13.
	lf := Character value: 10.
	count := 0.
	index := anIndex - 1.
	[index > 0 and: 
			[char := aString at: index.
			char isSeparator]] 
		whileTrue: 
			[char == lf 
				ifTrue: 
					[count := count + 1.
					(aString at: (index - 1 max: 1)) == cr ifTrue: [index := index - 1]].
			char == cr ifTrue: [count := count + 1].
			index := index - 1].
	^count! !

!BRFormatter methodsFor: 'testing'!

startMessageSendOnNewLine: aMessageNode 
	(self selectorsToStartOnNewLine includes: aMessageNode selector) 
		ifTrue: [^true].
	(self selectorsToLeaveOnLine includes: aMessageNode selector) ifTrue: [^false].
	^aMessageNode selector numArgs > self maximumArgumentsPerLine!

tagBeforeTemporaries
	^BRParser isVisualWorks! !

!BRFormatter methodsFor: 'visiting'!

visitNode: aNode 
	| parenthesis |
	parenthesis := self needsParenthesisFor: aNode.
	parenthesis ifTrue: [codeStream nextPut: $(].
	aNode acceptVisitor: self.
	parenthesis ifTrue: [codeStream nextPut: $)]! !

!BRFormatter methodsFor: 'visitor-double dispatching'!

acceptAssignmentNode: anAssignmentNode 
	self indent: 2
		while: 
			[self visitNode: anAssignmentNode variable.
			codeStream nextPutAll: ' := '.
			self visitNode: anAssignmentNode value]!

acceptBlockNode: aBlockNode 
	| seqNode multiline formattedBody formatter |
	seqNode := aBlockNode body.
	formatter := (self copy) lineStart: 0;
				yourself.
	formattedBody := formatter format: seqNode.
	multiline := self lineLength + formattedBody size > self maxLineSize
				or: [formatter isMultiLine].
	multiline ifTrue: [self indent].
	codeStream nextPut: $[.
	aBlockNode arguments do: 
			[:each | 
			codeStream nextPut: $:.
			self visitNode: each.
			codeStream nextPut: $ ].
	aBlockNode arguments isEmpty ifFalse: 
			[codeStream nextPutAll: '| '.
			multiline ifTrue: [self indent]].
	codeStream nextPutAll: formattedBody;
		nextPut: $]!

acceptCascadeNode: aCascadeNode 
	| messages |
	messages := aCascadeNode messages.
	self visitNode: messages first receiver.
	self indentWhile: 
			[self 
				for: messages
				do: 
					[:each | 
					self
						indent;
						indentWhile: [self formatMessage: each cascade: true]]
				separatedBy: [codeStream nextPut: $;]]!

acceptLiteralNode: aLiteralNode 
	^self formatLiteral: aLiteralNode value!

acceptMessageNode: aMessageNode 
	| newFormatter code |
	newFormatter := self copy.
	code := newFormatter format: aMessageNode receiver.
	codeStream nextPutAll: code.
	codeStream nextPut: $ .
	newFormatter isMultiLine
		ifTrue: [lineStart := codeStream position - newFormatter lastLineLength].
	self indent: (newFormatter isMultiLine ifTrue: [2] ifFalse: [1])
		while: [self formatMessage: aMessageNode cascade: false]!

acceptMethodNode: aMethodNode 
	self formatMethodPatternFor: aMethodNode.
	self indentWhile: 
			[self formatMethodCommentFor: aMethodNode indentBefore: true.
			self indent.
			self tagBeforeTemporaries ifTrue: [self formatTagFor: aMethodNode].
			aMethodNode body statements isEmpty 
				ifFalse: [self visitNode: aMethodNode body]]!

acceptOptimizedNode: anOptimizedNode 
	codeStream nextPutAll: '##('.
	self visitNode: anOptimizedNode body.
	codeStream nextPut: $)!

acceptReturnNode: aReturnNode 
	codeStream nextPut: $^.
	self visitNode: aReturnNode value!

acceptSequenceNode: aSequenceNode 
	self formatMethodCommentFor: aSequenceNode indentBefore: false.
	self formatTemporariesFor: aSequenceNode.
	self tagBeforeTemporaries ifFalse: 
			[| parent |
			parent := aSequenceNode parent.
			(parent notNil and: [parent isMethod]) ifTrue: [self formatTagFor: parent]].
	self formatStatementsFor: aSequenceNode!

acceptVariableNode: aVariableNode 
	codeStream nextPutAll: aVariableNode name! !

BRFormatter class
	instanceVariableNames: ''!


BRValueNode subclass: #BRLiteralNode
	instanceVariableNames: 'token '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!

BRLiteralNode comment:
'BRLiteralNode is an AST node that represents literals (e.g., #foo, #(1 2 3), true, etc.).

Instance Variables:
	token	<BRLiteralToken>	the token that contains the literal value as well as its source positions

'!


!BRLiteralNode methodsFor: 'accessing'!

precedence
	^0!

startWithoutParentheses
	^token start!

stopWithoutParentheses
	^token stop!

token
	^token!

value
	^token realValue! !

!BRLiteralNode methodsFor: 'comparing'!

= anObject 
	self == anObject ifTrue: [^true].
	self class == anObject class ifFalse: [^false].
	self value class == anObject value class ifFalse: [^false].
	^self value = anObject value!

hash
	^self value hash! !

!BRLiteralNode methodsFor: 'initialize-release'!

literalToken: aLiteralToken 
	token := aLiteralToken! !

!BRLiteralNode methodsFor: 'matching'!

copyInContext: aDictionary
	^self class literalToken: token removePositions! !

!BRLiteralNode methodsFor: 'testing'!

isImmediate
	^true!

isLiteral
	^true! !

!BRLiteralNode methodsFor: 'visitor'!

acceptVisitor: aProgramNodeVisitor 
	^aProgramNodeVisitor acceptLiteralNode: self! !

BRLiteralNode class
	instanceVariableNames: ''!



!BRLiteralNode class methodsFor: 'instance creation'!

literalToken: aLiteralToken 
	^self new literalToken: aLiteralToken!

value: aValue 
	^self literalToken: (BRLiteralToken value: aValue)! !

BRParseTreeRule subclass: #BRReplaceRule
	instanceVariableNames: 'verificationBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRReplaceRule methodsFor: 'initialize-release'!

initialize
	super initialize.
	verificationBlock := [:aNode | true]! !

!BRReplaceRule methodsFor: 'matching'!

canMatch: aProgramNode 
	^verificationBlock value: aProgramNode!

foundMatchFor: aProgramNode 
	self subclassResponsibility! !

BRReplaceRule class
	instanceVariableNames: ''!


BRReplaceRule subclass: #BRStringReplaceRule
	instanceVariableNames: 'replaceTree '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRStringReplaceRule methodsFor: 'initialize-release'!

methodReplaceString: replaceString 
	replaceTree := BRParser parseRewriteMethod: replaceString!

replaceString: replaceString 
	replaceTree := BRParser parseRewriteExpression: replaceString.
	searchTree isSequence = replaceTree isSequence 
		ifFalse: 
			[searchTree isSequence 
				ifTrue: [replaceTree := BRSequenceNode statements: (Array with: replaceTree)]
				ifFalse: [searchTree := BRSequenceNode statements: (Array with: searchTree)]]!

searchFor: searchString replaceWith: replaceString 
	self searchString: searchString.
	self replaceString: replaceString!

searchFor: searchString replaceWith: replaceString when: aBlock 
	self searchFor: searchString replaceWith: replaceString.
	verificationBlock := aBlock!

searchForMethod: searchString replaceWith: replaceString 
	self methodSearchString: searchString.
	self methodReplaceString: replaceString!

searchForMethod: searchString replaceWith: replaceString when: aBlock 
	self searchForMethod: searchString replaceWith: replaceString.
	verificationBlock := aBlock!

searchForTree: aBRProgramNode replaceWith: replaceNode 
	searchTree := aBRProgramNode.
	replaceTree := replaceNode!

searchForTree: aBRProgramNode replaceWith: replaceString when: aBlock 
	self searchForTree: aBRProgramNode replaceWith: replaceString.
	verificationBlock := aBlock! !

!BRStringReplaceRule methodsFor: 'matching'!

foundMatchFor: aProgramNode 
	| newTree |
	newTree := replaceTree copyInContext: self context.
	newTree copyCommentsFrom: aProgramNode.
	^newTree! !

BRStringReplaceRule class
	instanceVariableNames: ''!



!BRStringReplaceRule class methodsFor: 'instance creation'!

searchFor: searchString replaceWith: replaceString 
	^self new searchFor: searchString replaceWith: replaceString!

searchFor: searchString replaceWith: replaceString when: aBlock 
	^self new 
		searchFor: searchString
		replaceWith: replaceString
		when: aBlock!

searchForMethod: searchString replaceWith: replaceString
	^self new searchForMethod: searchString replaceWith: replaceString!

searchForMethod: searchString replaceWith: replaceString when: aBlock 
	^self new 
		searchForMethod: searchString
		replaceWith: replaceString
		when: aBlock!

searchForTree: searchString replaceWith: replaceString 
	^self new searchForTree: searchString replaceWith: replaceString!

searchForTree: searchString replaceWith: replaceString when: aBlock 
	^self new 
		searchForTree: searchString
		replaceWith: replaceString
		when: aBlock! !

BRReplaceRule subclass: #BRBlockReplaceRule
	instanceVariableNames: 'replaceBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Refactory-Parser'!



!BRBlockReplaceRule methodsFor: 'initialize-release'!

initialize
	super initialize.
	replaceBlock := [:aNode | aNode]!

searchFor: searchString replaceWith: aBlock 
	self searchString: searchString.
	replaceBlock := aBlock!

searchFor: searchString replaceWith: replBlock when: verifyBlock 
	self searchFor: searchString replaceWith: replBlock.
	verificationBlock := verifyBlock!

searchForMethod: searchString replaceWith: aBlock 
	self methodSearchString: searchString.
	replaceBlock := aBlock!

searchForMethod: searchString replaceWith: replBlock when: verifyBlock 
	self searchForMethod: searchString replaceWith: replBlock.
	verificationBlock := verifyBlock!

searchForTree: aBRProgramNode replaceWith: aBlock 
	searchTree := aBRProgramNode.
	replaceBlock := aBlock!

searchForTree: aBRProgramNode replaceWith: replBlock when: verifyBlock 
	self searchForTree: aBRProgramNode replaceWith: replBlock.
	verificationBlock := verifyBlock! !

!BRBlockReplaceRule methodsFor: 'matching'!

foundMatchFor: aProgramNode 
	^replaceBlock value: aProgramNode! !

BRBlockReplaceRule class
	instanceVariableNames: ''!



!BRBlockReplaceRule class methodsFor: 'instance creation'!

searchFor: searchString replaceWith: replaceBlock 
	^self new searchFor: searchString replaceWith: replaceBlock!

searchFor: searchString replaceWith: replaceBlock when: aBlock 
	^self new 
		searchFor: searchString
		replaceWith: replaceBlock
		when: aBlock!

searchForMethod: searchString replaceWith: replaceBlock 
	^self new searchForMethod: searchString replaceWith: replaceBlock!

searchForMethod: searchString replaceWith: replaceBlock when: aBlock 
	^self new 
		searchForMethod: searchString
		replaceWith: replaceBlock
		when: aBlock!

searchForTree: searchString replaceWith: replaceBlock 
	^self new searchForTree: searchString replaceWith: replaceBlock!

searchForTree: searchString replaceWith: replaceBlock when: aBlock 
	^self new 
		searchFor: searchString
		replaceWith: replaceBlock
		when: aBlock! !

BRScanner initialize!

BRParser initialize!


