

!StandardSystemController publicMethodsFor: 'menu messages'!

inspectView

	self view model inspect! !

!StandardSystemController class publicMethodsFor: 'class initialization'!

initializeForAdditions
	"Initialize the menu."

	ScheduledBlueButtonMenu := 
		Menu 
			labels: 'relabel as...\refresh\move\resize\front\back\collapse\close\inspect' withCRs 
			lines: #(1 7 8)
			values: #( #newLabel #display #move #resize #front #back #collapse #close #inspectView).! !Object subclass: #MyToolsImplementor
	instanceVariableNames: ''
	classVariableNames: 'SmalltalkFlavour '
	poolDictionaries: ''!

MyToolsImplementor comment: 'I''m the implementation part of the bridge pattern, I or my subclasses have concrete implementations specific to a particular platform.  By convention, concrete methods are prefixed with ''implement'' to denote that they are actually bridged.

Q: How come you denote methods that should be overriden with "self notImplementedYet" as opposed to "self subclassResponsibility"?
A: This provides a friendlier interface to users or porters.  Instead of throwing an unhandled exception, a (somewhat) informative dialog appears.  A friendlier interface could be done via handling the error, but since each flavour implements exception handling differently, this lowest common denominator is a better solution.
'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!




!MyToolsImplementor class publicMethodsFor: 'bridging'!

implementorToUse
	"FIXME - quick-n-dirty test for which implementor to use."

	ParagraphEditor classPool at: #Keyboard
		ifAbsent: 
			[SmalltalkFlavour := 'VisualWorks v3.x'.
			^Vw31_Implementor].
	"implicit else"
	SmalltalkFlavour := 'VisualWorks v2.5.2'.
	^Vw252_Implementor!

smalltalkFlavourString

	^SmalltalkFlavour!

smalltalkTypeString

	^self smalltalkFlavourString! !

!MyToolsImplementor class publicMethodsFor: 'class initialization'!

implementAddToLauncher
	
	self notImplementedYet.!

implementInitialize

	"Stub - do nothing by default."! !

!MyToolsImplementor class publicMethodsFor: 'customization'!

implementSetDefaultKeyBindings

	self notImplementedYet.!

implementSetDeleteKey

	"Stub out by default, implement as needed in subclasses."!

implementSetEmacsKeyBindings

	self notImplementedYet.!

implementSetFileBrowserToShowAllFiles

	self notImplementedYet.! !

!MyToolsImplementor class publicMethodsFor: 'errors - warnings'!

extensionsNotFoundNote: noteText

	Transcript show: noteText; cr.
	Dialog warn: noteText.!

fileNotFoundError: aFileName

	| errorText |
	errorText :=  'ERROR: could not find file: "', aFileName asString, '" to load'.
	Transcript show: errorText; cr.
	Dialog warn: errorText.!

notImplemented

	Dialog warn: 'NOTE:  Not implemented for ', self smalltalkFlavourString.!

notImplementedYet

	Dialog warn: 'NOTE:  Not implemented for ', self smalltalkFlavourString, ' yet.'.!

shouldNotRun

	Dialog warn: 'WARN:  method under construction, not meant to be run yet.  If you are not sure about running this method, break execution now.'.! !

!MyToolsImplementor class publicMethodsFor: 'extensions'!

implementExtendRefactoringBrowserMenus

	self notImplementedYet.! !

!MyToolsImplementor class publicMethodsFor: 'forensics'!

implementMethodToAddWith: aMethod
	"Override as neccessary to return method, method definition, or whatever else is needed.  This concrete implementation is used in the text searching methods."

	^aMethod! !

MyToolsImplementor subclass: #Vw_Implementor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''!

Vw_Implementor comment: 'I implement VW specific code
'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!




!Vw_Implementor class publicMethodsFor: 'class initialization'!

implementAddToLauncher
	"Snippet to add to the Tools menu for the current VisualLauncher(s)."

	| myToolsItem menuChannelIndex launcherClass launcherControllers launcherWindows menuBar menu toolsItem uiSettingsClass |
	myToolsItem := (MenuItem labeled: 'My Tools') 
		submenu: (Menu labelArray: #('MyTools help' 
									'Find complete string...' 
									'Find substring...' 
									'Find substring, background...' 
									'Add extensions to Refactoring Browser'
									'Build MyTools')
						values: ((OrderedCollection new)
								add: [MyTools helpScreen];
								add: [MyTools findString];
								add: [MyTools findSubString];
								add: [MyTools findSubStringInBackground];
								add: [MyTools extendRefactoringBrowserMenus];
								add: [MyTools build];
								asArray)).	

	"Collect the open launchers"
	launcherClass := Smalltalk at: #VisualLauncher ifAbsent: [Dialog warn: 'WARN:  Can''t find VisualLauncher class!!'].
	launcherControllers := ScheduledControllers scheduledControllers select: 
					[:aSchedualedController | aSchedualedController view class == ApplicationWindow 
						and: [aSchedualedController view model isKindOf: launcherClass]].
	launcherWindows := launcherControllers collect: [:aLauncher | aLauncher view].

	"Need to access the menuChannel indirectly here because the menuChannel accessor is not implemented in all versions of VW."
	menuChannelIndex := MenuBar instVarIndexFor: 'menuChannel'.
	menuChannelIndex = 0 ifTrue: [Dialog warn: 'WARN:  Can''t find menu channel index'].

	launcherWindows do: 
			[:aLauncherWindow | 
			menuBar := aLauncherWindow menuBar.
			menu := (menuBar instVarAt: menuChannelIndex) value.
			toolsItem := menu menuItemLabeled: 'Tools' ifNone: [Dialog warn: 'WARN:  Can''t find Tools menu item'].
			(toolsItem submenu labels includes: 'My Tools')
				ifFalse: [toolsItem submenu addItem: myToolsItem.
						menuBar updateMenu]].

	" watch for changes to UISettings"
	uiSettingsClass := Smalltalk at: #UISettings.
	uiSettingsClass notNil 
		ifTrue: 
			[(uiSettingsClass preferenceModelFor: #hostWindowManagerSelector) 
				onChangeSend: #myAddToToolsMenu
				to: self]!

implementInitialize

	StandardSystemController initializeForAdditions.! !

!Vw_Implementor class publicMethodsFor: 'customization'!

implementSetFileBrowserToShowAllFiles

	FileBrowser defaultPattern: '*'! !

!Vw_Implementor class publicMethodsFor: 'forensics'!

implementMethodSelectorFor: method
	
	self notImplementedYet.! !

Vw_Implementor subclass: #Vw252_Implementor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''!

Vw252_Implementor comment: 'I implement VW 2.5.2 specific code
'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!




!Vw252_Implementor class publicMethodsFor: 'class initialization'!

implementInitialize

	super implementInitialize.
	self
		implementSetDefaultKeyBindings;
		implementSetDeleteKey! !

!Vw252_Implementor class publicMethodsFor: 'customization'!

implementSetDefaultKeyBindings
	"Set some reasonable key bindings. Only takes effect on newly opened windows"

	(ParagraphEditor classPool at: #Keyboard)
		bindValue: #copyKey: to: (TextConstants at: #Ctrlx);	
			"A little misleading perhaps, but not so much as over-riding the default break key for copy."
		bindValue: #pasteKey: to: (TextConstants at: #Ctrlv);
		bindValue: #findDialogKey: to: (TextConstants at: #Ctrlf);
		bindValue: #findKey: to: (TextConstants at: #Ctrla);
		bindValue: #acceptKey: to: (TextConstants at: #Ctrls);
		bindValue: #undoKey: to: (TextConstants at: #Ctrlz).

	"Decided not to override Ctrl-c to Micro$oft copy shortcut.  We could do this and override the interrupt key to somethink like Ctrl-q, but for the default toolkit this probably creates more confusion than good.  The trade-off here is to map Ctrlx to copy instead of cut, leaving the default Smalltalk break key in place."

	"If you want to override Ctrl-c, uncomment below text and comment out above two lines"
	"	InputState interruptKeyValue: (TextConstants at: #Ctrlq).
	(ParagraphEditor classPool at: #Keyboard)
		bindValue: #copyKey: to: (TextConstants at: #Ctrlc);
		bindValue: #cutKey: to: (TextConstants at: #Ctrlx);
		bindValue: #pasteKey: to: (TextConstants at: #Ctrlv);
	"
	"The TextEditorController initializes its class var from the ParagraphEditor dispatch table, so we need to reinitialize it."
	TextEditorController initialize.!

implementSetDeleteKey
	"Fix that darn delete key to do what it's supposed to do"

	LookPreferences deleteForward: true!

implementSetEmacsKeyBindings
	" Only takes effect on newly opened windows"

	(ParagraphEditor classPool at: #Keyboard)
		bindValue: #pasteKey: to: (TextConstants at: #Ctrly);
		bindValue: #findDialogKey: to: (TextConstants at: #Ctrls);
		bindValue: #homeKey: to: (TextConstants at: #Ctrla);
		bindValue: #endKey: to: (TextConstants at: #Ctrle);
		bindValue: #deleteKey: to: (TextConstants at: #Ctrld);
		bindValue: #cursorRightKey: to: (TextConstants at: #Ctrlf);
		bindValue: #cursorLeftKey: to: (TextConstants at: #Ctrlb);
		bindValue: #cursorUpKey: to: (TextConstants at: #Ctrlp);
		bindValue: #cursorDownKey: to: (TextConstants at: #Ctrln).
	"The TextEditorController initializes its class var from the ParagraphEditor dispatch table, so we need to reinitialize it."
	TextEditorController initialize.! !

!Vw252_Implementor class publicMethodsFor: 'extensions'!

implementExtendRefactoringBrowserMenus
	"A non-intrusive way to extend the RB menus."

	RefactoringBrowser allInstances do: 
			[:aBrowser | 
			(aBrowser navigator selectorMenu labels includes: 'send this method') 
				ifFalse: 
					[(aBrowser navigator selectorMenu)
						addSeparator;
						add: 'send this method' -> #sendSelectedMethod;
						add: 'inspect this method' -> #inspectSelectedMethod.
					(aBrowser navigator classMenu)
						addSeparator;
						add: 'copy full class spec' -> #copyFullClassSpec;
						add: 'Find substring...' -> #findSubStringOverClasses]]! !

!Vw252_Implementor class publicMethodsFor: 'forensics'!

implementGetSourceFor: method
	
	^method sourceString.!

implementMethodSelectorFor: method
	
	^method mclass printString, ' ', method selector.! !

Vw_Implementor subclass: #Vw31_Implementor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''!

Vw31_Implementor comment: 'I implement VW 3.0 specific code.'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!




!Vw31_Implementor class publicMethodsFor: 'customization'!

implementSetDefaultKeyBindings
	"Set some reasonable key bindings."

	UIFeelPolicy new keyboardDispatchTableWithBellsAndWhistles! !

!Vw31_Implementor class publicMethodsFor: 'extensions'!

implementExtendRefactoringBrowserMenus
	"A non-intrusive way to extend the RB menus."

	RefactoringBrowser allInstances do: 
			[:aBrowser | 
			(aBrowser navigator selectorMenu value labels includes: 'send this method') 
				ifFalse: 
					[aBrowser navigator selectorMenu value 
						addItemGroupLabels: #('send this method' 'inspect this method')
						values: #(#sendSelectedMethod #inspectSelectedMethod).
					aBrowser navigator classMenu value 
						addItemGroupLabels: #('copy full class spec' 'Find substring...')
						values: #(#copyFullClassSpec #findSubStringOverClasses)]]! !

!Vw31_Implementor class publicMethodsFor: 'forensics'!

implementGetSourceFor: method

	^method getSource.!

implementMethodSelectorFor: method
	
	^method definition!

implementMethodToAddWith: aMethod

	^aMethod definition! !

Object subclass: #MyTools
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''!

MyTools comment: 'I''m a proxy for the MyTools toolkit, my responsibility is to act as a single point of reference and forward all messages to MyToolsEnvy (if it exists), or to MyToolsBase otherwise.  This way, I insulate the access point to the toolkit regardless of whether I''m loaded in a VW image or a VW+ENVY image.

Ideally, I should be subclassed off of nil, but since filing out and filing back in doesn''t allow for this, subclassing off of Object is the next best thing to do.  To clean up this proxy, you can either subclass me off of nil after filing in, or override inherited msgs from Object with method forwarders.  (The latter solution has been done for only needed protocol, and not all of Object''s protocol)'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!




!MyTools class publicMethodsFor: 'forwarding'!

doesNotUnderstand: aMessage 
	| realSubject |
	realSubject := Smalltalk at: #MyToolsEnvy
				ifAbsent: 
					[Smalltalk at: #MyToolsBase
						ifAbsent: [Dialog warn: 'MyToolsBase not found - try reloading MyTools.']].
	^realSubject perform: aMessage selector withArguments: aMessage arguments.!

initialize
	"Check for MyToolsBase is done to avoid having this initialization method sent during filein. The only minor problem with implementing a forwarding initialize method on myself, is that for the situations where MyTools is saved in an ENVY app, and the installation files still exist in the home directory, then the user will get 2 prompts about reloading extensions (this also happens with VW when MyTools is saved with an image).  The alternative (assuming subclassing off of nil is not done due to fileout constraints) is to not implement this method, and to make sure people call the initialize method directly from MyToolsBase.  Out of these two choices, the first one is the lessor of two evils.  If I'm subclassed off of nil, this is a nonissue, and my initialize method may be deleted."

	Smalltalk at: #MyToolsBase ifAbsent: [^nil].
	self doesNotUnderstand: (Message selector: #initialize)! !

MyTools initialize!


Object subclass: #MyToolsBase
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''!

MyToolsBase comment: 'MyTools v0.4 - Help information

MyTools is YAST (Yet Another Smalltalk Tool) - this is a collection of various tools and helpful snippets.  Why write yet another one?  Primarily because I haven''t seen a toolkit that is open, portable, and extendible by the community.  Secondly, my intention is that if (when? the Java juggernaut beckons) I leave the Smalltalk community, that I find a good home for MyTools so it keeps getting updated.  This toolkit is freely distributable with all source code, with two caveats:  1) It is not distributed/included for profit (if you want to do this, contact me).  2) That any modifications/improvements/updates are sent to me for inclusion in a future version.  I''ll be sure to add/withhold credit as requested.  See notes in help/documentation protocol of MyToolsBase for more information - especially before firing off questions/suggestions to me.  Enjoy.
		Environments tried in:
			* VW 2.5.2
			* VW 3.1
			* ENVY 3.01
Jason Steffler
jagwar@magma.ca

[R] - indicates it''s for the Refactoring Browser
[E] - indicates it''s for ENVY

Menu Items
	* Find complete string
		* Looks for occurances of entire string within the image. (case sensitive, relatively fast)
	* Find substring
		* Allows you to scope the search by specifying class to start in.  Will search all subclasses.  
		* Not case sensitive, relatively slow (why scoping is good) and runs in the foreground.
	* Find substring, background.
		* Same as find substring, but runs in background for those times when you can''t reduce the scope.
      * Add extensions to Refactoring Browser
		* Only addes extensions to currently open refactoring browsers
      * Build MyTools
		* Make MyTools fileouts for the various platforms that it is possible for current environment to do

Miscellaneous Protocol
	"An easier BOSS interface."
	MyTools bossOutObject: anObject on: ''aFilenameString''.
	MyTools bossInObjectFrom: ''aFilenameString''.

	"Various Customizations"
	MyTools addToLauncher.			"MyTools submenu is dropped on image save for some reason."
	MyTools setDefaultKeyBindings.
	MyTools setEmacsKeyBindings.
	MyTools setDeleteKey.
	MyTools setFileBrowserToShowAllFiles.
	MyTools setUiLookToMotif.
[R]	MyTools extendRefactoringBrowserMenus 	"A non-intrusive way to extend the RB menus."

	"Misc Envy stuff."
[E]	MyTools loadConfigMapsWithRequired.
[E] 	MyTools loadConfigMapsWithRequired: ''ENVY/Manager''.
[E]	MyTools reportOnConfigMapDiffs.	
[E]	MyTools reportOnConfigMapDiffs: #(''ConfigMap1'' ''ConfigMap2'').
[E]	MyTools changeOwnershipsOfUser.		"Use this method with care!!"

Examples
	"There''s a number of examples of snippets of how to do things in the examples protocol on the class side of MyToolsBase"
	MyTools browseAllClassExamples.
	MyTools changeImageColours.		"Not the best choice in the world for the defaults ;-)"
	MyTools openWindowWithTextInIt.
	MyTools openWindowWithCapturedImage.

Your own extensions
	If your extensions are generic, please send them to me for future inclusion in the toolkit.  If your extensions are employer/situation-specific, then your best option is to subclass off of the MyTools proxy class, or if you prefer, you can subclass off of MyToolsBase or MyToolsEnvy, as the case may be.

Extensions to base
	*NOTE* These are the extensions to the base classes.  
		They will show up in the category where the base class is defined.
[E]		They will show up in the default application.
	StandardSystemController>inspectView
	StandardSystemController class>initializeForAdditions
[R]	SystemNavigator>findSubStringOverClasses
[R]	SystemNavigator>inspectSelectedMethod
[R]	SystemNavigator>sendSelectedMethod
[R]	SystemNavigator>copyFullClassSpec
[E]	EtApplicationsChangesBrowser>printChangesReportToTranscript'!

"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!




!MyToolsBase class publicMethodsFor: 'bossing'!

bossInObjectFrom: aFilenameString 
	"A friendlier BOSS i/f.  Answer the first object in the BOSS File named aFilename"

	| b obj |
	^Cursor read
		showWhile: 
			[b := BinaryObjectStorage onOld: aFilenameString asFilename readStream.
			[obj := b next]
				valueNowOrOnUnwindDo: [b close].
			obj]!

bossOutObject: anObject on: aFilenameString 
	"A friendlier BOSS i/f.  Boss out anObject on a filename."

	| b |
	Cursor write
		showWhile: 
			[b := BinaryObjectStorage onNew: aFilenameString asFilename writeStream.
			[b nextPut: anObject]
				valueNowOrOnUnwindDo: [b close]]! !

!MyToolsBase class publicMethodsFor: 'bridging'!

myBridge

	^MyToolsImplementor implementorToUse.! !

!MyToolsBase class publicMethodsFor: 'building'!

build

	Transcript cr; show: 'Starting fileout of MyTools...'.
	self myNotifyAboutEnvyExtensions.
	self myNotifyAboutRefactoringExtensions.
	self myFileOutForVw.
	self myFileOutForVwEnvy.
	Transcript cr; cr; show: '...done filing out MyTools'.!

myBaseClasses
	"MyToolsBase is last in this order, as it has the initialization code."

	^#(#MyToolsImplementor #Vw_Implementor #Vw252_Implementor #Vw31_Implementor #MyTools #MyToolsBase)!

myBaseClassExtensions

	^(Dictionary new)
		at: #StandardSystemController
			put: #(#inspectView #initializeForAdditions) asSet;
		yourself!

myBaseFileName

	^('MyTools-Vw-' , self version , '.st') asFilename!

myEnvyExtensionsFileName

	^('MyTools-VwEnvyExtensions-' , self version , '.dat') asFilename!

myEnvyFileName

	^('MyTools-VwEnvy-' , self version , '.st') asFilename!

myEnvyRefactoringExtensionsFileName

	^('MyTools-VwEnvyRefactoringExtensions-' , self version , '.dat') asFilename!

myFileInRefactoringExtensionsFileName
	"Have this indirection to allow for the same initialization method, and two different filenames to be used in MyToolsBase and MyToolsEnvy."

	^self myRefactoringExtensionsFileName.!

myFileOutClasses: classes andExtensions: classExtensions onFileName: aFileName 
	"Extensions are filed out first, since the base initialization references an extension."

	| sourceStream firstClass |
	Transcript cr; show: '### Filing out: ', aFileName asString, '...'.
	sourceStream := SourceCodeStream write: aFileName encoding: #Source.
	classExtensions notNil 
		ifTrue: 
			[classExtensions keys do: 
					[:anExtension | 
					sourceStream 
						fileOutMessages: (classExtensions at: anExtension)
						for: (Smalltalk at: anExtension)
						logging: true.
					sourceStream 
						fileOutMessages: (classExtensions at: anExtension)
						for: (Smalltalk at: anExtension) class
						logging: true]].
	firstClass := true.
	classes notNil 
		ifTrue: 
			[classes do: 
					[:aClass | 
					firstClass 
						ifTrue: [firstClass := false]
						ifFalse: 
							[sourceStream cr; newPage; cr].
					(Smalltalk at: aClass
						ifAbsent: 
							[Dialog warn: 'Couldn''t find class ' , aClass asString , 'to file out']) 
							fileOutSourceOn: sourceStream]].
	sourceStream
		finishInitializations;
		close!

myFileOutForVw

	self 
		myFileOutClasses: self myBaseClasses
		andExtensions: self myBaseClassExtensions
		onFileName: self myBaseFileName.

	self refactoringExtensionsLoaded ifTrue: [
		"The RB extensions are filed out separately to reduce coupling."
		self 
			myFileOutClasses: nil
			andExtensions: self myRefactoringExtensions
			onFileName: self myRefactoringExtensionsFileName].!

myFileOutForVwEnvy

	| fileNames |
	fileNames := (OrderedCollection new)
				add: self myEnvyFileName;		
				yourself.
	self 
		myFileOutClasses: self myBaseClasses
		andExtensions: self myBaseClassExtensions
		onFileName: self myEnvyFileName.

	self refactoringExtensionsLoaded ifTrue: [
		"The RB extensions are filed out separately to reduce coupling."
		self 
			myFileOutClasses: nil
			andExtensions: self myRefactoringExtensions
			onFileName: self myEnvyRefactoringExtensionsFileName.
			fileNames add: self myEnvyRefactoringExtensionsFileName].

	self myMassageForVwFileOuts: fileNames.!

myMassageFilterWithReadStream: readStream andWriteStream: writeStream
	"Filter depends on method of massage.  For myself, I'm doing VW->VW+ENVY, and need to take out any references to categories, and can accomplish this without references to the writeStream."

	readStream skipToAll: '!!'.!

myMassageForVwFileOuts: fileNames

"FIXME - the filing out process should be changed to have all base classes in the base fileout, and all envy classes+extensions in a .dat file"

	| readStream writeStream stringToSearchFor fileNameTmp1 fileNameTmp2 |
	"Need to massage fileout from flavour1 to be readable by flavour2 - ie: VW->VW+ENVY"
	fileNameTmp1 := 'MyTools-Vw-Tmp1.st' asFilename.
	fileNameTmp2 := 'MyTools-Vw-Tmp2.st' asFilename.
	fileNames do: 
			[:aFileName | 
			aFileName moveTo: fileNameTmp1.

			"Stage 1 - remove categories."
			readStream := fileNameTmp1 readStream.
			writeStream := fileNameTmp2 writeStream.
			[readStream atEnd] whileFalse: 
					[writeStream
						nextPutAll: (readStream throughAll: 'poolDictionaries: ' , '''''').
					readStream atEnd 
						ifFalse: [self myMassageFilterWithReadStream: readStream andWriteStream: writeStream]].
			readStream close.
			writeStream close.

			"Stage 2 - massage flavour1method defs to flavour2 method defs."	
			readStream := fileNameTmp2 readStream.
			writeStream := aFileName writeStream.
			stringToSearchFor := self myMassageMethodStringToChange. 
			[readStream atEnd] whileFalse: 
					[writeStream nextPutAll: (readStream upToAll: stringToSearchFor).
					readStream atEnd 
						ifFalse: 
							[writeStream nextPutAll: self myMassageMethodStringReplacement.
							readStream throughAll: stringToSearchFor]].
			readStream close.
			writeStream close.
			fileNameTmp1 delete.
			fileNameTmp2 delete]!

myMassageMethodStringReplacement

	^'publicMethodsFor: '.!

myMassageMethodStringToChange
	"Need to avoid meta corruption.  Set the string to search for by concatenating so when filing out this method, the string to search for isn't replaced, thus corrupting this fileout method for the next time around."

	^'methods' , 'For: '.!

myNotifyAboutEnvyExtensions

	self envyExtensionsLoaded 
		ifFalse: [Transcript cr; show: '--->NOTE:  Cannot build ENVY extensions file since you are building out of VW<---'.]!

myNotifyAboutRefactoringExtensions

	self refactoringExtensionsLoaded 
		ifFalse: [Transcript cr; show: '--->NOTE:  Cannot build refactoring browser extensions file since they are not loaded<---'.]!

myRefactoringExtensions

	^(Dictionary new)
		at: #SystemNavigator
			put: #(#findSubStringOverClasses #inspectSelectedMethod #sendSelectedMethod  #copyFullClassSpec);
		yourself!

myRefactoringExtensionsFileName

	^('MyTools-VwRefactoringExtensions-' , self version , '.dat') asFilename! !

!MyToolsBase class publicMethodsFor: 'class initialization'!

addToLauncher

	self myBridge implementAddToLauncher!

initialize

	self
		myLoadApplicableExtensions;
		addToLauncher;
		setFileBrowserToShowAllFiles.
	self	myBridge implementInitialize.!

myLoadApplicableExtensions

	| loadRefactoringExtensions loadEnvyExtensions fileNameToLoad|

	self myCheckForLoadingExtensionsIfInEnvy ifFalse: [^nil].

	fileNameToLoad := self myFileInRefactoringExtensionsFileName.
	loadRefactoringExtensions := true.
	Smalltalk at: self myRefactoringExtensions keys asOrderedCollection first
		ifAbsent: 
			[self myBridge extensionsNotFoundNote: 'NOTE:  Refactoring Browser not found, these extensions will not be loaded.'.
			loadRefactoringExtensions := false].
	loadRefactoringExtensions
		ifTrue: [fileNameToLoad exists 
			ifTrue: [fileNameToLoad fileIn] 
			ifFalse: [self myBridge fileNotFoundError: fileNameToLoad]].

	fileNameToLoad := self myEnvyExtensionsFileName.
	loadEnvyExtensions := true.
	self inEnvyEnvironment 
		ifFalse: [self myBridge extensionsNotFoundNote:  'NOTE:  Envy not found, these extensions will not be loaded.'.
				loadEnvyExtensions := false].
	loadEnvyExtensions 
		ifTrue: [fileNameToLoad exists 
			ifTrue: [fileNameToLoad fileIn] 
			ifFalse: [self myBridge fileNotFoundError: fileNameToLoad]].! !

!MyToolsBase class publicMethodsFor: 'customization'!

setDefaultKeyBindings

	self myBridge implementSetDefaultKeyBindings.!

setDeleteKey

	self myBridge implementSetDeleteKey.!

setEmacsKeyBindings
	"For those old-timers out there."

	self myBridge implementSetEmacsKeyBindings.!

setFileBrowserToShowAllFiles

	self myBridge implementSetFileBrowserToShowAllFiles.!

setUiLookToMotif
	"to change UI Look"

	(UISettings preferenceModelFor: #defaultLookSelector) 
		value: #MotifLookPolicy! !

!MyToolsBase class publicMethodsFor: 'envy stubbing'!

changeOwnershipsOfUser

	"Stub out here to give VW users feedback if they try and exercise the ENVY protocol in VW"
	self onlyImplementedInEnvy!

loadConfigMapsWithRequired 

	"Stub out here to give VW users feedback if they try and exercise the ENVY protocol in VW"
	self onlyImplementedInEnvy!

loadConfigMapsWithRequired: configMapNamesToLoad

	"Stub out here to give VW users feedback if they try and exercise the ENVY protocol in VW"
	self onlyImplementedInEnvy!

onlyImplementedInEnvy

	Dialog warn: 'NOTE:  Sorry, this function is only implemented in ENVY.'!

reportOnConfigMapDiffs

	"Stub out here to give VW users feedback if they try and exercise the ENVY protocol in VW"
	self onlyImplementedInEnvy!

reportOnConfigMapDiffs: configMapsToCompare

	"Stub out here to give VW users feedback if they try and exercise the ENVY protocol in VW"
	self onlyImplementedInEnvy! !

!MyToolsBase class publicMethodsFor: 'examples'!

browseAllClassExamples

	Browser browseAllClassExamples.!

changeImageColours
	"Not the best selection of colours for sure... ;-)"

	MotifWidgetPolicy defaultColorWidgetColors
	  	matchAt: SymbolicPaint background put: ColorValue pink.
	Screen default updatePaintPreferences.
	ScheduledControllers restore.!

openWindowWithCapturedImage

	| aScheduledWindow |
	aScheduledWindow := ScheduledWindow
	  	model: nil
			label: 'Image'
			minimumSize: 200 @ 200.
	aScheduledWindow component: Image fromUser.
	aScheduledWindow open!

openWindowWithTextInIt

	ComposedTextView open: (ValueHolder with: 'Hello World')! !

!MyToolsBase class publicMethodsFor: 'extensions'!

extendRefactoringBrowserMenus
	"FIXME - this is defensive programming; if the RB is not loaded, then it shouldn't be a menu option.  But, short on time at the moment, so I'll just do this defensively. (this is in to do notes)"

	MyTools refactoringExtensionsLoaded 
		ifTrue: [self myBridge implementExtendRefactoringBrowserMenus.]
		ifFalse: [self myBridge extensionsNotFoundNote: 'The refactoring browser is not loaded.'].! !

!MyToolsBase class publicMethodsFor: 'forensics'!

findString

	| searchString |
	searchString := Dialog request: 'String to search for'
				initialAnswer: ParagraphEditor currentSelection string.
	searchString isEmpty ifTrue: [^nil].
	Browser browseAllCallsOn: searchString.!

findSubString
	"Searches in foreground for specified substring (no wildcards needed), and recurses down subclasses."

	| methods startingClass wildCardString allSelectors searchString |
	startingClass := Dialog request: 'Class?'
				initialAnswer: ParagraphEditor currentSelection string.
	startingClass isEmpty ifTrue: [^nil].
	startingClass := Smalltalk at: startingClass asSymbol
				ifAbsent: 
					[Dialog warn: 'Class ' , startingClass , ' wasn''t found.'.
					^nil].
	searchString := Dialog request: 'String to search for'
				initialAnswer: ParagraphEditor currentSelection string.
	searchString isEmpty ifTrue: [^nil].
	wildCardString := '*' , searchString , '*'.
	methods := OrderedCollection new.
	Cursor wait showWhile: 
			[startingClass withAllSubclasses do: 
					[:cls | 
					allSelectors := cls selectors.	"Add the instance selectors of the class"
					allSelectors addAll: cls class selectors.	"Add the class selectors of the class"
					allSelectors do: 
							[:selector | 
							"Get the selector if it is an instance method, failing that get a selector if it is 
							a class method, failing that print an error to the transcript and return false"
							| method source |
							method := cls compiledMethodAt: selector
										ifAbsent: 
											[cls class compiledMethodAt: selector
												ifAbsent: 
													[Transcript
														show: 'missing method: ';
														show: selector;
														cr.
													false]].
							method == false 
								ifFalse: 
									[source := self myBridge implementGetSourceFor: method.
									(source notNil and: [wildCardString match: source]) 
										ifTrue: [methods add: (self myBridge implementMethodSelectorFor: method)]]]]].
	methods isEmpty 
		ifTrue: [Dialog warn: 'No occurances of: ' , wildCardString , ' were found.']
		ifFalse: 
			[MethodListBrowser 
				openListBrowserOn: methods
				label: 'Methods with: ' , wildCardString printString
				initialSelection: searchString]!

findSubStringForClasses: anArrayOfClassNames 
	"FIXME - work in progress, needs refactoring.  Searches in foreground for specified substring (no wildcards needed) - no recursion."

	| methods wildCardString allSelectors searchString anArrayOfClasses |
	searchString := Dialog request: 'String to search for'
				initialAnswer: ParagraphEditor currentSelection string.
	searchString isEmpty ifTrue: [^nil].
	wildCardString := '*' , searchString , '*'.
	methods := OrderedCollection new.

	"FIXME - refactoring note, this method differs from findSubString method in that it accepts args, and the line below."
	anArrayOfClasses := anArrayOfClassNames 
				collect: [:aClassName | Smalltalk at: aClassName].
	Cursor wait showWhile: 
			[anArrayOfClasses do: 
					[:cls | 
					allSelectors := cls selectors.	"Add the instance selectors of the class"
					allSelectors addAll: cls class selectors.	"Add the class selectors of the class"
					allSelectors do: 
							[:selector | 
							"Get the selector if it is an instance method, failing that get a selector if it is a class method, failing that print an error to the transcript and return false"

							| method source |
							method := cls compiledMethodAt: selector
										ifAbsent: 
											[cls class compiledMethodAt: selector
												ifAbsent: 
													[Transcript
														show: 'missing method: ';
														show: selector;
														cr.
													false]].
							method == false 
								ifFalse: 
									[source := self myBridge implementGetSourceFor: method.
									(source notNil and: [wildCardString match: source]) 
										ifTrue: [methods add: (self myBridge implementMethodSelectorFor: method)]]]]].
	methods isEmpty 
		ifTrue: [Dialog warn: 'No occurances of: ' , wildCardString , ' were found.']
		ifFalse: 
			[MethodListBrowser 
				openListBrowserOn: methods
				label: 'Methods with: ' , wildCardString printString
				initialSelection: searchString]!

findSubStringInBackground
	"Searches in background for specified substring (no wildcards needed), and recurses down subclasses."
	"FIXME - consider refactoring this method and findSubString method - what about high granularity though?"
	| methods startingClass wildCardString allSelectors searchString |

	startingClass := Dialog request: 'Class?' initialAnswer: ParagraphEditor currentSelection string.
	startingClass isEmpty ifTrue: [^nil].
	startingClass := Smalltalk at: startingClass asSymbol
				ifAbsent: 
					[Dialog warn: 'Class ' , startingClass , ' wasn''t found.'.
					^nil].
	searchString := Dialog request: 'String to search for' initialAnswer: ParagraphEditor currentSelection string.
	searchString isEmpty ifTrue: [^nil].
	wildCardString := '*' , searchString , '*'.

  methods := OrderedCollection new.
  startingClass withAllSubclasses do: 
	[:cls | 
	"For each class we are searching, yield control at a lower priority than the 
	userSchedulingPriority, as well as a lower priority than for each method 
	below, so the methods being searched resume before iterating over the 
	classes."
	[allSelectors := cls selectors.  "Add the instance selectors of the class"
	allSelectors addAll: cls class selectors. "Add the class selectors of the class"
	allSelectors do: 
	  [:selector | 
		[| method source |
		  "Get the selector if it is an instance method, failing that get a 
		  selector if it is a class method, failing that print an error to the 
		  transcript and return false"

		method := cls compiledMethodAt: selector 
				  ifAbsent: [cls class compiledMethodAt: selector
					ifAbsent: [Transcript show: 'missing method: '; show: selector; cr. false]].
		(method == false)
	  ifFalse: 
	    [source := self myBridge implementGetSourceFor: method.
	    (source notNil and: [wildCardString match: source])
	      ifTrue: ["If there is a match, add the method to the 
		OrderedCollection methods, yield control, and fork at a priority slightly lower 
		than the userSchedulingPriority. This allows VW to refresh 
		itself so the user can busy themselves with other tasks."
		methods add: (self myBridge implementMethodSelectorFor: method)]].
		 Processor yield] 
	   forkAt: 49].
	"Check to see if we're on the last class of the OrderedCollection 
	we are checking, and if so open the MethodsBrowser and get a screen 
	refresh so that the user doesn't have to manually refresh the window."
	(cls = startingClass withAllSubclasses last)
	  ifTrue: 
		[methods isEmpty
			ifTrue: [Dialog warn: 'No occurances of: ', wildCardString, ' were found.']
			ifFalse: [MethodListBrowser 
				openListBrowserOn: methods
				label:  'Methods with: ' , wildCardString printString
				initialSelection: searchString.
				ScheduledControllers restore]].
	Processor yield]
  forkAt: 48]! !

!MyToolsBase class publicMethodsFor: 'help/documentation'!

helpScreen
	"Open a help screen based on my class comment."

	ComposedTextView open: (ValueHolder with: self comment)
		label: 'MyTools help information'
		icon: (Icon constantNamed: #workspace)
		extent: 500@600!

notesBuilding

^'
The intention is to be able to build for all applicable environments from any one environment.  Obviously, if you are running in a nonENVY environment, you will not be able to build for that though.  If this toolkit is ported to other flavours, I expect the filing out methods will need to be bridged.

For example, if you are running in VW252+ENVY301, build for:  VW252, VW252+ENVY301, VW30, VW30+ENVY301.  
For example, if you are running in VW252, build for: VW252, VW30.
'!

notesChangeHistory

^'
Change history:
0.4
  * Modularized ENVY extensions into their own .dat file - a cleaner implementation and makes exchanging code between VW platforms easier
  * Bugs addressed: 
	003) Busy cursor is not displayed when doing a foreground search.
	005) [VW 2.5.2] Installed keyboard shortcuts do not work in text pane of RB
	  * Turns out the RB uses a valueholder on Text, which has events processed via TextEditorController, which in turn has its own copy of the dispatch table.  Just needed to reinitialize TextEditorController.
	006) [ENVY] Loaded in MyTools, versioned and released classes to app.  When reloading app, the extensions are reloaded too causing unnecessary class editions. 
	  * If extension files are detected, user is now prompted if they want to refilein the extensions
	  * Note: this behaves the same in VW when MyTools is already loaded in the image.
	007) The building facilities assume the RB is loaded, if it is not loaded a walkback occurs.
	008) [RB] Add extensions to RB menu item assumes that RB is loaded, if it is not loaded, a walkback occurs
	  * Put in a defensive fix so that a dialog appears when this condition is met.  Certainly not the best fix (would prefer to just disable or remove the inappropriate menu option alltogether), but I am short on time right now so this will have to do.

0.3
  * Added some ENVY extensions
	* building in ENVY, finding string tweaks, Config map loading, config map diffing, change ownerships, copy class specification [R], storing workspaces in comments of methods
  * Decoupled Refactoring Browser extensions from the base installation

0.2
  * Added help text
  * Added notes in help/documentation protocol of MyToolsBase
  * Refactored design to make it more portable across versions/dialects of Smalltalk (put in proxy, bridge).  However, as noted in the help text, the primary platform for this toolkit is VWNC.  I do not have the spare time to regularly or ever try/test it on other platforms.
  * Added a packaging facility (file out).
  * Added naming convention to indicate that a method is bridged (implement*)
  * Added naming convention to indicate private methods (my*)
  * Added emacs keybindings
  * Added refactoring browser extensions
 
0.1
  * Initial implementation, purely for myself and a few friends on VWNC
'!

notesDevelopmentConsiderations

^'
This toolkit started out in a VWNC environment, which is its primary target.  Though I have tried to keep portability in mind while adding to it, my spare time is too constrained to make efforts to try it out and tweak it in other environments.  I would be happy to include fixes necessitated by other environments if sent to me.  If sending fixes/updates, just change the version indicator (MyToolsBase>version) to be something like: "0.3+JoeSmithAdditions", then use the Tools>MyTools>Build MyTools menu option to get the fileout to send to me (do not change ordering of classes/extensions in building methods, as that will mess up my sdiff - just tack on changes to the end)  Note:  if you are planning on doing large changes/extensions, pls contact me first as this method of adding changes/extensions will probably break down with large deltas.

This toolkit follows an old tradition of accumulating tools in class methods for easy access during gigs.  Constraining the refactoring and writing at a high level of granularity is intended to make it easier to grab a method to print out as a sample for sharing, aid in porting between versions/flavours of Smalltalks, and simplify the toolkit interface.  A number of tools are available in the class methods of MyTools and MyToolsEnvy, but are accessed through the MyTools proxy class for a single point of access.  As part of programming to the lowest common denominator, keep these things in mind:
  * If programming in ENVY, only use public protocols.  
  * Do not use public protocols denoted as private on MyTools* classes (ie: building-private).  Since there should not be many private methods while programming at a high level of granularity, having private protocols is unnecessary clutter.  (Maybe this will change, will have to see how the toolkit evolves)
  * If you need to refactor, please prefix private methods with "my" to indicate they are private
  * To aid portability
	* Use the supplied bridge to implement flavour-specific code.  
	* By convention, concrete methods are prefixed with "implement" to denote that they are actually bridged.

Q: Why do you have the bridge set up the way you do?  Why not have refined abstractions represent the different flavours of Smalltalk and have the concrete implementors represent the different configuration managemnt tools available to Smalltalk?
A: This view of the design is the first one that intuitively suggests itself, but I chose to do it in the vice-versa manner because I expect there to be more variety of differences in the Smalltalks realm.  My intent is to insulate users of the ToolKit by pushing the greater variety to the implementor of the bridge, allowing for a cleaner, easier use interface for the toolkit.  This also avoids additional complexity of implementing a chain of responsibility between abstraction and implementors that would be needed (Proxy would need to point to ENVY implementor if ENVY was present, as ENVY enhancements are on top of the Smalltalk base, and the ENVY class would in turn need to point to the Smalltalk flavour being used to catch interface selectors that it does not implement)

Q: How come you have the error/warning dialogs on the bridge, and not on MyToolsBase?
A: It is a tossup as to whether I should put them on MyToolsBase or on MyToolsImplementor.  Either way, I am going to need to delegate sometimes.  Since I expect to use these methods more on bridge implementation code, I put them on the bridge.

Q: Why is the proxy #MyTools not subclassed off of nil?  That is a much better Smalltalk implementation, otherwise the proxy understands a number of messages that I might want forwarded.
Q: Because doing that does not work with fileouts.  Trying to refile back in will cause an error.  Subclassing off of Object is the best lowest common denominator solution.  By all means, change the subclassing of the proxy to be off of nil after filing back in.

Q: I added a class or class extension to the toolkit, why is it not being filed out?
A: You need to update the filing out methods myFileOutFor* as appropriate.  

Q: If you want this to be portable, how come you have not used collaborator methods?
A: See MyTools notesToDo

Q: How come you denote methods that should be overriden with "self notImplementedYet" as opposed to "self subclassResponsibility"?
A: This provides a friendlier interface to users or porters.  Instead of throwing an unhandled exception, a (somewhat) informative dialog appears.  A friendlier interface could be done via handling the error, but since each flavour implements exception handling differently, this lowest common denominator is a better solution.
'!

notesDevelopmentReminders

^'
If you are updating MyTools, remember:
* denote methods that need further work with a "FIXME" string
* when adding a method 
  * add method to MyToolsBase class comment for the help screen
* when adding a bridged method
  * be sure to add a self subclass responsibility method to superclass
  * add a concrete implementation in other implementors as well, if you know of them
* when adding an ENVY method
  * be sure to stub out in MyToolsBase (see envy stubbing protocol)
* if making a new version of the toolkit, be sure to update the MyToolsBase>version method, as well as the version text in the help text  (MyToolsBase class comment).
'!

notesForFilingOut

^'
Given the desire for portability, there is a need to use the lowest common denominator for exchanging code, which is filing out.  To achive portable fileouts, there is a need to build a matrix of fileout methods that will file out from one environment and massage the fileout for other enviroments.
'!

notesKnownBugs

^' (Last bug number used: 008)
001) [RB] When no app selected in RB, and choose copy full class spec, get an error when ask nil isVersion
002) MyTools extension to Tools menu on launcher is lost when launcher is saved
'!

notesMisc

^'
Miscellaneous Notes
	* To do work-specific extensions, just subclass MyTools and extend as necessary.
	* ENVY
		* Though I have tried to identify the aspects of this toolkit that use/require ENVY, it has all been written in an ENVYized image and its primary target is for ENVYized images, so there may be methods not categorized into the ENVY protocols that I missed.
		* If you are using ENVY, it is a good idea to put <MyTools addToToolsMenu> in the application loaded method to autoinitialize the classes.
		* An easy way to store workspaces for future reference is in the comment of a method.  If you are using EMVY, if you have a workspace(s) that you often use, just put the workspace invocation in the loaded method of the class app.  This allows for developers to store/share whatever workspaces they want, without namespace problems.  To try this facility, do: MyTools workspaceJasonDefault.

Change history:
0.2
  * Refactored design to make it more portable across versions/dialects of Smalltalk
  * Added a building facility (file out).
'!

notesPortability

^'
Though I wrote this toolkit with portability in mind, this is not something I really have the time or access to environments to try out.  Well, maybe I will try it out in Squeak someday.  Anyway, if you want to port this toolkit to another environment, I will be glad to incorporate the changes required in future versions of the toolkit.  Also, I am willing to offer tips and what assistance my spare time allows.  Alternatively, providing the environment (and $) would also help :-).
'!

notesTesting

^'
Since this is something I am doing in my spare time, I just test enough for my own personal purposes.  If there is something that is outside of my typical use scope, then I will probably not pick up on it.  But, here is a list of things that can be used as a basis for testing.  Ideally, it would be nice to get Becks free testing framework to automate some unit tests, but that is another thing my spare time does not allow for.

* Ensure that protocol actually supported in MyToolsBase and MyToolsEnvy is in the help screen (class comment of MyToolsBase)
* In all environments:
  * Try the filein (with and without RB loaded)
  * Try the build (with and without RB loaded)
  * Check to see if MyTools has been added to the Transcript
  * Try all actions off of the MyTools submenu
  * Exercise all applicable protocol specified in the help screen
  * Try all extensions to RB
'!

notesToDo

^'
...In my copious amounts of spare time :-)

  * Document design (HotDraw, html?)
  * Work the known bugs
  * Add these various notes to submenu off of MyTools submenu
  * Go through code, and address FIXMEs
  * Fix menus so that only available options are shown or enabled.  Currently, defensive code is in place. (ie: MyToolsBase>extendRefactoringBrowserMenus)
  * come up with a better implementation of MyToolsImplementor>implmentorToUse
  * refactor filing out code for better reuse (across MyToolsBase & MyToolsEnvy, as well as for ENVY & RB extensions)
  * Move the searching code to the bridge?  Probabaly a lot of VW specific code in there, but not sure how much.
  * Do some sort of sanity check that the code filed in OK
  * Be able to scope text searches by categories/applications in various browsers (done for [R])
  * Look into adding some good public domain code to the toolkit
  * Find out why addition to tools menu is lost on saving the image
  * Move nicer BOSS i/f to bridge (BOSS is a VW only protocol)
  * Find a way to extend the EtApplicationsManager without having to over-ride the base class side method
  * Think about a way to store workspaces in VW in a convenient manner (with intermixed comments and code)
  * Think about insulating class refs by using collaborator methods.  See example in method
'
"
---
classUiFeelPolicy
	^Smalltalk at: #UIFeelPolicy ifAbsent: [self warnClassNotFound: 'UIFeelPolicy'.].

warnClassNotFound: aClassName
	self classDialog warn: 'Warning:  Class ''', aClassName, ''' was not found.'.
	^nil

	* Use collaborator methods to insulate against referencing nonexistant classes in different flavours of Smalltalk
	  * By convention, these methods are prefixed with 'class' to denote they're a collaboborator method.
		* Since error handling varies a lot from platform to platform, errors with classes not found are simply handled by opening a warning dialog to this effect (self warnClassNotFound: aClass), and then letting the walkback occur.
---
"! !

!MyToolsBase class publicMethodsFor: 'misc'!

copySpecForClass: selectedClass inApplication: selectedApp

	"Stub out here to give VW users feedback if they try and exercise the ENVY protocol in VW"
	self onlyImplementedInEnvy!

version
	"If there's a '+' in my version string, that indicates a work-in-progress."

	^'0.4'! !

!MyToolsBase class publicMethodsFor: 'testing'!

envyExtensionsLoaded

	Smalltalk at: #MyToolsEnvy ifAbsent: [^false].
	^true.!

inEnvyEnvironment
	"Don't query ENVY extensions for a class to check, as MyToolsEnvy knows this and isn't necessarily around to query."

	Smalltalk at: #EtApplicationsChangesBrowser ifAbsent: [^false].
	^true.!

myCheckForLoadingExtensionsIfInEnvy
	"Only load my extensions if I'm being filed in.  This check is done to compensate for ENVY systems reloading me, if that's the case then I don't want to refilein my extensions and create unnecessary editions.  If this is always the case and you're frequently reloading the application, you may want to over-ride this method to just return false to save you time."

	self myFileInRefactoringExtensionsFileName exists ifFalse: [^false].
	self envyExtensionsLoaded 
		ifTrue: [^Dialog confirm: 'WARN:  MyTools files have been found, do you wish to *filein* extensions? (May cause unnecessary class editions)' initialAnswer: false.]
		ifFalse: [^true].!

refactoringExtensionsLoaded

	Smalltalk at: #SystemNavigator ifAbsent: [^false].
	^true.! !

MyToolsBase initialize!

