Object subclass: #EndToolState
	instanceVariableNames: 'name command '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
EndToolState comment:
'EndToolState is a state for tools. When this state is entered, the tool evaluates its command, and then returns to the initial state of the tool.

EndToolState is also the superclass of ToolState. While it might be better to have an abstract class that is the superclass for both these classes. The abstract class and EndToolState would be essentially the same. The only difference would be that EndToolState would have its isEndState return true.

Instance Variables:
	command	<BlockClosure>	the command that is executed when we are entered, it takes two arguments, the tool and the event
	name	<String>	our state name
'!


!EndToolState methodsFor: 'accessing'!

command
	^command!

command: anObject
	command := anObject!

name
	^name!

name: anObject
	name := anObject! !

!EndToolState methodsFor: 'printing'!

printOn: aStream 
	aStream
		nextPutAll: self class name;
		nextPut: $(;
		nextPutAll: self name;
		nextPut: $)! !

!EndToolState methodsFor: 'testing'!

isEndState
	^true! !

!EndToolState methodsFor: 'tool-accessing'!

evaluateIn: aTool event: anEvent 
	command
		value: aTool
		value: anEvent!

nextStateForEvent: anEvent tool: aTool 
	^nil! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

EndToolState class
	instanceVariableNames: ''!


!EndToolState class methodsFor: 'instance creation'!

name: aString command: aBlock 
	^(self new) name: aString;
		command: aBlock;
		yourself! !

EndToolState subclass: #ToolState
	instanceVariableNames: 'keyPress keyRelease mouseMove redButtonPress redButtonRelease yellowButtonPress yellowButtonRelease doubleClick '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
ToolState comment:
'ToolState is a state in a Tool''s state machine. It inherits its action from its superclass, and has several transition tables. One for each type of event that can be processed.

Instance Variables:
	doubleClick	<TransitionTable>	where to go to for a double click event
	keyPress	<TransitionTable>	...
	keyRelease	<TransitionTable>	
	mouseMove	<TransitionTable>	
	redButtonPress	<TransitionTable>	
	redButtonRelease	<TransitionTable>	
	yellowButtonPress	<TransitionTable>	
	yellowButtonRelease	<TransitionTable>	

'!


!ToolState methodsFor: 'accessing'!

doubleClick
	^doubleClick!

doubleClick: aTransitionTable 
	doubleClick := aTransitionTable!

keyPress
	^keyPress!

keyPress: aTransitionTable 
	keyPress := aTransitionTable!

keyRelease
	^keyRelease!

keyRelease: aTransitionTable 
	keyRelease := aTransitionTable!

mouseMove
	^mouseMove!

mouseMove: aTransitionTable 
	mouseMove := aTransitionTable!

name
	^name isNil 
		ifTrue: [self identityHash printString] 
		ifFalse: [name]!

redButtonPress
	^redButtonPress!

redButtonPress: aTransitionTable 
	redButtonPress := aTransitionTable!

redButtonRelease
	^redButtonRelease!

redButtonRelease: aTransitionTable 
	redButtonRelease := aTransitionTable!

yellowButtonPress
	^yellowButtonPress!

yellowButtonPress: aTransitionTable 
	yellowButtonPress := aTransitionTable!

yellowButtonRelease
	^yellowButtonRelease!

yellowButtonRelease: aTransitionTable 
	yellowButtonRelease := aTransitionTable! !

!ToolState methodsFor: 'events'!

blueButtonPressedEvent: aMouseButtonEvent 
	^nil!

blueButtonReleasedEvent: aMouseButtonEvent 
	^nil!

doubleClickEvent: event 
	^doubleClick!

enterEvent: anEvent
	^nil!

exitEvent: anEvent
	^nil!

keyPressedEvent: anEvent
	^keyPress!

keyReleasedEvent: anEvent
	^keyRelease!

mouseMovedEvent: aMouseMovedEvent 
	^mouseMove!

redButtonPressedEvent: aMouseButtonEvent 
	^redButtonPress!

redButtonReleasedEvent: aMouseButtonEvent 
	^redButtonRelease!

unknownEvent: anEvent
	^nil!

windowEnterEvent: anEvent
	^nil!

windowExitEvent: anEvent
	^nil!

yellowButtonPressedEvent: aMouseButtonEvent 
	^yellowButtonPress!

yellowButtonReleasedEvent: aMouseButtonEvent 
	^yellowButtonRelease! !

!ToolState methodsFor: 'testing'!

isEndState
	^false! !

!ToolState methodsFor: 'tool-accessing'!

nextStateForEvent: anEvent tool: aTool 
	| transitionTable |
	transitionTable := anEvent dispatchTo: self.
	transitionTable isNil ifTrue: [^nil].
	^transitionTable nextStateForTool: aTool event: anEvent! !

Object subclass: #TransitionTable
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
TransitionTable comment:
'TransitionTable is an abstract class that represents a transition between two states. When sent the nextStateForTool:event: message, it returns the next state for that type of event. If there isn''t a next state for that event, it returns nil.

Subclasses must implement the following messages:
	accessing
		goto:	a default constructer method that makes it easier to create the transition tables independently
		nextStateForTool:event:	returns the next state for the event or nil if there isn''t a next state for that event

'!


!TransitionTable methodsFor: 'accessing'!

goto: aState
	^self subclassResponsibility!

nextStateForTool: aTool event: anEvent
	^self subclassResponsibility! !

!TransitionTable methodsFor: 'initialize-release'!

initialize! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TransitionTable class
	instanceVariableNames: ''!


!TransitionTable class methodsFor: 'instance creation'!

new
	^super new initialize! !

TransitionTable subclass: #SimpleTransitionTable
	instanceVariableNames: 'nextState '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
SimpleTransitionTable comment:
'SimpleTransitionTable is a transition table that always goes to another state no matter what the event is.

Instance Variables:
	nextState	<EndToolState>	the next state to goto

'!


!SimpleTransitionTable methodsFor: 'accessing'!

goto: aState
	self nextState: aState!

nextStateForTool: aTool event: anEvent
	^nextState! !

!SimpleTransitionTable methodsFor: 'initialize-release'!

nextState: anObject
	nextState := anObject! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SimpleTransitionTable class
	instanceVariableNames: ''!


!SimpleTransitionTable class methodsFor: 'instance creation'!

goto: aToolState 
	^(self new) nextState: aToolState;
		yourself! !

TransitionTable subclass: #FigureTransitionTable
	instanceVariableNames: 'figureLookupStates '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
FigureTransitionTable comment:
'FigureTransitionTable is a transition table for mouse events. It checks what type of figure the mouse is over and then decides what is the next state.

Instance Variables:
	figureLookupStates	<Dictionary key: Figure class value: EndToolState>	our table, if the figure''s class is in the table, then we take it entry, otherwise we lookup its superclass

'!


!FigureTransitionTable methodsFor: 'accessing'!

goto: aState
	^self
		on: Figure 
		goto: aState!

nextStateForTool: aTool event: anEvent 
	| class state |
	class := (aTool figureAtEvent: anEvent) class.
	
	[state := figureLookupStates at: class ifAbsent: [nil].
	state notNil or: [class isNil]] 
			whileFalse: [class := class superclass].
	^state!

on: figureClass goto: aToolState 
	figureLookupStates at: figureClass put: aToolState! !

!FigureTransitionTable methodsFor: 'initialize-release'!

initialize
	super initialize.
	figureLookupStates := Dictionary new! !

TransitionTable subclass: #CharacterTransitionTable
	instanceVariableNames: 'characterMap '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
CharacterTransitionTable comment:
'CharacterTransitionTable is a transition table for characters. It decides our next state by looking at a table of characters (or symbols for special keys -- #F1).

Instance Variables:
	characterMap	<SequenceableCollection of: (Association key: String value: EndToolState)>	a mapping for characters to the next state

'!


!CharacterTransitionTable methodsFor: 'accessing'!

forCharacters: characterCollection goto: aToolState 
	characterMap add: characterCollection -> aToolState!

goto: aState 
	self
		forCharacters: ((0 to: 255) collect: [:each | Character value: each])
		goto: aState!

nextStateForTool: aTool event: anEvent 
	anEvent isKeyboard ifFalse: [^nil].
	1 to: characterMap size
		do: 
			[:i | 
			| each |
			each := characterMap at: i.
			(each key includes: anEvent keyValue) ifTrue: [^each value]].
	^nil! !

!CharacterTransitionTable methodsFor: 'initialize-release'!

initialize
	super initialize.
	characterMap := OrderedCollection new! !

Object subclass: #Tool
	instanceVariableNames: 'controller cursor currentState initialState passInputDown data figure '
	classVariableNames: 'Debug Grid States '
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
Tool comment:
'Tool are used by drawings to manipulate the figures. They represent different strategies for manipulation. They contain a state diagram that describes their actions.

Instance Variables:
	controller	<DrawingController>	the controller for the drawing
	currentState	<ToolState>	the state we''re currently in
	cursor	<Cursor>	the cursor that should be displayed
	data	<Dictionary>	data that can be passed from one state to another
	figure	<Figure>	the figure that the current event is for (this is used for efficiency)
	initialState	<ToolState>	the state that this tool initially starts in
	passInputDown	<Boolean>	should we pass input through to subviews or consume it ourselves

Class Variables:
	Debug	<Boolean>	when set to true, prints the state name to the transcript when it is enter
	States	<Dictionary key: String value: EndToolState>	our collection of tool states

'!


!Tool methodsFor: 'accessing'!

controller
	^controller!

controller: aController
	controller := aController!

cursor
	^cursor!

cursor: aCursor 
	cursor := aCursor.
	cursor show!

cursorPointFor: anEvent 
	| point |
	point := self primCursorPointFor: anEvent.
	^Grid isNil 
		ifTrue: [point]
		ifFalse: [point // Grid * Grid]!

drawing
	^controller view!

figureAtEvent: anEvent 
	^figure isNil
		ifTrue: [figure := self drawing figureAt: (self primCursorPointFor: anEvent)]
		ifFalse: [figure]!

figureUnderCursor
	^self drawing figureAt: self sensor cursorPoint!

graphicsContext
	^self drawing graphicsContext!

initialState
	^initialState!

sensor
	^controller sensor! !

!Tool methodsFor: 'controller accessing'!

passInputDown: aBoolean 
	passInputDown := aBoolean!

selected
	"The current tool has just been selected, evaluate the first state"

	self changeToState: initialState event: nil! !

!Tool methodsFor: 'events'!

changeToState: aToolState event: anEvent 
	aToolState isNil ifTrue: [^self].
	currentState := aToolState.
	Debug ifTrue: [self printDebuggingInformation].
	currentState == initialState ifTrue: [data := Dictionary new].
	[currentState evaluateIn: self event: anEvent] 
		valueOnUnwindDo: [currentState := initialState].	"Fail into the initialState"
	currentState isEndState 
		ifTrue: [self changeToState: initialState event: anEvent]!

handleEvent: anEvent 
	| nextState |
	figure := nil.
	nextState := currentState nextStateForEvent: anEvent tool: self.
	self changeToState: nextState event: anEvent.
	^nextState notNil! !

!Tool methodsFor: 'initialize-release'!

initialize
	passInputDown := true.
	data := Dictionary new.
	cursor := Cursor normal!

startState: aState 
	initialState := aState.
	self changeToState: initialState event: nil! !

!Tool methodsFor: 'private'!

primCursorPointFor: anEvent 
	^anEvent isMouseEvent 
		ifTrue: [self sensor mousePointFor: anEvent]
		ifFalse: [self sensor cursorPoint]!

printDebuggingInformation
	Transcript
		print: currentState;
		cr;
		flush! !

!Tool methodsFor: 'testing'!

isIdle
	^currentState == initialState!

passInputDown
	^passInputDown and: [self isIdle]! !

!Tool methodsFor: 'variable accessing'!

valueAt: aSymbol 
	^data at: aSymbol ifAbsent: [nil]!

valueAt: aSymbol put: anObject
	^data at: aSymbol put: anObject! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Tool class
	instanceVariableNames: ''!


!Tool class methodsFor: 'accessing'!

clearStates
	States := nil!

debug: aBoolean
	"self debug: true"
	"self debug: false"

	Debug := aBoolean!

grid: aPoint 
	"self grid: 8 @ 8"

	Grid := aPoint!

stateFor: aString 
	^States at: aString!

states
	^States! !

!Tool class methodsFor: 'class initialization'!

initialize
	"self initialize"

	Debug := false.
	Grid := nil.
	self initializeStates!

initializeStates
	| selectors |
	selectors := self 
				sortSelectors: (self class organization listAtCategoryNamed: #'tool states')
				initial: Dictionary new.
	States := Dictionary new.
	selectors do: [:each | self perform: each].
	States at: 'Hand Tool'
		put: (States at: 'Selection Tool' ifAbsent: [EndToolState new])!

reinitialize
	"Tool reinitialize"

	| oldStates |
	oldStates := States.
	[self initializeStates] valueOnUnwindDo: [States := oldStates]!

sortSelectors: selectors initial: aDictionary 
	| runnable |
	runnable := selectors select: 
					[:each | 
					States := aDictionary copy.
					Dictionary keyNotFoundSignal handle: [:ex | ex returnWith: false]
						do: 
							[self perform: each.
							true]].
	runnable isEmpty 
		ifTrue: [self error: 'Can''t initialize the tools, circular dependencies'].
	^runnable size == selectors size 
		ifTrue: [^selectors]
		ifFalse: 
			[States := aDictionary copy.
			runnable do: [:each | self perform: each].
			runnable 
				, (self sortSelectors: (selectors asOrderedCollection removeAll: runnable; yourself) initial: States)]! !

!Tool class methodsFor: 'instance creation'!

new
	^super new initialize!

selectionTool
	^self toolFor: 'Selection Tool'!

toolFor: aString 
	^(self new)
		startState: (States at: aString);
		yourself! !

!Tool class methodsFor: 'tool states'!

initializeBezierTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'End bezier line'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure diff |
figure := tool valueAt: #figure.
diff := (figure pointAt: 4) - (figure pointAt: 1).
figure pointAt: 2 put: diff / 3.0 + (figure pointAt: 1);
		pointAt: 3 put: diff * 2 / 3.0 + (figure pointAt: 1)]' runs: (RunArray runs: #(229) values: #(nil))); instVarAt: 4 put: (Point x: 266 y: 345); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Bezier Tool'; instVarAt: 3 put: (Text string: '[:tool :event | tool cursor: Cursor crossHair]' runs: (RunArray runs: #(46) values: #(nil))); instVarAt: 4 put: (Point x: 225 y: 20); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Pressed Bezier Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure point |
point := tool cursorPointFor: event.
figure := BezierFigure start: point stop: point.
tool valueAt: #figure put: figure.
tool drawing add: figure]' runs: (RunArray runs: #(180) values: #(nil))); instVarAt: 4 put: (Point x: 195 y: 121); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Move Bezier Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
(tool valueAt: #figure) stopPoint: (tool cursorPointFor: event)]' runs: (RunArray runs: #(81) values: #(nil))); instVarAt: 4 put: (Point x: 102 y: 218); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Cancel Figure Creation'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 310 y: 249); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 48 y: 99); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Bezier Tool'); to: (stateTable at: 'Pressed Bezier Tool'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 261 y: 51); add: (Point x: 258 y: 121); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Bezier Tool'); to: (stateTable at: 'End bezier line'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 174 y: 248); add: (Point x: 296 y: 346); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Pressed Bezier Tool'); to: (stateTable at: 'Move Bezier Tool'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 235 y: 151); add: (Point x: 171 y: 218); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Bezier Tool'); to: (stateTable at: 'Move Bezier Tool'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 102 y: 232); add: (Point x: 50 y: 249); add: (Point x: 45 y: 273); add: (Point x: 96 y: 273); add: (Point x: 132 y: 248); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Pressed Bezier Tool'); to: (stateTable at: 'Cancel Figure Creation'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 281 y: 151); add: (Point x: 368 y: 249); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Bezier Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 225 y: 36); add: (Point x: 82 y: 102); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'End bezier line' put: (EndToolState name: 'End bezier line' command: [:tool :event | 
| figure diff |
figure := tool valueAt: #figure.
diff := (figure pointAt: 4) - (figure pointAt: 1).
figure pointAt: 2 put: diff / 3.0 + (figure pointAt: 1);
		pointAt: 3 put: diff * 2 / 3.0 + (figure pointAt: 1)]).
	Tool states at: 'Bezier Tool' put: (ToolState name: 'Bezier Tool' command: [:tool :event | tool cursor: Cursor crossHair]).
	Tool states at: 'Pressed Bezier Tool' put: (ToolState name: 'Pressed Bezier Tool' command: [:tool :event | 
| figure point |
point := tool cursorPointFor: event.
figure := BezierFigure start: point stop: point.
tool valueAt: #figure put: figure.
tool drawing add: figure]).
	Tool states at: 'Move Bezier Tool' put: (ToolState name: 'Move Bezier Tool' command: [:tool :event | 
(tool valueAt: #figure) stopPoint: (tool cursorPointFor: event)]).
	(Tool stateFor: 'Move Bezier Tool') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'End bezier line'); yourself).
	(Tool stateFor: 'Move Bezier Tool') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Bezier Tool'); yourself).
	(Tool stateFor: 'Bezier Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Pressed Bezier Tool'); yourself).
	(Tool stateFor: 'Bezier Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Pressed Bezier Tool') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Cancel Figure Creation'); yourself).
	(Tool stateFor: 'Pressed Bezier Tool') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Bezier Tool'); yourself).!

initializeBringToFrontTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Bring To Front Tool'; instVarAt: 3 put: (Text string: '[:tool :event | tool cursor: Cursor crossHair]' runs: (RunArray runs: #(46) values: #(nil))); instVarAt: 4 put: (Point x: 193 y: 47); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Bring To Front Press'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure |
figure := tool figureAtEvent: event.
(figure isHandle or: [figure isDrawing])
	ifFalse: [tool drawing bringToFront: figure]]' runs: (RunArray runs: #(152) values: #(nil))); instVarAt: 4 put: (Point x: 192 y: 228); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 395 y: 165); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Bring To Front Tool'); to: (stateTable at: 'Bring To Front Press'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 253 y: 78); add: (Point x: 256 y: 228); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Bring To Front Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 283 y: 76); add: (Point x: 401 y: 168); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Bring To Front Tool' put: (ToolState name: 'Bring To Front Tool' command: [:tool :event | tool cursor: Cursor crossHair]).
	Tool states at: 'Bring To Front Press' put: (EndToolState name: 'Bring To Front Press' command: [:tool :event | 
| figure |
figure := tool figureAtEvent: event.
(figure isHandle or: [figure isDrawing])
	ifFalse: [tool drawing bringToFront: figure]]).
	(Tool stateFor: 'Bring To Front Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Bring To Front Press'); yourself).
	(Tool stateFor: 'Bring To Front Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeConnectionState
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Connection State'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure startPoint |
figure := (tool figureAtEvent: event) owner.
startPoint := tool cursorPointFor: event.
tool valueAt: #figure put: figure;
	valueAt: #startPoint put: startPoint;
	valueAt: #lastPoint put: startPoint.
(figure canConnectFromPoint: startPoint)
	ifFalse: [tool changeToState: tool initialState event: event]]' runs: (RunArray runs: #(342) values: #(nil))); instVarAt: 4 put: (Point x: 74 y: 35); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Connection Drag'; instVarAt: 3 put: (Text string: '[:tool :event | 
| currentPoint origin |
origin := tool valueAt: #startPoint.
tool drawing
	invalidateRectangle:
		(Rectangle vertex: origin vertex: (tool valueAt: #lastPoint))
	repairNow: true.
currentPoint := tool cursorPointFor: event.
(LineSegment from: currentPoint to: origin)
	displayStrokedOn: tool graphicsContext.
tool valueAt: #lastPoint put: currentPoint]' runs: (RunArray runs: #(367) values: #(nil))); instVarAt: 4 put: (Point x: 239 y: 142); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'End Connection'; instVarAt: 3 put: '[:tool :event | 
| endFigure startFigure origin lastPoint |
origin := tool valueAt: #startPoint.
lastPoint := tool valueAt: #lastPoint.
tool drawing
	invalidateRectangle: 
		(Rectangle vertex: origin vertex: (tool valueAt: #lastPoint))
	repairNow: true.
endFigure := (tool figureAtEvent: event) owner.
startFigure := tool valueAt: #figure.
(startFigure canConnectFromPoint: origin
				to: endFigure
				at: lastPoint) ifTrue: 
			[tool drawing add: (startFigure connectFromPoint: origin
				to: endFigure
				at: lastPoint)]]'; instVarAt: 4 put: (Point x: 68 y: 256); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Connection State'); to: (stateTable at: 'Connection Drag'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 154.898 y: 64.6094); add: (Point x: 268.109 y: 141.395); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Connection Drag'); to: (stateTable at: 'Connection Drag'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 337.937 y: 147.828); add: (Point x: 397 y: 137); add: (Point x: 393 y: 187); add: (Point x: 333.938 y: 168.297); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Connection Drag'); to: (stateTable at: 'End Connection'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 263.016 y: 171.672); add: (Point x: 138.977 y: 256.333); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Connection State'); to: (stateTable at: 'End Connection'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 126.969 y: 66.7422); add: (Point x: 119.031 y: 255.261); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Connection State' put: (ToolState name: 'Connection State' command: [:tool :event | 
| figure startPoint |
figure := (tool figureAtEvent: event) owner.
startPoint := tool cursorPointFor: event.
tool valueAt: #figure put: figure;
	valueAt: #startPoint put: startPoint;
	valueAt: #lastPoint put: startPoint.
(figure canConnectFromPoint: startPoint)
	ifFalse: [tool changeToState: tool initialState event: event]]).
	Tool states at: 'Connection Drag' put: (ToolState name: 'Connection Drag' command: [:tool :event | 
| currentPoint origin |
origin := tool valueAt: #startPoint.
tool drawing
	invalidateRectangle:
		(Rectangle vertex: origin vertex: (tool valueAt: #lastPoint))
	repairNow: true.
currentPoint := tool cursorPointFor: event.
(LineSegment from: currentPoint to: origin)
	displayStrokedOn: tool graphicsContext.
tool valueAt: #lastPoint put: currentPoint]).
	Tool states at: 'End Connection' put: (EndToolState name: 'End Connection' command: [:tool :event | 
| endFigure startFigure origin lastPoint |
origin := tool valueAt: #startPoint.
lastPoint := tool valueAt: #lastPoint.
tool drawing
	invalidateRectangle: 
		(Rectangle vertex: origin vertex: (tool valueAt: #lastPoint))
	repairNow: true.
endFigure := (tool figureAtEvent: event) owner.
startFigure := tool valueAt: #figure.
(startFigure canConnectFromPoint: origin
				to: endFigure
				at: lastPoint) ifTrue: 
			[tool drawing add: (startFigure connectFromPoint: origin
				to: endFigure
				at: lastPoint)]]).
	(Tool stateFor: 'Connection Drag') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Connection Drag'); yourself).
	(Tool stateFor: 'Connection Drag') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'End Connection'); yourself).
	(Tool stateFor: 'Connection State') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Connection Drag'); yourself).
	(Tool stateFor: 'Connection State') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'End Connection'); yourself).!

initializeDeleteTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Delete Tool'; instVarAt: 3 put: (Text string: '[:tool :event | tool cursor: Cursor crossHair]' runs: (RunArray runs: #(46) values: #(nil))); instVarAt: 4 put: (Point x: 242 y: 63); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Delete Figure Under Cursor'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure |
figure := tool figureAtEvent: event.
(figure isHandle or: [figure isDrawing])
	ifFalse: [tool drawing remove: figure]]' runs: (RunArray runs: #(146) values: #(nil))); instVarAt: 4 put: (Point x: 170 y: 224); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 370 y: 158); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Delete Tool'); to: (stateTable at: 'Delete Figure Under Cursor'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 275 y: 94); add: (Point x: 252 y: 224); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Delete Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 296 y: 92); add: (Point x: 376 y: 162); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Delete Tool' put: (ToolState name: 'Delete Tool' command: [:tool :event | tool cursor: Cursor crossHair]).
	Tool states at: 'Delete Figure Under Cursor' put: (EndToolState name: 'Delete Figure Under Cursor' command: [:tool :event | 
| figure |
figure := tool figureAtEvent: event.
(figure isHandle or: [figure isDrawing])
	ifFalse: [tool drawing remove: figure]]).
	(Tool stateFor: 'Delete Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Delete Figure Under Cursor'); yourself).
	(Tool stateFor: 'Delete Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeImageTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Image Tool'; instVarAt: 3 put: (Text string: '[:tool :event | tool cursor: Cursor origin]' runs: (RunArray runs: #(43) values: #(nil))); instVarAt: 4 put: (Point x: 214 y: 50); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Create Image'; instVarAt: 3 put: (Text string: '[:tool :event | tool drawing add: (ImageFigure createAt: (tool cursorPointFor: event))]' runs: (RunArray runs: #(87) values: #(nil))); instVarAt: 4 put: (Point x: 205 y: 185); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 382 y: 122); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Image Tool'); to: (stateTable at: 'Create Image'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 249 y: 81); add: (Point x: 248 y: 185); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Image Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 276 y: 77); add: (Point x: 385 y: 128); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Image Tool' put: (ToolState name: 'Image Tool' command: [:tool :event | tool cursor: Cursor origin]).
	Tool states at: 'Create Image' put: (EndToolState name: 'Create Image' command: [:tool :event | tool drawing add: (ImageFigure createAt: (tool cursorPointFor: event))]).
	(Tool stateFor: 'Image Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Create Image'); yourself).
	(Tool stateFor: 'Image Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeLineTools
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Polyline Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool valueAt: #class put: PolylineFigure.
tool cursor: Cursor crossHair]' runs: (RunArray runs: #(89) values: #(nil))); instVarAt: 4 put: (Point x: 34 y: 18); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Pressed Line Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure point |
point := tool cursorPointFor: event.
figure := (tool valueAt: #class) createAt: point.
tool valueAt: #figure put: figure.
tool drawing add: figure]' runs: (RunArray runs: #(181) values: #(nil))); instVarAt: 4 put: (Point x: 110 y: 129); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Move Line'; instVarAt: 3 put: (Text string: '[:tool :event | 
(tool valueAt: #figure) stopPoint: (tool cursorPointFor: event)]' runs: (RunArray runs: #(81) values: #(nil))); instVarAt: 4 put: (Point x: 134 y: 255); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Line Tool End State'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 31 y: 364); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Add Point to Line (Tool)'; instVarAt: 3 put: (Text string: '[:tool :event | 
(tool valueAt: #figure) addPoint: (tool cursorPointFor: event)]' runs: (RunArray runs: #(80) values: #(nil))); instVarAt: 4 put: (Point x: 324 y: 341); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Spline Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool valueAt: #class put: SplineFigure.
tool cursor: Cursor crossHair]' runs: (RunArray runs: #(87) values: #(nil))); instVarAt: 4 put: (Point x: 240 y: 19); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 418 y: 97); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Cancel Figure Creation'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 384 y: 151); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Polyline Tool'); to: (stateTable at: 'Pressed Line Tool'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 87 y: 48); add: (Point x: 154 y: 129); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Pressed Line Tool'); to: (stateTable at: 'Move Line'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 167 y: 160); add: (Point x: 167 y: 255); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Pressed Line Tool'); to: (stateTable at: 'Move Line'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 191 y: 158); add: (Point x: 247 y: 193); add: (Point x: 182 y: 255); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Line'); to: (stateTable at: 'Move Line'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 186 y: 256); add: (Point x: 234 y: 225); add: (Point x: 259 y: 242); add: (Point x: 196 y: 260); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Line'); to: (stateTable at: 'Move Line'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 142 y: 260); add: (Point x: 41 y: 222); add: (Point x: 24 y: 265); add: (Point x: 134 y: 268); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Line'); to: (stateTable at: 'Line Tool End State'); type: (TransitionType basicNew instVarAt: 1 put: #doubleClick; yourself); points: ((OrderedCollection new) add: (Point x: 144 y: 282); add: (Point x: 96 y: 315); add: (Point x: 93 y: 364); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Line'); to: (stateTable at: 'Line Tool End State'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 156 y: 285); add: (Point x: 102 y: 364); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Line'); to: (stateTable at: 'Add Point to Line (Tool)'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 196 y: 280); add: (Point x: 361 y: 342); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Add Point to Line (Tool)'); to: (stateTable at: 'Move Line'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 324 y: 357); add: (Point x: 246 y: 361); add: (Point x: 181 y: 285); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Add Point to Line (Tool)'); to: (stateTable at: 'Move Line'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 391 y: 341); add: (Point x: 334 y: 284); add: (Point x: 202 y: 272); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Spline Tool'); to: (stateTable at: 'Pressed Line Tool'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 257 y: 48); add: (Point x: 181 y: 129); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Spline Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 297 y: 48); add: (Point x: 421 y: 104); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Polyline Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 116 y: 27); add: (Point x: 254 y: 10); add: (Point x: 404 y: 23); add: (Point x: 421 y: 69); add: (Point x: 432 y: 97); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Add Point to Line (Tool)'); to: (stateTable at: 'Cancel Figure Creation'); type: (CharacterTransitionType new event: #keyPress; characters: (String fromIntegerArray: #[27])); points: ((OrderedCollection new) add: (Point x: 400 y: 341); add: (Point x: 448 y: 182); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Move Line'); to: (stateTable at: 'Cancel Figure Creation'); type: (CharacterTransitionType new event: #keyPress; characters: (String fromIntegerArray: #[27])); points: ((OrderedCollection new) add: (Point x: 202 y: 268); add: (Point x: 366 y: 214); add: (Point x: 407 y: 178); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Pressed Line Tool'); to: (stateTable at: 'Cancel Figure Creation'); type: (CharacterTransitionType new event: #keyPress; characters: (String fromIntegerArray: #[27])); points: ((OrderedCollection new) add: (Point x: 224 y: 147); add: (Point x: 261 y: 149); add: (Point x: 389 y: 159); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Polyline Tool' put: (ToolState name: 'Polyline Tool' command: [:tool :event | 
tool valueAt: #class put: PolylineFigure.
tool cursor: Cursor crossHair]).
	Tool states at: 'Pressed Line Tool' put: (ToolState name: 'Pressed Line Tool' command: [:tool :event | 
| figure point |
point := tool cursorPointFor: event.
figure := (tool valueAt: #class) createAt: point.
tool valueAt: #figure put: figure.
tool drawing add: figure]).
	Tool states at: 'Move Line' put: (ToolState name: 'Move Line' command: [:tool :event | 
(tool valueAt: #figure) stopPoint: (tool cursorPointFor: event)]).
	Tool states at: 'Line Tool End State' put: (EndToolState name: 'Line Tool End State' command: [:tool :event | ]).
	Tool states at: 'Add Point to Line (Tool)' put: (ToolState name: 'Add Point to Line (Tool)' command: [:tool :event | 
(tool valueAt: #figure) addPoint: (tool cursorPointFor: event)]).
	Tool states at: 'Spline Tool' put: (ToolState name: 'Spline Tool' command: [:tool :event | 
tool valueAt: #class put: SplineFigure.
tool cursor: Cursor crossHair]).
	(Tool stateFor: 'Polyline Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Pressed Line Tool'); yourself).
	(Tool stateFor: 'Polyline Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Add Point to Line (Tool)') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Line'); yourself).
	(Tool stateFor: 'Add Point to Line (Tool)') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Line'); yourself).
	(Tool stateFor: 'Add Point to Line (Tool)') keyPress: ((CharacterTransitionTable new) forCharacters: (String fromIntegerArray: #[27]) goto: (Tool stateFor: 'Cancel Figure Creation'); yourself).
	(Tool stateFor: 'Pressed Line Tool') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Line'); yourself).
	(Tool stateFor: 'Pressed Line Tool') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Line'); yourself).
	(Tool stateFor: 'Pressed Line Tool') keyPress: ((CharacterTransitionTable new) forCharacters: (String fromIntegerArray: #[27]) goto: (Tool stateFor: 'Cancel Figure Creation'); yourself).
	(Tool stateFor: 'Move Line') doubleClick: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Line Tool End State'); yourself).
	(Tool stateFor: 'Move Line') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Line'); yourself).
	(Tool stateFor: 'Move Line') keyPress: ((CharacterTransitionTable new) forCharacters: (String fromIntegerArray: #[27]) goto: (Tool stateFor: 'Cancel Figure Creation'); yourself).
	(Tool stateFor: 'Move Line') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Line Tool End State'); yourself).
	(Tool stateFor: 'Move Line') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Add Point to Line (Tool)'); yourself).
	(Tool stateFor: 'Move Line') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Move Line'); yourself).
	(Tool stateFor: 'Spline Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Pressed Line Tool'); yourself).
	(Tool stateFor: 'Spline Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeOriginCornerCreationTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Create Origin/Corner Figure'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure |
tool cursor: Cursor corner.
figure := (tool valueAt: #class) createAt: (tool cursorPointFor: event).
tool valueAt: #figure put: figure.
tool drawing add: figure.
tool valueAt: #moveBlock put: [:aPoint | figure bottomRight: aPoint]]' runs: (RunArray runs: #(259) values: #(nil))); instVarAt: 4 put: (Point x: 139 y: 147); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Track handle move'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 104 y: 278); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Cancel Figure Creation'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool drawing remove: (tool valueAt: #figure)]' runs: (RunArray runs: #(62) values: #(nil))); instVarAt: 4 put: (Point x: 298 y: 268); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Ellipse Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool valueAt: #class put: EllipseFigure.
tool cursor: Cursor origin]' runs: (RunArray runs: #(85) values: #(nil))); instVarAt: 4 put: (Point x: 51 y: 65); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Rectangle Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool valueAt: #class put: RectangleFigure.
tool cursor: Cursor origin]' runs: (RunArray runs: #(87) values: #(nil))); instVarAt: 4 put: (Point x: 394 y: 53); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 239 y: 375); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Rounded Rectangle Tool'; instVarAt: 3 put: '[:tool :event | 
tool valueAt: #class put: RoundedRectangleFigure.
tool cursor: Cursor origin]'; instVarAt: 4 put: (Point x: 114 y: 15); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Arc Tool'; instVarAt: 3 put: '[:tool :event | 
tool valueAt: #class put: ArcFigure.
tool cursor: Cursor origin]'; instVarAt: 4 put: (Point x: 295 y: 18); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Create Origin/Corner Figure'); to: (stateTable at: 'Track handle move'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 215 y: 177); add: (Point x: 169 y: 278); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Create Origin/Corner Figure'); to: (stateTable at: 'Cancel Figure Creation'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 242 y: 177); add: (Point x: 352 y: 268); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Ellipse Tool'); to: (stateTable at: 'Create Origin/Corner Figure'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 114 y: 93); add: (Point x: 200 y: 147); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Rectangle Tool'); to: (stateTable at: 'Create Origin/Corner Figure'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 408 y: 80); add: (Point x: 254 y: 147); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Rectangle Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 442 y: 84); add: (Point x: 453 y: 323); add: (Point x: 371 y: 355); add: (Point x: 280 y: 383); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Ellipse Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 86 y: 96); add: (Point x: 67 y: 330); add: (Point x: 240 y: 383); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Rounded Rectangle Tool'); to: (stateTable at: 'Create Origin/Corner Figure'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 198 y: 45); add: (Point x: 231 y: 147); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Rounded Rectangle Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 115 y: 34); add: (Point x: 30 y: 35); add: (Point x: 12 y: 374); add: (Point x: 239 y: 388); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Arc Tool'); to: (stateTable at: 'Create Origin/Corner Figure'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 320 y: 49); add: (Point x: 236 y: 147); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Arc Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 353 y: 30); add: (Point x: 393 y: 26); add: (Point x: 413 y: 25); add: (Point x: 436 y: 25); add: (Point x: 493 y: 26); add: (Point x: 495 y: 209); add: (Point x: 470 y: 372); add: (Point x: 281 y: 391); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Create Origin/Corner Figure' put: (ToolState name: 'Create Origin/Corner Figure' command: [:tool :event | 
| figure |
tool cursor: Cursor corner.
figure := (tool valueAt: #class) createAt: (tool cursorPointFor: event).
tool valueAt: #figure put: figure.
tool drawing add: figure.
tool valueAt: #moveBlock put: [:aPoint | figure bottomRight: aPoint]]).
	Tool states at: 'Cancel Figure Creation' put: (EndToolState name: 'Cancel Figure Creation' command: [:tool :event | 
tool drawing remove: (tool valueAt: #figure)]).
	Tool states at: 'Ellipse Tool' put: (ToolState name: 'Ellipse Tool' command: [:tool :event | 
tool valueAt: #class put: EllipseFigure.
tool cursor: Cursor origin]).
	Tool states at: 'Rectangle Tool' put: (ToolState name: 'Rectangle Tool' command: [:tool :event | 
tool valueAt: #class put: RectangleFigure.
tool cursor: Cursor origin]).
	Tool states at: 'Rounded Rectangle Tool' put: (ToolState name: 'Rounded Rectangle Tool' command: [:tool :event | 
tool valueAt: #class put: RoundedRectangleFigure.
tool cursor: Cursor origin]).
	Tool states at: 'Arc Tool' put: (ToolState name: 'Arc Tool' command: [:tool :event | 
tool valueAt: #class put: ArcFigure.
tool cursor: Cursor origin]).
	(Tool stateFor: 'Arc Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Create Origin/Corner Figure'); yourself).
	(Tool stateFor: 'Arc Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Rectangle Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Create Origin/Corner Figure'); yourself).
	(Tool stateFor: 'Rectangle Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Create Origin/Corner Figure') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Cancel Figure Creation'); yourself).
	(Tool stateFor: 'Create Origin/Corner Figure') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Track handle move'); yourself).
	(Tool stateFor: 'Rounded Rectangle Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Create Origin/Corner Figure'); yourself).
	(Tool stateFor: 'Rounded Rectangle Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Ellipse Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Create Origin/Corner Figure'); yourself).
	(Tool stateFor: 'Ellipse Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeSelectionToolMultiseletion
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool Multi-select'; instVarAt: 3 put: (Text string: '[:tool :event | 
| point |
point := tool cursorPointFor: event.
tool valueAt: #initialPoint put: point;
	valueAt: #lastPoint put: point]' runs: (RunArray runs: #(136) values: #(nil))); instVarAt: 4 put: (Point x: 16 y: 45); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'multi-selection move'; instVarAt: 3 put: (Text string: '[:tool :event | 
| gc newPoint origin |
origin := tool valueAt: #initialPoint.
tool drawing
	invalidateRectangle: (Rectangle vertex: origin vertex: (tool valueAt: #lastPoint))
	repairNow: true.
gc := tool drawing graphicsContext.
newPoint := tool cursorPointFor: event.
gc displayRectangularBorder: (Rectangle vertex: origin vertex: newPoint).
tool valueAt: #lastPoint put: newPoint]' runs: (RunArray runs: #(383) values: #(nil))); instVarAt: 4 put: (Point x: 135 y: 156); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'multi-selection release'; instVarAt: 3 put: (Text string: '[:tool :event | 
| origin lastPoint rect figures |
origin := tool valueAt: #initialPoint.
lastPoint := tool valueAt: #lastPoint.
rect := Rectangle vertex: origin vertex: lastPoint.
tool drawing invalidateRectangle: rect.
figures := tool drawing figuresIn: rect.
tool sensor shiftDown 
	ifTrue: [tool drawing toggleSelections: figures] 
	ifFalse: [tool drawing selections: figures]]' runs: (RunArray runs: #(381) values: #(nil))); instVarAt: 4 put: (Point x: 35 y: 332); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool Multi-select'); to: (stateTable at: 'multi-selection move'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 109 y: 75); add: (Point x: 184 y: 156); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'multi-selection move'); to: (stateTable at: 'multi-selection move'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 245 y: 160); add: (Point x: 323 y: 143); add: (Point x: 323 y: 163); add: (Point x: 261 y: 167); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'multi-selection move'); to: (stateTable at: 'multi-selection release'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 190 y: 186); add: (Point x: 111 y: 332); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool Multi-select'); to: (stateTable at: 'multi-selection release'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 97 y: 76); add: (Point x: 104 y: 332); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Selection Tool Multi-select' put: (ToolState name: 'Selection Tool Multi-select' command: [:tool :event | 
| point |
point := tool cursorPointFor: event.
tool valueAt: #initialPoint put: point;
	valueAt: #lastPoint put: point]).
	Tool states at: 'multi-selection move' put: (ToolState name: 'multi-selection move' command: [:tool :event | 
| gc newPoint origin |
origin := tool valueAt: #initialPoint.
tool drawing
	invalidateRectangle: (Rectangle vertex: origin vertex: (tool valueAt: #lastPoint))
	repairNow: true.
gc := tool drawing graphicsContext.
newPoint := tool cursorPointFor: event.
gc displayRectangularBorder: (Rectangle vertex: origin vertex: newPoint).
tool valueAt: #lastPoint put: newPoint]).
	Tool states at: 'multi-selection release' put: (EndToolState name: 'multi-selection release' command: [:tool :event | 
| origin lastPoint rect figures |
origin := tool valueAt: #initialPoint.
lastPoint := tool valueAt: #lastPoint.
rect := Rectangle vertex: origin vertex: lastPoint.
tool drawing invalidateRectangle: rect.
figures := tool drawing figuresIn: rect.
tool sensor shiftDown 
	ifTrue: [tool drawing toggleSelections: figures] 
	ifFalse: [tool drawing selections: figures]]).
	(Tool stateFor: 'Selection Tool Multi-select') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'multi-selection release'); yourself).
	(Tool stateFor: 'Selection Tool Multi-select') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'multi-selection move'); yourself).
	(Tool stateFor: 'multi-selection move') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'multi-selection release'); yourself).
	(Tool stateFor: 'multi-selection move') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'multi-selection move'); yourself).!

initializeSelectionToolSelection
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool Select'; instVarAt: 3 put: (Text string: '[:tool :event | 
| drawing lastPoint figure |
drawing := tool drawing.
lastPoint := tool cursorPointFor: event.
tool valueAt: #lastPoint put: lastPoint.
figure := drawing figureAt: lastPoint.
tool sensor shiftDown
	ifTrue: [drawing toggleSelection: figure]
	ifFalse: [(drawing isSelected: figure) ifFalse: [drawing selection: figure]]]' runs: (RunArray runs: #(335) values: #(nil))); instVarAt: 4 put: (Point x: 46 y: 51); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool Move Figure'; instVarAt: 3 put: (Text string: '[:tool :event | 
| delta newPoint |
newPoint := tool cursorPointFor: event.
delta := newPoint - (tool valueAt: #lastPoint).
tool valueAt: #lastPoint put: newPoint.
tool drawing selections do: [:each | each translateBy: delta]]' runs: (RunArray runs: #(226) values: #(nil))); instVarAt: 4 put: (Point x: 133 y: 140); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'End State'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 55 y: 253); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool Select'); to: (stateTable at: 'Selection Tool Move Figure'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 137 y: 81); add: (Point x: 222 y: 140); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool Move Figure'); to: (stateTable at: 'Selection Tool Move Figure'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 252 y: 141); add: (Point x: 358 y: 131); add: (Point x: 358 y: 151); add: (Point x: 299 y: 154); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool Move Figure'); to: (stateTable at: 'End State'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 209 y: 171); add: (Point x: 104 y: 253); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool Select'); to: (stateTable at: 'End State'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 117 y: 82); add: (Point x: 90 y: 253); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Selection Tool Select' put: (ToolState name: 'Selection Tool Select' command: [:tool :event | 
| drawing lastPoint figure |
drawing := tool drawing.
lastPoint := tool cursorPointFor: event.
tool valueAt: #lastPoint put: lastPoint.
figure := drawing figureAt: lastPoint.
tool sensor shiftDown
	ifTrue: [drawing toggleSelection: figure]
	ifFalse: [(drawing isSelected: figure) ifFalse: [drawing selection: figure]]]).
	Tool states at: 'Selection Tool Move Figure' put: (ToolState name: 'Selection Tool Move Figure' command: [:tool :event | 
| delta newPoint |
newPoint := tool cursorPointFor: event.
delta := newPoint - (tool valueAt: #lastPoint).
tool valueAt: #lastPoint put: newPoint.
tool drawing selections do: [:each | each translateBy: delta]]).
	Tool states at: 'End State' put: (EndToolState name: 'End State' command: [:tool :event | ]).
	(Tool stateFor: 'Selection Tool Select') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'End State'); yourself).
	(Tool stateFor: 'Selection Tool Select') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Selection Tool Move Figure'); yourself).
	(Tool stateFor: 'Selection Tool Move Figure') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'End State'); yourself).
	(Tool stateFor: 'Selection Tool Move Figure') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Selection Tool Move Figure'); yourself).!

initializeSelectionToolStates
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool'; instVarAt: 3 put: (Text string: '[:tool :event | tool cursor: Cursor normal]' runs: (RunArray runs: #(43) values: #(nil))); instVarAt: 4 put: (Point x: 198 y: 21); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool Select'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 26 y: 140); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool controller processMenuAt: event globalPoint
	local: (tool cursorPointFor: event)
	for: (tool figureAtEvent: event)]' runs: (RunArray runs: #(137) values: #(nil))); instVarAt: 4 put: (Point x: 109 y: 261); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool Multi-select'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 226 y: 334); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Handle Processing'; instVarAt: 3 put: (Text string: '[:tool :event | tool changeToState: (tool figureAtEvent: event) toolState event: event]' runs: (RunArray runs: #(87) values: #(nil))); instVarAt: 4 put: (Point x: 411 y: 253); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Selection Tool Remove Figures'; instVarAt: 3 put: (Text string: '[:tool :event | tool drawing removeAll: (tool drawing selections copy)]' runs: (RunArray runs: #(71) values: #(nil))); instVarAt: 4 put: (Point x: 345 y: 111); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool'); to: (stateTable at: 'Selection Tool Select'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 221 y: 50); add: (Point x: 109 y: 140); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 236 y: 51); add: (Point x: 136 y: 261); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool'); to: (stateTable at: 'Selection Tool Multi-select'); type: (FigureTransitionType basicNew instVarAt: 1 put: #redButtonPress; instVarAt: 2 put: ((Array new: 1) at: 1 put: Drawing; yourself); yourself); points: ((OrderedCollection new) add: (Point x: 246 y: 52); add: (Point x: 304 y: 334); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool'); to: (stateTable at: 'Handle Processing'); type: (FigureTransitionType basicNew instVarAt: 1 put: #redButtonPress; instVarAt: 2 put: ((Array new: 1) at: 1 put: Handle; yourself); yourself); points: ((OrderedCollection new) add: (Point x: 259 y: 51); add: (Point x: 455 y: 253); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Selection Tool'); to: (stateTable at: 'Selection Tool Remove Figures'); type: (CharacterTransitionType new event: #keyPress; characters: (String fromIntegerArray: #[8 127])); points: ((OrderedCollection new) add: (Point x: 290 y: 39); add: (Point x: 364 y: 39); add: (Point x: 409 y: 111); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Selection Tool' put: (ToolState name: 'Selection Tool' command: [:tool :event | tool cursor: Cursor normal]).
	Tool states at: 'Menu' put: (EndToolState name: 'Menu' command: [:tool :event | 
tool controller processMenuAt: event globalPoint
	local: (tool cursorPointFor: event)
	for: (tool figureAtEvent: event)]).
	Tool states at: 'Handle Processing' put: (ToolState name: 'Handle Processing' command: [:tool :event | tool changeToState: (tool figureAtEvent: event) toolState event: event]).
	Tool states at: 'Selection Tool Remove Figures' put: (EndToolState name: 'Selection Tool Remove Figures' command: [:tool :event | tool drawing removeAll: (tool drawing selections copy)]).
	(Tool stateFor: 'Selection Tool') keyPress: ((CharacterTransitionTable new) forCharacters: (String fromIntegerArray: #[8 127]) goto: (Tool stateFor: 'Selection Tool Remove Figures'); yourself).
	(Tool stateFor: 'Selection Tool') redButtonPress: ((FigureTransitionTable new)  on: Handle goto: (Tool stateFor: 'Handle Processing');  on: Drawing goto: (Tool stateFor: 'Selection Tool Multi-select'); goto: (Tool stateFor: 'Selection Tool Select'); yourself).
	(Tool stateFor: 'Selection Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeSendToBackTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Send To Back Tool'; instVarAt: 3 put: (Text string: '[:tool :event | tool cursor: Cursor crossHair]' runs: (RunArray runs: #(46) values: #(nil))); instVarAt: 4 put: (Point x: 171 y: 50); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Send To Back Press'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure |
figure := tool figureAtEvent: event.
(figure isHandle or: [figure isDrawing])
	ifFalse: [tool drawing sendToBack: figure]]' runs: (RunArray runs: #(150) values: #(nil))); instVarAt: 4 put: (Point x: 100 y: 189); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 362 y: 191); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Send To Back Tool'); to: (stateTable at: 'Send To Back Press'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 232 y: 81); add: (Point x: 172 y: 189); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Send To Back Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 271 y: 78); add: (Point x: 371 y: 193); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Send To Back Tool' put: (ToolState name: 'Send To Back Tool' command: [:tool :event | tool cursor: Cursor crossHair]).
	Tool states at: 'Send To Back Press' put: (EndToolState name: 'Send To Back Press' command: [:tool :event | 
| figure |
figure := tool figureAtEvent: event.
(figure isHandle or: [figure isDrawing])
	ifFalse: [tool drawing sendToBack: figure]]).
	(Tool stateFor: 'Send To Back Tool') redButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Send To Back Press'); yourself).
	(Tool stateFor: 'Send To Back Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).!

initializeTextTool
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Text Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool valueAt: #canCreate put: false]' runs: (RunArray runs: #(53) values: #(nil))); instVarAt: 4 put: (Point x: 87 y: 41); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Select Text Figure'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure startIndex |
figure := tool figureAtEvent: event.
tool drawing selection: figure.
tool valueAt: #figure put: figure.
startIndex := figure indexForPoint: (tool cursorPointFor: event).
tool valueAt: #startIndex put: startIndex.
figure setIndices: (startIndex to: startIndex)]' runs: (RunArray runs: #(299) values: #(nil))); instVarAt: 4 put: (Point x: 60 y: 173); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Highlight Text'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure newIndex startIndex |
figure := tool valueAt: #figure.
startIndex := tool valueAt: #startIndex.
newIndex := figure indexForPoint: (tool cursorPointFor: event).
figure setIndices: 
	((newIndex min: startIndex) to: (newIndex max: startIndex))]' runs: (RunArray runs: #(267) values: #(nil))); instVarAt: 4 put: (Point x: 59 y: 243); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Text Figure Selected'; instVarAt: 3 put: (Text string: '[:tool :event | ]' runs: (RunArray runs: #(17) values: #(nil))); instVarAt: 4 put: (Point x: 276 y: 242); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Text Figure Creation Tool'; instVarAt: 3 put: (Text string: '[:tool :event | 
tool valueAt: #canCreate put: true]' runs: (RunArray runs: #(52) values: #(nil))); instVarAt: 4 put: (Point x: 273 y: 27); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Create Text'; instVarAt: 3 put: (Text string: '[:tool :event | 
| figure |
(tool valueAt: #canCreate) 
	ifTrue: 
		[figure := TextFigure string: '''' 
				at: (tool cursorPointFor: event).
		figure setIndices: (1 to: 1).
		tool drawing add: figure; selection: figure.
		tool valueAt: #figure put: figure] 
	ifFalse: 
		[tool changeToState: (Tool stateFor: ''End State'') 
			event: event]]' runs: (RunArray runs: #(276 61 1) values: #(nil #bold nil))); instVarAt: 4 put: (Point x: 375 y: 109); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Edit Text Figure'; instVarAt: 3 put: (Text string: '[:tool :event | 
| character figure offset replacement startIndex |
figure := tool valueAt: #figure.
character := event keyValue.
offset := 0.
(character == Character backspace or: [character == Character del])
	ifTrue: [figure selectedIndices size = 1 ifTrue: [offset := -1].
		replacement := Text new]
	ifFalse: [replacement := (String with: character) asText].
startIndex := figure selectedIndices first.
figure replaceFrom: (startIndex + offset max: 1)
	to: figure selectedIndices last - 1
	with: replacement.
startIndex := startIndex + replacement size + offset max: 1.
figure setIndices: (startIndex to: startIndex)
]' runs: (RunArray runs: #(623) values: #(nil))); instVarAt: 4 put: (Point x: 303 y: 312); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Menu'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 316 y: 369); instVarAt: 5 put: false; instVarAt: 6 put: true; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Tool'); to: (stateTable at: 'Select Text Figure'); type: (FigureTransitionType basicNew instVarAt: 1 put: #redButtonPress; instVarAt: 2 put: ((Array new: 1) at: 1 put: TextFigure; yourself); yourself); points: ((OrderedCollection new) add: (Point x: 117 y: 72); add: (Point x: 117 y: 173); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Select Text Figure'); to: (stateTable at: 'Highlight Text'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 114 y: 204); add: (Point x: 105 y: 243); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Highlight Text'); to: (stateTable at: 'Highlight Text'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 107 y: 274); add: (Point x: 117 y: 311); add: (Point x: 90 y: 318); add: (Point x: 88 y: 273); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Highlight Text'); to: (stateTable at: 'Text Figure Selected'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 148 y: 256); add: (Point x: 172 y: 256); add: (Point x: 276 y: 256); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Select Text Figure'); to: (stateTable at: 'Text Figure Selected'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 171 y: 195); add: (Point x: 200 y: 206); add: (Point x: 305 y: 244); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Selected'); to: (stateTable at: 'Select Text Figure'); type: (FigureTransitionType basicNew instVarAt: 1 put: #redButtonPress; instVarAt: 2 put: ((Array new: 1) at: 1 put: TextFigure; yourself); yourself); points: ((OrderedCollection new) add: (Point x: 327 y: 242); add: (Point x: 260 y: 161); add: (Point x: 176 y: 185); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Creation Tool'); to: (stateTable at: 'Create Text'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 361 y: 57); add: (Point x: 401 y: 109); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Creation Tool'); to: (stateTable at: 'Select Text Figure'); type: (FigureTransitionType basicNew instVarAt: 1 put: #redButtonPress; instVarAt: 2 put: ((Array new: 1) at: 1 put: TextFigure; yourself); yourself); points: ((OrderedCollection new) add: (Point x: 319 y: 57); add: (Point x: 139 y: 173); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Create Text'); to: (stateTable at: 'Text Figure Selected'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 404 y: 139); add: (Point x: 348 y: 242); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Selected'); to: (stateTable at: 'Create Text'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 383 y: 245); add: (Point x: 422 y: 165); add: (Point x: 420 y: 139); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Edit Text Figure'); to: (stateTable at: 'Edit Text Figure'); type: (TransitionType basicNew instVarAt: 1 put: #keyPress; yourself); points: ((OrderedCollection new) add: (Point x: 405 y: 327); add: (Point x: 425 y: 331); add: (Point x: 425 y: 351); add: (Point x: 396 y: 337); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Selected'); to: (stateTable at: 'Edit Text Figure'); type: (TransitionType basicNew instVarAt: 1 put: #keyPress; yourself); points: ((OrderedCollection new) add: (Point x: 343 y: 273); add: (Point x: 350 y: 312); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Edit Text Figure'); to: (stateTable at: 'Select Text Figure'); type: (FigureTransitionType basicNew instVarAt: 1 put: #redButtonPress; instVarAt: 2 put: ((Array new: 1) at: 1 put: TextFigure; yourself); yourself); points: ((OrderedCollection new) add: (Point x: 307 y: 334); add: (Point x: 179 y: 367); add: (Point x: 52 y: 312); add: (Point x: 71 y: 198); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Edit Text Figure'); to: (stateTable at: 'Create Text'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 394 y: 317); add: (Point x: 433 y: 272); add: (Point x: 437 y: 137); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Edit Text Figure'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 348 y: 343); add: (Point x: 339 y: 369); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Selected'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 335 y: 273); add: (Point x: 256 y: 294); add: (Point x: 226 y: 339); add: (Point x: 259 y: 388); add: (Point x: 316 y: 385); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Figure Creation Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 427 y: 45); add: (Point x: 483 y: 57); add: (Point x: 508 y: 203); add: (Point x: 491 y: 390); add: (Point x: 357 y: 389); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Text Tool'); to: (stateTable at: 'Menu'); type: (TransitionType basicNew instVarAt: 1 put: #yellowButtonPress; yourself); points: ((OrderedCollection new) add: (Point x: 93 y: 66); add: (Point x: 29 y: 142); add: (Point x: 22 y: 273); add: (Point x: 37 y: 385); add: (Point x: 105 y: 402); add: (Point x: 180 y: 401); add: (Point x: 317 y: 390); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Text Tool' put: (ToolState name: 'Text Tool' command: [:tool :event | 
tool valueAt: #canCreate put: false]).
	Tool states at: 'Select Text Figure' put: (ToolState name: 'Select Text Figure' command: [:tool :event | 
| figure startIndex |
figure := tool figureAtEvent: event.
tool drawing selection: figure.
tool valueAt: #figure put: figure.
startIndex := figure indexForPoint: (tool cursorPointFor: event).
tool valueAt: #startIndex put: startIndex.
figure setIndices: (startIndex to: startIndex)]).
	Tool states at: 'Highlight Text' put: (ToolState name: 'Highlight Text' command: [:tool :event | 
| figure newIndex startIndex |
figure := tool valueAt: #figure.
startIndex := tool valueAt: #startIndex.
newIndex := figure indexForPoint: (tool cursorPointFor: event).
figure setIndices: 
	((newIndex min: startIndex) to: (newIndex max: startIndex))]).
	Tool states at: 'Text Figure Selected' put: (ToolState name: 'Text Figure Selected' command: [:tool :event | ]).
	Tool states at: 'Text Figure Creation Tool' put: (ToolState name: 'Text Figure Creation Tool' command: [:tool :event | 
tool valueAt: #canCreate put: true]).
	Tool states at: 'Create Text' put: (ToolState name: 'Create Text' command: [:tool :event | 
| figure |
(tool valueAt: #canCreate) 
	ifTrue: 
		[figure := TextFigure string: '' 
				at: (tool cursorPointFor: event).
		figure setIndices: (1 to: 1).
		tool drawing add: figure; selection: figure.
		tool valueAt: #figure put: figure] 
	ifFalse: 
		[tool changeToState: (Tool stateFor: 'End State') 
			event: event]]).
	Tool states at: 'Edit Text Figure' put: (ToolState name: 'Edit Text Figure' command: [:tool :event | 
| character figure offset replacement startIndex |
figure := tool valueAt: #figure.
character := event keyValue.
offset := 0.
(character == Character backspace or: [character == Character del])
	ifTrue: [figure selectedIndices size = 1 ifTrue: [offset := -1].
		replacement := Text new]
	ifFalse: [replacement := (String with: character) asText].
startIndex := figure selectedIndices first.
figure replaceFrom: (startIndex + offset max: 1)
	to: figure selectedIndices last - 1
	with: replacement.
startIndex := startIndex + replacement size + offset max: 1.
figure setIndices: (startIndex to: startIndex)
]).
	(Tool stateFor: 'Text Figure Creation Tool') redButtonPress: ((FigureTransitionTable new)  on: TextFigure goto: (Tool stateFor: 'Select Text Figure'); goto: (Tool stateFor: 'Create Text'); yourself).
	(Tool stateFor: 'Text Figure Creation Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Select Text Figure') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Text Figure Selected'); yourself).
	(Tool stateFor: 'Select Text Figure') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Highlight Text'); yourself).
	(Tool stateFor: 'Text Figure Selected') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Text Figure Selected') keyPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Edit Text Figure'); yourself).
	(Tool stateFor: 'Text Figure Selected') redButtonPress: ((FigureTransitionTable new)  on: TextFigure goto: (Tool stateFor: 'Select Text Figure'); goto: (Tool stateFor: 'Create Text'); yourself).
	(Tool stateFor: 'Highlight Text') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Text Figure Selected'); yourself).
	(Tool stateFor: 'Highlight Text') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Highlight Text'); yourself).
	(Tool stateFor: 'Edit Text Figure') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Edit Text Figure') redButtonPress: ((FigureTransitionTable new)  on: TextFigure goto: (Tool stateFor: 'Select Text Figure'); goto: (Tool stateFor: 'Create Text'); yourself).
	(Tool stateFor: 'Edit Text Figure') keyPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Edit Text Figure'); yourself).
	(Tool stateFor: 'Text Tool') redButtonPress: ((FigureTransitionTable new)  on: TextFigure goto: (Tool stateFor: 'Select Text Figure'); yourself).
	(Tool stateFor: 'Text Tool') yellowButtonPress: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Menu'); yourself).
	(Tool stateFor: 'Create Text') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Text Figure Selected'); yourself).!

initializeTrackHandleStates
	"This method was automatically generated by the HotDraw ToolStateMachineEditor.
	To edit this method, evaluate the comment below"

	"| states transitions stateTable |
	states := ((OrderedCollection new) add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Track Handle'; instVarAt: 3 put: (Text string: '[:tool :event | 
| handle |
handle := tool figureAtEvent: event.
tool valueAt: #moveBlock put: handle moveBlock]' runs: (RunArray runs: #(112) values: #(nil))); instVarAt: 4 put: (Point x: 149 y: 30); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Track handle move'; instVarAt: 3 put: (Text string: '[:tool :event | (tool valueAt: #moveBlock) value: (tool cursorPointFor: event)]' runs: (RunArray runs: #(79) values: #(nil))); instVarAt: 4 put: (Point x: 170 y: 173); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Track Handle Release'; instVarAt: 3 put: '[:tool :event | ]'; instVarAt: 4 put: (Point x: 61 y: 275); instVarAt: 5 put: true; instVarAt: 6 put: false; yourself); add: (ToolStateModel basicNew instVarAt: 1 put: nil; instVarAt: 2 put: 'Tentative Position Handle'; instVarAt: 3 put: (Text string: '[:tool :event | 
| handle |
handle := tool figureAtEvent: event.
handle owner addPoint: (tool cursorPointFor: event) beforeIndex: handle index.
tool drawing selections: tool drawing selections copy.
tool changeToState: (Tool stateFor: ''Track Handle'') event: event]' runs: (RunArray runs: #(264) values: #(nil))); instVarAt: 4 put: (Point x: 280 y: 87); instVarAt: 5 put: false; instVarAt: 6 put: false; yourself); yourself).
	stateTable := Dictionary new.
	states do: [:each | stateTable at: each name put: each].

	transitions := OrderedCollection new.
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Track Handle'); to: (stateTable at: 'Track handle move'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 185 y: 60); add: (Point x: 223 y: 173); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Track handle move'); to: (stateTable at: 'Track handle move'); type: (TransitionType basicNew instVarAt: 1 put: #mouseMove; yourself); points: ((OrderedCollection new) add: (Point x: 287 y: 184); add: (Point x: 332 y: 184); add: (Point x: 332 y: 204); add: (Point x: 280 y: 197); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Track handle move'); to: (stateTable at: 'Track Handle Release'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 205 y: 203); add: (Point x: 130 y: 275); yourself); yourself).
	transitions add: (ToolStateTransitionModel new from: (stateTable at: 'Track Handle'); to: (stateTable at: 'Track Handle Release'); type: (TransitionType basicNew instVarAt: 1 put: #redButtonRelease; yourself); points: ((OrderedCollection new) add: (Point x: 153 y: 52); add: (Point x: 104 y: 275); yourself); yourself).
	ToolStateMachineEditor openWithStates: states connections: transitions"

	Tool states at: 'Track Handle' put: (ToolState name: 'Track Handle' command: [:tool :event | 
| handle |
handle := tool figureAtEvent: event.
tool valueAt: #moveBlock put: handle moveBlock]).
	Tool states at: 'Track handle move' put: (ToolState name: 'Track handle move' command: [:tool :event | (tool valueAt: #moveBlock) value: (tool cursorPointFor: event)]).
	Tool states at: 'Track Handle Release' put: (EndToolState name: 'Track Handle Release' command: [:tool :event | ]).
	Tool states at: 'Tentative Position Handle' put: (ToolState name: 'Tentative Position Handle' command: [:tool :event | 
| handle |
handle := tool figureAtEvent: event.
handle owner addPoint: (tool cursorPointFor: event) beforeIndex: handle index.
tool drawing selections: tool drawing selections copy.
tool changeToState: (Tool stateFor: 'Track Handle') event: event]).
	(Tool stateFor: 'Track Handle') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Track Handle Release'); yourself).
	(Tool stateFor: 'Track Handle') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Track handle move'); yourself).
	(Tool stateFor: 'Track handle move') redButtonRelease: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Track Handle Release'); yourself).
	(Tool stateFor: 'Track handle move') mouseMove: ((SimpleTransitionTable new) goto: (Tool stateFor: 'Track handle move'); yourself).! !
Tool initialize!


