Figure subclass: #Handle
	instanceVariableNames: 'owner toolState '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Handles'!
Handle comment:
'Handle is a special figure that is displayed when another figure in the drawing is selected. Handles aren''t permant figures, but are added and removed whenever another figure is selected/deselected. They are useful for defining special manipulation actions such as resizing, connecting figures, etc.

Instance Variables:
	owner	<Figure>	the figure that we are for
	toolState	<EndToolState>	whenever the tool presses us, goto this state

'!


!Handle methodsFor: 'accessing'!

menuAt: aPoint 
	^Menu new!

owner
	^owner!

owner: aFigure 
	owner := aFigure!

toolState
	^toolState!

toolState: aToolState 
	toolState := aToolState! !

!Handle methodsFor: 'copying'!

postCopy
	super postCopy.
	owner := nil! !

!Handle methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext
	self class image displayOn: aGraphicsContext at: self origin! !

!Handle methodsFor: 'initialize-release'!

initialize
	super initialize.
	self selectable: false.
	bounds := 0 @ 0 extent: self class image bounds extent.
	toolState := EndToolState name: 'No-op' command: [:tool :event | ]! !

!Handle methodsFor: 'testing'!

isHandle
	^true! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Handle class
	instanceVariableNames: 'image '!


!Handle class methodsFor: 'accessing'!

buildImage
	| size |
	size := 6 @ 6.
	image := CompositePart new.
	image
		add: ((GraphicsAttributesWrapper 
					on: (FillingWrapper on: (0 @ 0 extent: size))) 
						attributes: (GraphicsAttributes new paint: ColorValue black));
		add: ((GraphicsAttributesWrapper 
					on: (FillingWrapper on: (size // 2 - 1 extent: 2 @ 2))) 
						attributes: (GraphicsAttributes new paint: ColorValue white)).
	^image!

image
	^image isNil 
		ifTrue: [image := self buildImage] 
		ifFalse: [image]! !

!Handle class methodsFor: 'corner handles'!

allCornersOf: aFigure 
	^OrderedCollection 
		with: (self topLeftOf: aFigure)
		with: (self topRightOf: aFigure)
		with: (self bottomRightOf: aFigure)
		with: (self bottomLeftOf: aFigure)!

bottomLeftOf: aFigure 
	^self on: aFigure at: #bottomLeft!

bottomRightOf: aFigure 
	^self on: aFigure at: #bottomRight!

topLeftOf: aFigure 
	^self on: aFigure at: #topLeft!

topRightOf: aFigure 
	^self on: aFigure at: #topRight! !

!Handle class methodsFor: 'instance creation'!

connectionOn: aFigure at: aSymbol 
	^(self on: aFigure at: aSymbol)
		toolState: (Tool stateFor: 'Connection State');
		yourself!

on: aFigure at: aSymbol 
	| handle |
	handle := self new.
	handle
		owner: aFigure;
		constrain: #center to: (MessageSend receiver: aFigure selector: aSymbol).
	^handle!

on: aFigure at: aSymbol offset: aPoint 
	| handle |
	handle := self new.
	handle
		owner: aFigure;
		constrain: #center
			to: (MessageSend 
					receiver: aFigure
					selector: aSymbol
					arguments: (Array with: aPoint)).
	^handle! !

Handle subclass: #TrackHandle
	instanceVariableNames: 'moveBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Handles'!
TrackHandle comment:
'TrackHandle is a handle that executes a block whenever the cursor is moved. When the mouse button is release, it quits its action.

Instance Variables:
	moveBlock	<BlockClosure>	a block that is executed for each mouse position, it takes one argument (a Point object)

'!


!TrackHandle methodsFor: 'accessing'!

moveBlock
	^moveBlock!

moveBlock: aBlockClosure 
	moveBlock := aBlockClosure! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TrackHandle class
	instanceVariableNames: ''!


!TrackHandle class methodsFor: 'instance creation'!

bottomLeftOf: aFigure 
	^(super bottomLeftOf: aFigure)
		moveBlock: [:aPoint | aFigure bottomLeft: aPoint];
		yourself!

bottomRightOf: aFigure 
	^(super bottomRightOf: aFigure)
		moveBlock: [:aPoint | aFigure bottomRight: aPoint];
		yourself!

new
	| handle |
	handle := super new.
	handle toolState: (Tool stateFor: 'Track Handle').
	^handle!

pointAtIndex: anIndex of: aFigure 
	| handle |
	handle := self new.
	^handle
		owner: aFigure;
		constrain: #center
			to: (MessageSend 
					receiver: aFigure
					selector: #pointAt:
					arguments: (Array with: anIndex));
		moveBlock: [:aPoint | aFigure pointAt: anIndex put: aPoint];
		yourself!

topLeftOf: aFigure 
	^(super topLeftOf: aFigure)
		moveBlock: [:aPoint | aFigure topLeft: aPoint];
		yourself!

topRightOf: aFigure 
	^(super topRightOf: aFigure)
		moveBlock: [:aPoint | aFigure topRight: aPoint];
		yourself! !

TrackHandle subclass: #IndexedTrackHandle
	instanceVariableNames: 'index '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Handles'!
IndexedTrackHandle comment:
'IndexedTrackHandle is a TrackHandle that has a menu. This menu allows the user to remove points from the line that the handle is for.

Instance Variables:
	index	<Integer>	the index of the point on the line that we''re connected to

'!


!IndexedTrackHandle methodsFor: 'accessing'!

menuAt: aPoint 
	| menu |
	menu := super menuAt: aPoint.
	menu addItemLabel: 'Remove' value: #removePoint.
	^menu!

removePoint
	owner removePointAtIndex: index! !

!IndexedTrackHandle methodsFor: 'initialize-release'!

index: anObject
	index := anObject! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

IndexedTrackHandle class
	instanceVariableNames: ''!


!IndexedTrackHandle class methodsFor: 'instance creation'!

pointAtIndex: anIndex of: aFigure 
	| handle |
	handle := super pointAtIndex: anIndex of: aFigure.
	handle index: anIndex.
	^handle! !

TrackHandle subclass: #TentativePositionHandle
	instanceVariableNames: 'index '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Handles'!
TentativePositionHandle comment:
'TentativePositionHandle is a special handle that allows the user to create a new point on the line. Normally it is displayed between two real points on the line.

Instance Variables:
	index	<Index>	the index of the new point

'!


!TentativePositionHandle methodsFor: 'accessing'!

index
	^index!

index: anInteger 
	index := anInteger! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TentativePositionHandle class
	instanceVariableNames: ''!


!TentativePositionHandle class methodsFor: 'accessing'!

buildImage
	| size |
	size := 6 @ 6.
	image := CompositePart new.
	image
		add: ((GraphicsAttributesWrapper 
					on: (FillingWrapper on: (0 @ 0 extent: size))) 
						attributes: (GraphicsAttributes new paint: ColorValue gray));
		add: ((GraphicsAttributesWrapper 
					on: (FillingWrapper on: (size // 2 - 1 extent: 2 @ 2))) 
						attributes: (GraphicsAttributes new paint: ColorValue white)).
	^image! !

!TentativePositionHandle class methodsFor: 'instance creation'!

forSegment: anIndex of: aFigure 
	^(self new)
		toolState: (Tool stateFor: 'Tentative Position Handle');
		owner: aFigure;
		constrain: #center
			to: (MessageSend 
					receiver: aFigure
					selector: #centerOfSegment:
					arguments: (Array with: anIndex));
		index: anIndex + 1;
		moveBlock: [:aPoint | aFigure pointAt: anIndex + 1 put: aPoint];
		yourself! !

