Figure subclass: #EllipseFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
EllipseFigure comment:
'EllipseFigure is a figure that represents ellipses.'!


!EllipseFigure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	"Display an ellipse on aGraphicsContext. If the ellipse figure is not transparent then 
	display a filled ellipse using the current fillColor."

	self isOpaque ifTrue: [self displayFilledOn: aGraphicsContext].
	self lineWidth > 0 ifTrue: [self displayOutlineOn: aGraphicsContext]!

displayFilledOn: aGraphicsContext 
	aGraphicsContext paint: self fillColor.
	aGraphicsContext 
		displayWedgeBoundedBy: self bounds
		startAngle: self startAngle
		sweepAngle: self sweepAngle!

displayOutlineOn: aGraphicsContext 
	aGraphicsContext lineWidth: self lineWidth.
	aGraphicsContext paint: self lineColor.
	aGraphicsContext 
		displayArcBoundedBy: self bounds
		startAngle: self startAngle
		sweepAngle: self sweepAngle! !

!EllipseFigure methodsFor: 'initialize-release'!

ellipse: aRectangle 
	bounds := aRectangle.
	self opaque! !

!EllipseFigure methodsFor: 'private'!

startAngle
	^0!

sweepAngle
	^360! !

!EllipseFigure methodsFor: 'testing'!

containsPoint: aPoint 
	^self isOpaque
		ifTrue: 
			[(EllipticalArc boundingBox: self preferredBounds
				startAngle: self startAngle
				sweepAngle: self sweepAngle) regionIntersects: (aPoint - 1 extent: 2 @ 2)]
		ifFalse: 
			[(EllipticalArc boundingBox: self preferredBounds
				startAngle: self startAngle
				sweepAngle: self sweepAngle) outlineIntersects: (aPoint - 1 extent: 2 @ 2)]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

EllipseFigure class
	instanceVariableNames: ''!


!EllipseFigure class methodsFor: 'instance creation'!

createAt: aPoint 
	^self ellipse: (aPoint extent: 0 @ 0)!

ellipse: anEllipse 
	^self new ellipse: anEllipse! !

Figure subclass: #ViewAdapterFigure
	instanceVariableNames: 'component '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
ViewAdapterFigure comment:
'ViewAdapterFigure adapts VisualCompoents to the Figure protocol, allowing any VisualComponent to be inserted into a Drawing.

Instance Variables:
	component	<VisualComponent>	the visual component that is adapted

'!


!ViewAdapterFigure methodsFor: 'accessing'!

component
	^component!

component: aVisualComponent 
	component := aVisualComponent!

container: aContainer 
	super container: aContainer.
	aContainer isNil ifTrue: [^self].
	self visualComponentsIn: self
		do: [:each | (each isKindOf: ComposedTextView) ifTrue: [each initializeSelection]]! !

!ViewAdapterFigure methodsFor: 'bounds changing'!

changedPreferredBounds: aRectangle 
	component bounds: self bounds.
	super changedPreferredBounds: aRectangle! !

!ViewAdapterFigure methodsFor: 'broadcast'!

downcastEvent: aKey with: aParameter from: anInitiator 
	"This is a message passed down the view structure to all subparts."

	super downcastEvent: aKey
		with: aParameter
		from: anInitiator.
	component downcastEvent: aKey
		with: aParameter
		from: anInitiator!

flushCoordinateCaches
	"Flush caches that relate to coordinate translations between this component 
	and its container"

	super flushCoordinateCaches.
	component flushCoordinateCaches!

newGraphicsDevice: aGraphicsDevice 
	"Forward the newGraphicsDevice: aGraphicsDevice messages
	to the receiver's components."

	super newGraphicsDevice: aGraphicsDevice.
	component newGraphicsDevice: aGraphicsDevice! !

!ViewAdapterFigure methodsFor: 'changing'!

delete
	| topComponent |
	topComponent := self topComponent.
	topComponent isNil ifTrue: [^self].
	self visualComponentsIn: self
		do: 
			[:each | 
			topComponent keyboardProcessor keyboardConsumers remove: each ifAbsent: []].
	super delete! !

!ViewAdapterFigure methodsFor: 'control'!

objectWantingControl
	"Answer the first component that wishes to take control.  Answer nil if there is
	no component wanting control."

	^component objectWantingControl! !

!ViewAdapterFigure methodsFor: 'copying'!

postCopy
	super postCopy.
	component := component copy! !

!ViewAdapterFigure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	self isOpaque ifTrue: 
			[aGraphicsContext paint: self fillColor;
				displayRectangle: self bounds;
				paint: self lineColor].
	aGraphicsContext intersectClip: self bounds.
	aGraphicsContext translateBy: self origin.
	component displayOn: aGraphicsContext! !

!ViewAdapterFigure methodsFor: 'event driven'!

handlerForMouseEvent: anEvent 
	^component handlerForMouseEvent: anEvent! !

!ViewAdapterFigure methodsFor: 'initialize-release'!

release
	component release.
	super release!

view: aView in: aRectangle 
	bounds := aRectangle.
	component := aView.
	component container: self.
	component bounds: (0 @ 0 extent: aRectangle extent)! !

!ViewAdapterFigure methodsFor: 'private'!

changedBounds: oldBounds forComponent: aVisualComponent
	"The bounds of aVisualComponent has changed from oldBounds.
	Do nothing since the receiver has its own notion of bounds."

	^self!

changedContainer
	super changedContainer.
	container isNil ifTrue: [^self].
	self setKeyboardProcessor: self topComponent keyboardProcessor.
	self setOccluded!

changedPreferredBounds: oldPreferredBounds forComponent: aVisualComponent
	"The preferred bounds of aVisualComponent has changed from
	oldPreferredBounds."

	"Do nothing since the receiver has its own notion of bounds."

	^self!

compositionBoundsFor: aVisualPart 
	"The receiver is a container for aVisualPart.
	An actual bounding rectangle is being searched for by
	aVisualPart. Forward to the receiver's container."

	^container isNil ifTrue: [self bounds] ifFalse: [0 @ 0 extent: self extent]!

downcastLocalEvent: aKey with: aParameter at: aPoint from: anInitiator 
	"This is a message passed down the view structure to some single 
	part. Answer true if we accepted the event, or false if it should be 
	passed on to whatever's behind us."

	^super
		downcastLocalEvent: aKey
		with: aParameter
		at: (aPoint translatedBy: self origin negated)
		from: anInitiator!

graphicsContextFor: aVisualComponent
	"Answer a GraphicsContext set-up for aVisualComponent."

	| gc |
	gc := container graphicsContextFor: self.
	gc intersectClip: self bounds.
	gc translateBy: self origin.
	^gc!

invalidateRectangle: aRectangle repairNow: aBoolean forComponent: aVisualComponent 
	"Propagate damage up the hierarchy after translating and clipping."

	| rect |
	container isNil ifTrue: [^self].
	rect := aRectangle translatedBy: self origin.
	(self intersects: rect) 
		ifTrue: 
			[container 
				invalidateRectangle: (self bounds intersect: rect)
				repairNow: aBoolean
				forComponent: self]!

localPointToGlobal: aPoint 
	"Convert a point in local coordinates to a point in the top window's coordinate system."

	^aPoint translatedBy: (container localPointToGlobal: self origin)!

setComponentBoundsTo: newBounds 
	component bounds: (0 @ 0 extent: newBounds extent)!

setKeyboardProcessor: aKeyboardProcessor 
	self visualComponentsIn: self
		do: 
			[:each | 
			(each respondsTo: #controller)
				ifTrue: [aKeyboardProcessor sendKeyboardTo: each]]!

setOccluded
	self downcastEvent: #useInvalidation
		with: true
		from: self!

visualComponentsIn: aVisualComponent do: aBlock 
	"This is a hack to go over the components of aVisualComponent"

	aBlock value: aVisualComponent.
	(aVisualComponent respondsTo: #component)
		ifTrue: [self visualComponentsIn: aVisualComponent component do: aBlock].
	(aVisualComponent respondsTo: #components) ifTrue: 
			[aVisualComponent components
				do: [:each | self visualComponentsIn: each component do: aBlock]]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ViewAdapterFigure class
	instanceVariableNames: ''!


!ViewAdapterFigure class methodsFor: 'instance creation'!

view: aView in: aRectangle 
	^self new view: aView in: aRectangle! !

Figure subclass: #CachedFigure
	instanceVariableNames: 'cache '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
CachedFigure comment:
'CachedFigure is a figure that holds a pixmap of its display. It can be used for complicated figures to cache their display so that they display faster. However, as graphics cards become faster, this figure is less useful.

Subclasses must implement the following messages:
	private
		fillCache	returns a pixmap of our contents

Instance Variables:
	cache	<Pixmap>	what we display when we are drawn

'!


!CachedFigure methodsFor: 'accessing'!

handles
	^Handle allCornersOf: self!

origin: aPoint 
	bounds := aPoint extent: self cache extent! !

!CachedFigure methodsFor: 'displaying'!

displayFigureOn: aGC 
	self cache displayOn: aGC at: self origin rounded! !

!CachedFigure methodsFor: 'initialize-release'!

flushCaches
	super flushCaches.
	cache := nil!

initialize
	super initialize.
	self opaque! !

!CachedFigure methodsFor: 'private'!

cache
	^cache isNil 
		ifTrue: [cache := self fillCache] 
		ifFalse: [cache]!

fillCache
	"Draw each of my component Figures onto my cache."

	^self subclassResponsibility! !

!CachedFigure methodsFor: 'transforming'!

computePreferredBounds
	^bounds isNil 
		ifTrue: [0 @ 0 extent: self cache extent] 
		ifFalse: [bounds]! !

Figure subclass: #TextFigure
	instanceVariableNames: 'paragraph selection '
	classVariableNames: 'InsertionPoint '
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
TextFigure comment:
'TextFigure is a Figure that represents text.

Instance Variables:
	paragraph	<Paragraph>	the text that is displayed
	selection	<Interval | nil>	the selected text interval, when the figure is part of the drawing''s selection, then this selection is also highlighted

Class Variables:
	InsertionPoint	<Image>	an image of the ^ that is displayed at the insertion point.

'!


!TextFigure methodsFor: 'accessing'!

copyFrom: startIndex to: stopIndex 
	^paragraph asText copyFrom: startIndex to: stopIndex!

handles
	| handles |
	handles := OrderedCollection new.
	handles add: (Handle connectionOn: self at: #origin).
	handles add: ((TrackHandle topRightOf: self) 
				moveBlock: [:aPoint | self moveMarginTo: aPoint]).
	^handles!

indexForPoint: aPoint 
	"Answer the index into my text that most closely corresponds to aPoint."

	^(paragraph characterBlockAtPoint: aPoint - self origin) stringIndex!

margin
	^self extent x!

margin: aNumber 
	paragraph compositionWidth: aNumber rounded.
	self recomputePreferredBounds.
	^aNumber!

menuAt: aPoint 
	| mb |
	mb := MenuBuilder new.
	self addJustificationMenuTo: mb.
	self addVisibilityMenuTo: mb.
	self addStyleMenuTo: mb.
	^(mb menu)
		addItem: ((MenuItem labeled: 'other')
					submenu: (super menuAt: aPoint);
					yourself);
		yourself!

paragraph
	^paragraph!

print: anObject 
	self string: anObject printString!

replaceFrom: startIndex to: stopIndex with: aText 
	paragraph := (paragraph copy)
				replaceFrom: startIndex
					to: stopIndex
					with: aText;
				yourself.
	self margin: paragraph rightMarginForComposition!

replaceSelectionWith: aText 
	self 
		replaceFrom: selection first
		to: selection last - 1
		with: aText!

selectedIndices
	^selection!

setIndices: newIndices 
	"Set the indices of the highlighted part --- newly added"

	newIndices notNil 
		ifTrue: 
			[(selection notNil and: [selection first = newIndices first 
				and: [selection last = newIndices last]]) 
					ifFalse: 
						[selection := newIndices.
						self changed]]
		ifFalse: 
			[selection notNil 
				ifTrue: 
					[selection := nil.
					self changed]]!

string
	^paragraph asString!

string: aString 
	self text: aString asText!

text: aText 
	paragraph := aText asComposedText! !

!TextFigure methodsFor: 'attribute accessing'!

deselect
	super deselect.
	selection := nil.
	self changed!

select
	super select.
	self changed! !

!TextFigure methodsFor: 'bounds accessing'!

computePreferredBounds
	| extent |
	extent := paragraph width @ paragraph height.
	^self origin extent: extent! !

!TextFigure methodsFor: 'copying'!

postCopy
	super postCopy.
	paragraph := paragraph asString asComposedText! !

!TextFigure methodsFor: 'defaults'!

defaultCompositionWidth
	^100! !

!TextFigure methodsFor: 'displaying'!

displayCaretOn: aGraphicsContext 
	"Show a caret inbetween characters anInteger and anInteger+1. Actually, reverse the
	image of a caret at this point. Thus we can toggle the caret on and off by calling this method
	repeatedly."

	| ipExtent aPoint aBlock |
	ipExtent := InsertionPoint extent.
	aBlock := self paragraph characterBlockForIndex: selection first.
	aPoint := (aBlock left - (ipExtent x // 2)) 
				@ (aBlock top + self paragraph textStyle baseline) extent: ipExtent.
	InsertionPoint displayOn: aGraphicsContext at: self origin + aPoint origin!

displayEditOn: aGraphicsContext 
	selection isNil ifTrue: [^self].
	selection first = selection last 
		ifTrue: [self displayCaretOn: aGraphicsContext]
		ifFalse: [self highlightOn: aGraphicsContext]!

displayFigureOn: aGraphicsContext 
	"If I am opaque, draw a box in the background color obscuring those figures beneath me."

	self isOpaque 
		ifTrue: 
			[aGraphicsContext
				paint: self fillColor;
				displayRectangle: self bounds].
	aGraphicsContext
		paint: self lineColor;
		display: paragraph at: self origin!

displaySelectedFigureOn: aGraphicsContext 
	self displayFigureOn: aGraphicsContext.
	self displayEditOn: aGraphicsContext!

highlightOn: aGraphicsContext 
	"Display the image of my text from start to stop. Note they may not be in the proper
	order."

	| startPoint stopPoint lineRange first last start stop |
	start := selection first.
	stop := selection last.
	start = stop ifTrue: [^self].
	startPoint := paragraph characterBlockForIndex: (start min: stop).
	stopPoint := paragraph characterBlockForIndex: (start max: stop).
	aGraphicsContext paint: self selectionBackgroundColor.
	startPoint top = stopPoint top 
		ifTrue: 
			[aGraphicsContext displayRectangle: (startPoint origin + self origin 
						corner: stopPoint bottomLeft + self origin).
			aGraphicsContext paint: self selectionForegroundColor.
			^paragraph 
				displayFromCharacter: startPoint stringIndex
				to: stopPoint stringIndex - 1
				startX: startPoint left + self origin x
				forTranslation: self origin
				on: aGraphicsContext].
	lineRange := paragraph 
				lineRangeFor: (startPoint origin corner: stopPoint corner).
	first := (paragraph lineAt: lineRange first) first.
	last := (paragraph lineAt: lineRange last) last.
	startPoint stringIndex > first ifTrue: [first := startPoint stringIndex].
	stopPoint stringIndex - 1 < last 
		ifTrue: [last := stopPoint stringIndex - 1].
	aGraphicsContext displayRectangle: (startPoint origin + self origin 
				corner: self extent x @ startPoint bottom + self origin).
	aGraphicsContext 
		displayRectangle: (self origin x @ (startPoint bottom + self origin y) 
				corner: self extent x @ stopPoint top + self origin).
	aGraphicsContext 
		displayRectangle: (self origin x @ (startPoint bottom + self origin y) 
				corner: stopPoint bottomLeft + self origin).
	aGraphicsContext paint: self selectionForegroundColor.
	paragraph 
		displayFromCharacter: first
		to: last
		startX: startPoint left + self origin x
		forTranslation: self origin
		on: aGraphicsContext! !

!TextFigure methodsFor: 'formatting'!

alignUsing: aSymbol 
	self margin: self margin.
	paragraph textStyle: paragraph textStyle copy.
	paragraph perform: aSymbol.
	self changed!

styleUsing: aSymbol 
	| start stop |
	selection isNil 
		ifTrue: 
			[start := 1.
			stop := paragraph asString size]
		ifFalse: 
			[start := selection first.
			stop := selection last - 1].
	self paragraph asText 
		emphasizeFrom: start
		to: stop
		with: aSymbol.
	self recompose! !

!TextFigure methodsFor: 'initialize-release'!

setParagraph: aParagraph origin: aPoint 
	paragraph := aParagraph.
	paragraph width = 0 
		ifTrue: [paragraph compositionWidth: self defaultCompositionWidth].
	bounds := aPoint extent: 0 @ 0.
	self
		recomputePreferredBounds;
		setIndices: nil! !

!TextFigure methodsFor: 'private'!

addJustificationMenuTo: aMenuBuilder 
	aMenuBuilder beginSubMenuLabeled: 'justification'.
	#('flush left' 'centered' 'flush right' 'justified') 
		with: #(#leftFlush #centered #rightFlush #justified)
		do: [:label :symbol | aMenuBuilder add: label -> [self alignUsing: symbol]].
	aMenuBuilder endSubMenu!

addStyleMenuTo: aMenuBuilder 
	aMenuBuilder beginSubMenuLabeled: 'style'.
	#(#bold #italic #underline #strikeout #default #large #small #fixed) 
		do: [:each | aMenuBuilder add: each -> [self styleUsing: each]].
	aMenuBuilder endSubMenu! !

!TextFigure methodsFor: 'transforming'!

moveMarginTo: aPoint
	| newMargin |
	newMargin := (aPoint - self origin) x.
	newMargin <= 0 ifTrue: [^self].
	newMargin = self extent x ifTrue: [^self].
	self margin: newMargin!

recompose
	"Recompose my paragraph, resetting the width and possibly the right margin."

	self margin: paragraph composeAll!

textStyle: aTextStyle 
	aTextStyle alignment = paragraph textStyle alignment 
		ifTrue: [paragraph textStyle: aTextStyle]
		ifFalse: 
			[paragraph textStyle: ((aTextStyle copy)
						alignment: paragraph textStyle alignment;
						yourself)]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TextFigure class
	instanceVariableNames: ''!


!TextFigure class methodsFor: 'class initialization'!

initialize
	"TextFigure initialize"

	InsertionPoint := Image 
				extent: 7 @ 4
				depth: 1
				palette: CoveragePalette monoMaskPalette
				bits: #[8 28 62 62]
				pad: 8! !

!TextFigure class methodsFor: 'instance creation'!

createAt: aPoint with: aString 
	^self string: aString at: aPoint!

paragraph: aParagraph at: aPoint 
	^self new setParagraph: aParagraph origin: aPoint!

string: aString 
	^self string: aString at: 0 @ 0!

string: aString at: aPoint 
	^self paragraph: aString asComposedText at: aPoint! !

Figure subclass: #PolylineFigure
	instanceVariableNames: 'points '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
PolylineFigure comment:
'PolylineFigure is a figure that is represented by a series of points. The default polyline figure just draws straight lines between these points, but other subclasses can draw curved lines through the points. If this figure is filled, then the start and end points should be the same.

Instance Variables:
	points	<SequenceableCollection of: Point>	our points

'!


!PolylineFigure methodsFor: 'accessing'!

centerOfSegment: anIndex 
	^((points at: anIndex) + (points at: anIndex + 1)) // 2!

handles
	| handles |
	handles := OrderedCollection new.
	1 to: points size - 1
		do: 
			[:i | 
			handles 
				add: ((i == 1 ifTrue: [TrackHandle] ifFalse: [IndexedTrackHandle]) 
						pointAtIndex: i
						of: self).
			handles add: (TentativePositionHandle forSegment: i of: self)].
	handles add: (TrackHandle pointAtIndex: points size of: self).
	^handles!

pointAt: anIndex 
	^points at: anIndex!

pointAt: anIndex put: aValue 
	| oldBounds |
	oldBounds := self preferredBounds.
	points at: anIndex put: aValue.
	bounds := self computePreferredBounds.
	self changedPreferredBounds: oldBounds.
	anIndex == 1 ifTrue: [^self changed: #startPoint].
	anIndex == points size ifTrue: [^self changed: #stopPoint].
	self changed!

pointsSize
	^points size!

startPoint
	^points first!

startPoint: aPoint 
	self pointAt: 1 put: aPoint!

stopPoint
	^points last!

stopPoint: aPoint 
	self pointAt: points size put: aPoint! !

!PolylineFigure methodsFor: 'adding'!

addPoint: aPoint 
	points add: aPoint.
	self updatePreferredBoundsWith: aPoint.
	^aPoint!

addPoint: aPoint beforeIndex: aPosition 
	points add: aPoint beforeIndex: aPosition.
	self updatePreferredBoundsWith: aPoint.
	^aPoint! !

!PolylineFigure methodsFor: 'bounds accessing'!

computePreferredBounds
	| rect |
	rect := points inject: (points first extent: 0 @ 0)
				into: 
					[:sum :point | 
					sum
						origin: (sum origin min: point);
						corner: (sum corner max: point)].
	rect corner: rect corner + (self lineWidth // 2) + 1.	"Add one, so we get the horizontal/vertical lines"
	rect origin: rect origin - (self lineWidth // 2).
	^rect! !

!PolylineFigure methodsFor: 'changing'!

deletionUpdateFrom: aFigure 
	"Since we're probably connected to the figure that was deleted, remove ourself"

	super deletionUpdateFrom: aFigure.
	self drawing notNil ifTrue: [self drawing remove: self]! !

!PolylineFigure methodsFor: 'converting'!

asGeometric
	^Polyline vertices: points! !

!PolylineFigure methodsFor: 'copying'!

postCopy
	super postCopy.
	points := points copy! !

!PolylineFigure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	| geometric |
	geometric := self asGeometric.
	self isOpaque ifTrue: 
			[aGraphicsContext paint: self fillColor.
			geometric displayFilledOn: aGraphicsContext].
	self lineWidth > 0 ifTrue: 
			[aGraphicsContext lineWidth: self lineWidth.
			aGraphicsContext paint: self lineColor.
			geometric displayStrokedOn: aGraphicsContext]! !

!PolylineFigure methodsFor: 'initialize-release'!

initialize
	super initialize.
	points := #()!

points: aCollection 
	points := aCollection asOrderedCollection! !

!PolylineFigure methodsFor: 'private'!

updatePreferredBoundsWith: aPoint 
	| oldBounds |
	oldBounds := self preferredBounds.
	bounds := (oldBounds origin min: aPoint) corner: (oldBounds corner max: aPoint).
	oldBounds ~= bounds ifTrue: [self changedPreferredBounds: oldBounds]! !

!PolylineFigure methodsFor: 'removing'!

removePointAtIndex: aPosition 
	| isSelected |
	(isSelected := self isSelected) 
		ifTrue: [self drawing deselectFigure: self].
	points size > 2 ifFalse: [^self].
	points removeAtIndex: aPosition.
	self recomputePreferredBounds.
	isSelected ifTrue: [self drawing selectFigure: self]! !

!PolylineFigure methodsFor: 'testing'!

containsPoint: aPoint 
	^self isOpaque
		ifTrue: 
			[self asGeometric regionIntersects: (aPoint - 3 - (self lineWidth // 2)
						corner: aPoint + 3 + (self lineWidth // 2))]
		ifFalse: 
			[self asGeometric outlineIntersects: (aPoint - 3 - (self lineWidth // 2)
						corner: aPoint + 3 + (self lineWidth // 2))]! !

!PolylineFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	points := points collect: [:each | each + aPoint]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PolylineFigure class
	instanceVariableNames: ''!


!PolylineFigure class methodsFor: 'instance creation'!

connect: aFigure to: anotherFigure 
	| figure |
	figure := self withPoints: (Array with: aFigure center with: anotherFigure center).
	aFigure addDependent: figure.
	anotherFigure addDependent: figure.
	^figure!

createAt: aPoint
	^self withPoints: (Array with: aPoint with: aPoint)!

start: aPoint stop: anotherPoint 
	^self withPoints: (Array with: aPoint with: anotherPoint)!

withPoints: aCollection
	^self new points: aCollection! !

PolylineFigure subclass: #BezierFigure
	instanceVariableNames: 'bezier '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
BezierFigure comment:
'BezierFigure represents a bezier curve.

Instance Variables:
	bezier	<Bezier>	our curve

'!


!BezierFigure methodsFor: 'accessing'!

handles
	^(1 to: points size) collect: [:i | TrackHandle pointAtIndex: i of: self]!

pointAt: anIndex put: aValue 
	super pointAt: anIndex put: aValue.
	bezier := nil! !

!BezierFigure methodsFor: 'copying'!

postCopy
	super postCopy.
	bezier := nil! !

!BezierFigure methodsFor: 'private'!

asGeometric
	^self bezier!

bezier
	^bezier isNil 
		ifTrue: [bezier := self computeBezier] 
		ifFalse: [bezier]!

computeBezier
	^Bezier 
		start: points first
		end: points last
		controlPoint1: (points at: 2)
		controlPoint2: (points at: 3)!

computePreferredBounds
	| rect |
	rect := self bezier bounds.
	^rect origin - (self lineWidth // 2) 
		corner: rect corner + (self lineWidth // 2)! !

!BezierFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	super basicTranslateBy: aPoint.
	bezier := bezier translatedBy: aPoint! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BezierFigure class
	instanceVariableNames: ''!


!BezierFigure class methodsFor: 'instance creation'!

createAt: aPoint 
	^self start: aPoint stop: aPoint!

start: startPoint stop: stopPoint 
	| diff |
	diff := stopPoint - startPoint.
	^self withPoints: (Array 
				with: startPoint
				with: diff / 3.0 + startPoint
				with: diff * 2 / 3.0 + startPoint
				with: stopPoint)! !

PolylineFigure subclass: #SplineFigure
	instanceVariableNames: 'spline '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
SplineFigure comment:
'SplineFigure is a figure that represents a spline.

Instance Variables:
	spline	<Spline>	the geometric spline object

'!


!SplineFigure methodsFor: 'accessing'!

pointAt: anIndex put: aValue 
	super pointAt: anIndex put: aValue.
	spline := nil! !

!SplineFigure methodsFor: 'copying'!

postCopy
	super postCopy.
	spline := nil! !

!SplineFigure methodsFor: 'private'!

asGeometric
	^self spline!

computePreferredBounds
	| rect |
	rect := self spline bounds.
	^rect origin - (self lineWidth // 2) 
		corner: rect corner + (self lineWidth // 2) + 1!

computeSpline
	^Spline controlPoints: points asArray!

spline
	^spline isNil
		ifTrue: [spline := self computeSpline] 
		ifFalse: [spline]! !

!SplineFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	super basicTranslateBy: aPoint.
	spline := spline translatedBy: aPoint! !

PolylineFigure subclass: #LineFigure
	instanceVariableNames: 'annotations '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
LineFigure comment:
'LineFigure is a special PolylineFigure that can add annotations to its start and end points. These annotations can be arrows, circles, etc.

Instance Variables:
	annotations	<SequenceableCollection of: LineAnnotation>	the annotations to display on our line

'!


!LineFigure methodsFor: 'accessing'!

addAnnotation: aLineAnnotation 
	annotations := annotations copyWith: aLineAnnotation.
	aLineAnnotation container: self.
	self recomputePreferredBounds!

addStartArrow
	^self addAnnotation: ArrowAnnotation forStart!

addStopArrow
	^self addAnnotation: ArrowAnnotation forStop!

menuAt: aPoint 
	| menu submenu selectedAnnotation |
	submenu := MenuBuilder new.
	LineAnnotation allSubclasses do: 
			[:each | 
			| name |
			name := each name asString.
			(name copyFrom: (name size - 9 max: 1) to: name size) = 'Annotation' 
				ifTrue: [name := name copyFrom: 1 to: name size - 10].
			submenu
				add: ('Start ' , name) -> [self addAnnotation: each forStart];
				add: ('Stop ' , name) -> [self addAnnotation: each forStop]].
	selectedAnnotation := annotations 
				detect: [:each | each containsPoint: aPoint]
				ifNone: [nil].
	selectedAnnotation notNil 
		ifTrue: 
			[submenu
				line;
				add: 'Remove annotation' -> [self removeAnnotation: selectedAnnotation]].
	menu := super menuAt: aPoint.
	menu addItem: ((MenuItem labeled: 'Annotations') submenu: submenu menu).
	^menu!

removeAnnotation: aLineAnnotation 
	annotations := annotations copyWithout: aLineAnnotation.
	aLineAnnotation container: nil.
	self recomputePreferredBounds!

startDirection
	| direction |
	direction := points first - (points at: 2).
	^direction = (0 @ 0) 
		ifTrue: [1 @ 0] 
		ifFalse: [direction unitVector]!

stopDirection
	| direction |
	direction := points last - (points at: points size - 1).
	^direction = (0 @ 0) 
		ifTrue: [1 @ 0] 
		ifFalse: [direction unitVector]! !

!LineFigure methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	super displayOn: aGraphicsContext.
	annotations do: [:each | each displayOn: aGraphicsContext]! !

!LineFigure methodsFor: 'initialize-release'!

initialize
	super initialize.
	annotations := #()! !

!LineFigure methodsFor: 'private'!

computePreferredBounds
	^annotations inject: super computePreferredBounds
		into: [:sum :each | sum merge: each bounds]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

LineFigure class
	instanceVariableNames: ''!


!LineFigure class methodsFor: 'instance creation'!

connect: aFigure to: anotherFigure 
	| figure |
	figure := super connect: aFigure to: anotherFigure.
	figure addStopArrow.
	^figure! !

CachedFigure subclass: #ImageFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
ImageFigure comment:
'ImageFigure is a cached figure that can be used to grab images from the screen.

'!


!ImageFigure methodsFor: 'initialize-release'!

flushCaches
	"Do nothing, if our caches are flushed we cannot rebuild them."! !

!ImageFigure methodsFor: 'private'!

fillCache
	"Get the image from the user to display."

	^Image fromUser! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ImageFigure class
	instanceVariableNames: ''!


!ImageFigure class methodsFor: 'instance creation'!

createAt: aPoint 
	^(self new)
		origin: aPoint;
		yourself! !

Figure subclass: #RectangleFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
RectangleFigure comment:
'RectangleFigure is a figure that displays a rectangle.
'!


!RectangleFigure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	"Display an ellipse on aGraphicsContext. If the ellipse figure is not transparent then 
	display a filled ellipse using the current fillColor."

	self isOpaque ifTrue: [self displayFilledOn: aGraphicsContext].
	self lineWidth > 0 ifTrue: [self displayOutlineOn: aGraphicsContext]!

displayFilledOn: aGraphicsContext 
	aGraphicsContext paint: self fillColor.
	self bounds displayFilledOn: aGraphicsContext!

displayOutlineOn: aGraphicsContext 
	aGraphicsContext lineWidth: self lineWidth.
	aGraphicsContext paint: self lineColor.
	self bounds displayStrokedOn: aGraphicsContext! !

!RectangleFigure methodsFor: 'initialize-release'!

rectangle: aRectangle 
	bounds := aRectangle.
	self opaque! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

RectangleFigure class
	instanceVariableNames: ''!


!RectangleFigure class methodsFor: 'instance creation'!

createAt: aPoint 
	^self rectangle: (aPoint extent: 0 @ 0)!

rectangle: aRectangle 
	^(self new)
		rectangle: aRectangle;
		yourself! !

RectangleFigure subclass: #RoundedRectangleFigure
	instanceVariableNames: 'inset '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
RoundedRectangleFigure comment:
'RoundedRectangleFigure is a rectangle that has rounded corners.

Instance Variables:
	inset	<Point>	the offset for the ellipse that draws the rounded corners (if greater than our extent // 2, then we are an ellipse)

'!


!RoundedRectangleFigure methodsFor: 'accessing'!

handles
	| handles |
	handles := OrderedCollection withAll: super handles.
	handles add: ((TrackHandle on: self at: #insetPosition)
				moveBlock: [:aPoint | self inset: (aPoint - self origin max: 0 @ 0)];
				yourself).
	^handles! !

!RoundedRectangleFigure methodsFor: 'displaying'!

displayFilledOn: aGraphicsContext 
	| xinset yinset |
	aGraphicsContext paint: self fillColor.
	xinset := (self inset x - 1) @ 0.
	(self origin + xinset corner: self corner - xinset) 
		displayFilledOn: aGraphicsContext.
	yinset := 0 @ (self inset y - 1).
	(self origin + yinset corner: self corner - yinset) 
		displayFilledOn: aGraphicsContext.
	self arcs do: [:each | each displayFilledOn: aGraphicsContext]!

displayOutlineOn: aGraphicsContext 
	| xinset yinset |
	aGraphicsContext
		lineWidth: self lineWidth;
		paint: self lineColor.
	xinset := (self inset x - 1) @ 0.
	(LineSegment from: self origin + xinset to: self topRight - xinset) 
		displayStrokedOn: aGraphicsContext.
	(LineSegment from: self bottomLeft + xinset to: self corner - xinset) 
		displayStrokedOn: aGraphicsContext.
	yinset := 0 @ (self inset y - 1).
	(LineSegment from: self origin + yinset to: self bottomLeft - yinset) 
		displayStrokedOn: aGraphicsContext.
	(LineSegment from: self topRight + yinset to: self corner - yinset) 
		displayStrokedOn: aGraphicsContext.
	self arcs do: [:each | each displayStrokedOn: aGraphicsContext]! !

!RoundedRectangleFigure methodsFor: 'initialize-release'!

initialize
	super initialize.
	inset := 20 @ 20! !

!RoundedRectangleFigure methodsFor: 'private'!

arcs
	(inset x <= 0 or: [inset y <= 0]) ifTrue: [^#()].
	^Array 
		with: (EllipticalArc 
				boundingBox: (self origin corner: self origin + (self inset * 2))
				startAngle: 180
				sweepAngle: 90)
		with: (EllipticalArc 
				boundingBox: (self corner - (self inset * 2) corner: self corner)
				startAngle: 0
				sweepAngle: 90)
		with: (EllipticalArc 
				boundingBox: (Rectangle 
						vertex: self bottomLeft + (self inset x @ self inset y negated * 2) 
						vertex: self bottomLeft)
				startAngle: 90
				sweepAngle: 90)
		with: (EllipticalArc 
				boundingBox: (Rectangle 
						vertex: self topRight + (self inset x negated @ self inset y * 2)
						vertex: self topRight)
				startAngle: 270
				sweepAngle: 90)!

inset
	inset x <= 0 ifTrue: [^0 @ 0].
	inset y <= 0 ifTrue: [^0 @ 0].
	^inset min: self extent // 2!

inset: aPoint
	inset := aPoint.
	self changed!

insetPosition
	^self origin + inset! !

!RoundedRectangleFigure methodsFor: 'testing'!

containsPoint: aPoint 
	(super containsPoint: aPoint) ifFalse: [^false].
	(aPoint x between: self origin x + self inset x
		and: self corner x - self inset x) ifTrue: [^true].
	(aPoint y between: self origin y + self inset y
		and: self corner y - self inset y) ifTrue: [^true].
	^self arcs 
		contains: [:each | each regionIntersects: (aPoint - 1 extent: 2 @ 2)]! !

VisualPart subclass: #LineAnnotation
	instanceVariableNames: 'positionSymbol directionSymbol isFilled '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
LineAnnotation comment:
'LineAnnotation is a visual component that when added to a LineFigure allows you draw special symbols on the end points of the line (e.g., arrows, circles, etc.).

Subclasses must implement the following messages:
	private
		graphic	returns a Geometric object that can be displayed

Instance Variables:
	directionSymbol	<#startDirection | #stopDirection>	a symbol that when performed on our container, returns a unit vector of our direction
	isFilled	<Boolean>	are we filled?
	positionSymbol	<#startPoint | #stopPoint>	where should we be displayed at (when performed on our container returns our position)

'!


!LineAnnotation methodsFor: 'accessing'!

isFilled
	^isFilled!

isFilled: aBoolean 
	isFilled := aBoolean.
	self changed! !

!LineAnnotation methodsFor: 'bounds accessing'!

bounds
	^self graphic bounds! !

!LineAnnotation methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	aGraphicsContext
		lineWidth: container lineWidth;
		paint: container lineColor.
	isFilled 
		ifTrue: [self graphic displayFilledOn: aGraphicsContext]
		ifFalse: [self graphic displayStrokedOn: aGraphicsContext]! !

!LineAnnotation methodsFor: 'initialize-release'!

initialize
	super initialize.
	positionSymbol := #stopPoint.
	directionSymbol := #stopDirection.
	isFilled := true! !

!LineAnnotation methodsFor: 'private'!

direction: aSymbol
	directionSymbol := aSymbol!

directionVector
	^container perform: directionSymbol!

graphic
	^self subclassResponsibility!

location
	^container perform: positionSymbol!

position: aSymbol
	positionSymbol := aSymbol! !

!LineAnnotation methodsFor: 'testing'!

containsPoint: aPoint 
	^isFilled 
		ifTrue: [self graphic regionIntersects: (aPoint - 1 corner: aPoint + 1)]
		ifFalse: [self graphic outlineIntersects: (aPoint - 1 corner: aPoint + 1)]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

LineAnnotation class
	instanceVariableNames: ''!


!LineAnnotation class methodsFor: 'instance creation'!

forStart
	^(self new)
		direction: #startDirection;
		position: #startPoint;
		yourself!

forStop
	^(self new)
		direction: #stopDirection;
		position: #stopPoint;
		yourself! !

LineAnnotation subclass: #CircleAnnotation
	instanceVariableNames: 'radius '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
CircleAnnotation comment:
'CircleAnnotation is a line annotation that displays a circle at the end of the line.

Instance Variables:
	radius	<Integer>	the radius of the circle to display

'!


!CircleAnnotation methodsFor: 'accessing'!

radius
	^radius!

radius: aNumber 
	radius := aNumber.
	self changed! !

!CircleAnnotation methodsFor: 'initialize-release'!

initialize
	super initialize.
	radius := 4! !

!CircleAnnotation methodsFor: 'private'!

graphic
	| center origin |
	origin := self location.
	center := origin - (self directionVector * radius).
	^EllipticalArc 
		boundingBox: (center - radius corner: center + radius + 1)! !

LineAnnotation subclass: #ArrowAnnotation
	instanceVariableNames: 'length width '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
ArrowAnnotation comment:
'ArrowAnnotation is a line annotation that displays an arrow at the end of the line.

Instance Variables:
	length	<Integer>	the length of the arrow
	width	<Integer>	our width

'!


!ArrowAnnotation methodsFor: 'accessing'!

length
	^length!

length: aNumber 
	length := aNumber.
	self changed!

width
	^width!

width: aNumber 
	width := aNumber.
	self changed! !

!ArrowAnnotation methodsFor: 'initialize-release'!

initialize
	super initialize.
	width := 5.
	length := 10! !

!ArrowAnnotation methodsFor: 'private'!

graphic
	| vector u aPoint stopPoint points |
	vector := self directionVector.
	u := vector normal.
	stopPoint := self location.
	aPoint := stopPoint - (vector * length).
	points := Array 
				with: aPoint + (u * width)
				with: stopPoint
				with: aPoint - (u * width).
	isFilled 
		ifTrue: 
			[points := points 
						, (Array with: stopPoint - (vector * 0.8 * length) with: points first)].
	^Polyline vertices: points! !

ArrowAnnotation subclass: #DiamondAnnotation
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
DiamondAnnotation comment:
'DiamondAnnotation is a line annotation that displays a diamond shape at the end of line (similar to those in some case tool diagrams).'!


!DiamondAnnotation methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	isFilled ifTrue: [^super displayOn: aGraphicsContext].
	aGraphicsContext
		lineWidth: container lineWidth;
		paint: container fillColor.
	self graphic displayFilledOn: aGraphicsContext.
	aGraphicsContext paint: container lineColor.
	self graphic displayStrokedOn: aGraphicsContext! !

!DiamondAnnotation methodsFor: 'initialize-release'!

initialize
	super initialize.
	isFilled := false! !

!DiamondAnnotation methodsFor: 'private'!

graphic
	| vector u aPoint stopPoint points |
	vector := self directionVector.
	u := vector normal.
	stopPoint := self location.
	aPoint := stopPoint - (vector * length).
	points := (Array new: 5)
				at: 1 put: stopPoint;
				at: 2 put: aPoint - (u * width);
				at: 3 put: stopPoint - (vector * 2 * length);
				at: 4 put: aPoint + (u * width);
				at: 5 put: stopPoint;
				yourself.
	^Polyline vertices: points! !

EllipseFigure subclass: #ArcFigure
	instanceVariableNames: 'startAngle stopAngle '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
ArcFigure comment:
'ArcFigure is a figure that represents an EllipticalArc.

Instance Variables:
	startAngle	<Integer>	start angle for our ellipse
	stopAngle	<Integer>	start angle for our ellipse
'!


!ArcFigure methodsFor: 'accessing'!

handles
	| handles |
	handles := OrderedCollection withAll: super handles.
	handles
		add: ((TrackHandle on: self at: #startAnglePosition)
					moveBlock: [:aPoint | self trackStartAngle: aPoint];
					yourself);
		add: ((TrackHandle on: self at: #stopAnglePosition)
					moveBlock: [:aPoint | self trackStopAngle: aPoint];
					yourself).
	^handles!

startAngle
	^startAngle min: stopAngle!

startAngle: anInteger 
	startAngle := anInteger.
	self changed!

startAnglePosition
	^self positionForAngle: startAngle!

stopAngle
	^stopAngle max: startAngle!

stopAngle: anInteger 
	stopAngle := anInteger.
	self changed!

stopAnglePosition
	^self positionForAngle: stopAngle!

sweepAngle
	^stopAngle = startAngle 
		ifTrue: [360]
		ifFalse: [self stopAngle - self startAngle]!

trackStartAngle: aPoint 
	self startAngle: (self convertThetaToAngle: (self normalizePoint: aPoint) theta)!

trackStopAngle: aPoint 
	self stopAngle: (self convertThetaToAngle: (self normalizePoint: aPoint) theta)! !

!ArcFigure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	super displayFigureOn: aGraphicsContext.
	self lineWidth > 0 ifFalse: [^self].
	aGraphicsContext
		lineWidth: self lineWidth;
		paint: self lineColor.
	(LineSegment from: self center to: self startAnglePosition) 
		displayStrokedOn: aGraphicsContext.
	(LineSegment from: self center to: self stopAnglePosition) 
		displayStrokedOn: aGraphicsContext! !

!ArcFigure methodsFor: 'initialize-release'!

ellipse: aRect 
	super ellipse: aRect.
	startAngle := 0.
	stopAngle := 360! !

!ArcFigure methodsFor: 'private'!

convertAngleToTheta: anAngle 
	^anAngle / 360.0 * (2 * Double pi)!

convertThetaToAngle: aNumber 
	^(aNumber * 360 / (2 * Double pi)) rounded!

normalizePoint: aPoint 
	| vector extent |
	extent := self extent.
	(extent x <= 0 or: [extent y <= 0]) ifTrue: [^0 @ 0].
	vector := aPoint - self center.
	vector := (vector x asFloat / extent x) @ (vector y asFloat / extent y).
	0 @ 0 = vector ifTrue: [^0 @ 0].
	^vector!

positionForAngle: anAngle 
	| theta |
	theta := self convertAngleToTheta: anAngle.
	^self center 
		+ ((Point r: self extent x / 2.0 theta: theta) x 
				@ (Point r: self extent y / 2.0 theta: theta) y) rounded! !

!ArcFigure methodsFor: 'testing'!

containsPoint: aPoint 
	"Overriden, because regionIntersects: has a bug for small arcs"

	| outer angle strtAngle |
	self isOpaque ifFalse: [^super containsPoint: aPoint].
	self center = aPoint ifTrue: [^true].
	angle := self 
				convertThetaToAngle: (self normalizePoint: aPoint rounded) theta.
	strtAngle := self startAngle.
	(angle between: strtAngle and: strtAngle + self sweepAngle) 
		ifFalse: [^false].
	outer := self positionForAngle: angle.
	^aPoint rounded between: (self center min: outer)
		and: (self center max: outer)! !
TextFigure initialize!


