ExternalWriteStream subclass: #LatexStream
	instanceVariableNames: 'generateTableOfContents '
	classVariableNames: 'CharacterMap '
	poolDictionaries: ''
	category: 'Formatted-Streams'!
LatexStream comment:
'I translate Smalltalk into LaTeX.

Author:
	Carl McConnell
	mcconnel@cs.uiuc.edu
	Department of Computer Science
	University of Illinois at Urbana-Champaign
	5/90'!


!LatexStream methodsFor: 'beginning/ending'!

beginCategory: category
        self beginDocument.
        self titlePage: category.
        self tableOfContents.!

beginClass
	self beginDocument.
	self blankLines: 3!

endCategory
	self endDocument!

endClass
	self endDocument!

tableOfContents
        self generateTableOfContents: true.
        self line: '\pagenumbering{roman}'.
        self line: '\setcounter{page}{1}'.
        self line: '\tableofcontents'.
        self line: '\clearpage'.
        self line: '\pagenumbering{arabic}'.
        self line: '\setcounter{page}{1}'.
        self line: '%'!

titlePage: category
        self line: '\title{{\bf ' , category , '}}'.
        self line: '\author{' , self version , '}'.
        self line: '\date{' , Date today printString , ' ' , Time now printString , '}'.
        self line: '\maketitle'! !

!LatexStream methodsFor: 'printing'!

printClassComment: aString 
	aString isEmpty ifTrue: [^self].
	self blankLines: 1.
	self beginFont: '\it'.
	self printSmalltalkString: aString.
	self endFont; cr!

printClassDefinitionFor: aString from: aTable 
	self line: '\begin{center}'.
	self beginFont: '\Large \bf'.
	self nextPutAll: aString.
	self endFont; cr.
	self line: '\end{center}'.
	self printClassHeading: aString.
	self line: '\begin{tabular}{ll}'.
	aTable do: 
		[:assoc | 
		self nextPutAll: assoc key.
		self nextPutAll: '	& '.
		assoc value isEmpty
			ifTrue: [self beginFont: '\it'; nextPutAll: 'none'; endFont]
			ifFalse: [self nextPutAll: '\parbox[t]{4.0in}{\raggedright '; beginFont: '\bf'; nextPutAll: assoc value; endFont; nextPut: $}].
		assoc = aTable last
			ifTrue: [self cr]
			ifFalse: [self line: ' \\']].
	self line: '\end{tabular}'!

printClassHeading: aString
	self blankLines: 3.
	self line: '\markright{{\bf ' , aString , '} \hfill}'.
	self generateTableOfContents ifTrue: [self line: '\addcontentsline{toc}{section}{' , aString , '}'].!

printMethod: aSelector withBody: aString 
	self beginFont: '\bf'; printSmalltalkString: aSelector; endFont.
	self cr; line: '\nopagebreak'.
	self printSmalltalkString: aString.
	self cr!

printProtocolHeader: aString 
	self beginFont: '\sl'.
	self nextPutAll: 'Protocol for '; nextPutAll: aString.
	self endFont; cr.
	self line: '\nopagebreak'! !

!LatexStream methodsFor: 'separating'!

anotherClass
	self line: '%'.
	self line: '\clearpage'!

anotherMethod
	self blankLines: 2!

anotherProtocol
	self blankLines: 3! !

!LatexStream methodsFor: 'private'!

blankLines: n 
	self cr.
	self nextPutAll: '\addvspace{'; print: n; nextPutAll: 'ex}'; cr.
	self line: '\noindent'!

generateTableOfContents
	^generateTableOfContents isNil ifTrue: [false] ifFalse: [generateTableOfContents]!

generateTableOfContents: aBoolean
	generateTableOfContents := aBoolean!

line: aString
	self nextPutAll: aString; cr!

version
	"The version name contains a backslash, hence this hack."

	^Smalltalk version copyReplaceAll: '\' with: '$\backslash$'! !

!LatexStream methodsFor: 'private-printing'!

printSmalltalkString: aString 
	| inComment |
	inComment := false.
	aString
		do: 
			[:aCharacter | 
			| output |
			output := CharacterMap at: aCharacter asInteger.
			output notNil
				ifTrue: [self nextPutAll: output]
				ifFalse: ["This character requires special handling."
					aCharacter = $"
						ifTrue: 
							[inComment
								ifTrue: [self nextPutAll: ''''''; endFont]
								ifFalse: [self beginFont: '\it'; nextPutAll: '``'].
							inComment := inComment not]
						ifFalse: [self error: 'unrecognized special character']]]! !

!LatexStream methodsFor: 'private-beginning/ending'!

beginDocument
	self line: '\documentstyle[11pt]{report}'.
	self line: '\oddsidemargin 0in \evensidemargin 0in'.
	self line: '\marginparwidth 0in \marginparsep 0in'.
	self line: '\topmargin 0in \topskip 0in'.
	self line: '\headheight 0.25in \headsep 0.25in'.
	self line: '\footheight 0in \footskip 0.25in'.
	self line: '\textheight 8.25in \textwidth 6.5in'.
	self line: '\raggedbottom \sloppy'.
	self line: '\pagestyle{myheadings}'.
	self line: '%'.
	self line: '\begin{document}'!

beginFont: aString
	self nextPut: ${.
	self nextPutAll: aString; space!

endDocument
	self line: '\end{document}'!

endFont
	self nextPut: $}! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

LatexStream class
	instanceVariableNames: ''!


!LatexStream class methodsFor: 'class initialization'!

at: aCharacter put: anObject
	CharacterMap at: aCharacter asInteger put: anObject.
	^anObject!

initialize
	"Initialize the table for mapping characters to LaTeX. Characters 
	for which special (or no) action is to be taken have nil entries.
 	We won't have to handle NUL characters, so the table size is 127."

	CharacterMap := Array new: 127.
	1 to: CharacterMap size 
		do: [:i | CharacterMap at: i put: (String with: (Character value: i))].
	"Characters requiring special handling are mapped to nil."
	self at: $" put: nil.
	self at: Character cr put: (' \\$' replaceAll: $$ with: Character cr).
	self at: Character tab put: '\hspace*{4ex} '.
	#($_ $# $$ $% $& ${ $}) do: [:ch | self at: ch put: (String with: $\ with: ch)].
	#($| $> $< $+ $- $/ $* $@) do: [:ch | self at: ch put: (String with: $$ with: ch with: $$)].
	self at: $~ put: '{\char126}'.
	self at: $^ put: '{\char94}'.
	self at: $\ put: '$\backslash$'! !

!LatexStream class methodsFor: 'instance creation'!

for: aString do: aBlock 
	| fileName fileStream |
	fileName := Dialog
			requestFileName: 'Print LaTeX on'
			default: (Filename defaultClass suggest: (Filename defaultClass filterSeps: aString, '.tex')). 
	fileName = '' ifTrue: [^nil].
	fileStream :=  fileName asFilename latexStream.
	[aBlock value: fileStream]
		valueNowOrOnUnwindDo: [fileStream close]! !

ExternalWriteStream subclass: #RTFStream
	instanceVariableNames: 'generateTableOfContents '
	classVariableNames: 'CharacterMap '
	poolDictionaries: ''
	category: 'Formatted-Streams'!
RTFStream comment:
'I translate Smalltalk into RTF.

Author:
	Carl McConnell
	mcconnel@cs.uiuc.edu
	Department of Computer Science
	University of Illinois at Urbana-Champaign
	5/90'!


!RTFStream methodsFor: 'beginning/ending'!

beginCategory: category 
	self generateTableOfContents: true.
	self beginDocument!

beginClass
	self beginDocument.
	self blankLines: 3!

endCategory
	self endDocument!

endClass
	self endDocument!

tableOfContents
	self generateTableOfContents: true.! !

!RTFStream methodsFor: 'printing'!

printClassComment: aString 
	aString isEmpty ifTrue: [^self].
	self line: '{\cs17\i '; printSmalltalkString: aString; line: '\par }'!

printClassDefinitionFor: aString from: aTable 
	self printClassHeading: aString.
	self blankLines: 1.
	1 to: aTable size
		do: 
			[:i | 
			| assoc first last |
			assoc := aTable at: i.
			first := i == 1
						ifTrue: ['30']
						ifFalse: ['15'].
			last := i == aTable size
						ifTrue: ['30']
						ifFalse: ['15'].
			self nextPutAll: '\trowd \trgaph108\trleft-108\trbrdrt\brdrs\brdrw30 \trbrdrl\brdrs\brdrw30 
\trbrdrb\brdrs\brdrw30 \trbrdrr\brdrs\brdrw30 \trbrdrh\brdrs\brdrw15 \trbrdrv\brdrs\brdrw15 
\clbrdrt\brdrs\brdrw'.
			self nextPutAll: first; nextPutAll: ' \clbrdrl\brdrs\brdrw30 \clbrdrb\brdrs\brdrw'; nextPutAll: last; nextPutAll: ' \clbrdrr\brdrs\brdrw15 
\cellx2340\clbrdrt\brdrs\brdrw'; nextPutAll: first; nextPutAll: ' \clbrdrl\brdrs\brdrw15 \clbrdrb\brdrs\brdrw'; nextPutAll: last; nextPutAll: ' \clbrdrr\brdrs\brdrw30 \cellx8748 
\pard \widctlpar\intbl '; nextPutAll: assoc key; nextPutAll: '\cell '; nextPutAll: assoc value; nextPutAll: '\cell \pard \widctlpar\intbl \row '].
	self nextPutAll: '\pard \widctlpar \par'!

printClassHeading: aString
	self nextPutAll: '\par \pard\plain \s16\sb240\sa60\keepn\pagebb\widctlpar \b\f5\fs28\kerning28 {\cs15 '.
	self nextPutAll: aString.
	self nextPutAll: ' \par }\pard\plain \widctlpar \f4\fs20'!

printMethod: aSelector withBody: aString 
	self nextPutAll: '{\cs18\b '; nextPutAll: aSelector; nextPutAll: '} '.
	self printSmalltalkString: aString; nextPutAll: '\par'; cr!

printProtocolHeader: aString 
	self nextPutAll: '{\cs19\i\b '; nextPutAll: 'Protocol for '; nextPutAll: aString; line: '} '! !

!RTFStream methodsFor: 'separating'!

anotherClass!

anotherMethod
	self blankLines: 1!

anotherProtocol
	self blankLines: 2! !

!RTFStream methodsFor: 'private'!

blankLines: n 
	n
		timesRepeat: 
			[self nextPutAll: '\par '; cr]!

generateTableOfContents
	^generateTableOfContents isNil ifTrue: [false] ifFalse: [generateTableOfContents]!

generateTableOfContents: aBoolean
	generateTableOfContents := aBoolean!

line: aString
	self nextPutAll: aString; nextPutAll: '\par'; cr!

version
	"The version name contains a backslash, hence this hack."

	^Smalltalk version copyReplaceAll: '\' with: '\\'! !

!RTFStream methodsFor: 'private-printing'!

printSmalltalkString: aString 
	| inComment |
	inComment := false.
	aString
		do: 
			[:aCharacter | 
			| output |
			output := CharacterMap at: aCharacter asInteger.
			output notNil
				ifTrue: [self nextPutAll: output]
				ifFalse: ["This character requires special handling."
					aCharacter = $"
						ifTrue: 
							[inComment
								ifTrue: [self nextPutAll: '"}']
								ifFalse: [self nextPutAll: '{\cs17\i '; nextPut: $"].
							inComment := inComment not]
						ifFalse: [self error: 'unrecognized special character']]].
	inComment ifTrue: [self nextPutAll: '"}']! !

!RTFStream methodsFor: 'private-beginning/ending'!

beginDocument
	self line: '{\rtf1\ansi \deff4\deflang1033{\fonttbl{\f4\froman\fcharset0\fprq2 Times New Roman;}{\f5\fswiss\fcharset0\fprq2 Arial;}{\f11\fmodern\fcharset0\fprq1 Courier New;}}{\colortbl;\red0\green0\blue0;\red0\green0\blue255;
\red0\green255\blue255;\red0\green255\blue0;\red255\green0\blue255;\red255\green0\blue0;\red255\green255\blue0;\red255\green255\blue255;\red0\green0\blue128;\red0\green128\blue128;\red0\green128\blue0;\red128\green0\blue128;\red128\green0\blue0;
\red128\green128\blue0;\red128\green128\blue128;\red192\green192\blue192;}{\stylesheet{\widctlpar \f4\fs20 \snext0 Normal;}{\s1\sb240\sa60\keepn\widctlpar \b\f5\fs28\kerning28 \sbasedon0\snext0 heading 1;}{\*\cs10 \additive Default Paragraph Font;}{\*
\cs15 \additive\f5 \sbasedon10 Code;}{\s16\sb240\sa60\keepn\pagebb\widctlpar \b\f5\fs28\kerning28 \sbasedon1\snext0 Class;}{\*\cs17 \additive\i \sbasedon10 Comment;}{\*\cs18 \additive\b \sbasedon10 Message selector;}{\s19\sb360\widctlpar\tqr\tldot\tx8640 
\b\caps\f5 \sbasedon0\snext0 toc 1;}{\s20\sb240\widctlpar\tqr\tldot\tx8640 \b\f4\fs20 \sbasedon0\snext0 toc 2;}{\s21\li200\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 toc 3;}{\s22\li400\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 
toc 4;}{\s23\li600\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 toc 5;}{\s24\li800\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 toc 6;}{\s25\li1000\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 toc 7;}{
\s26\li1200\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 toc 8;}{\s27\li1400\widctlpar\tqr\tldot\tx8640 \f4\fs20 \sbasedon0\snext0 toc 9;}{\s28\widctlpar\tqc\tx4320\tqr\tx8640 \f4\fs20 \sbasedon0\snext28 footer;}{\*\cs29 \additive\sbasedon10 
page number;}{\*\cs30 \additive\f11\fs20 \sbasedon10 CODE;}}{\info{\author John Brant}{\operator John Brant}{\creatim\yr1996\mo6\dy28\hr18\min3}{\revtim\yr1996\mo6\dy29\min3}{\version2}{\edmins3}{\nofpages3}{\nofwords46}{\nofchars264}{\*\company  }
{\vern57431}}\widowctrl\ftnbj\aenddoc\hyphcaps0\formshade \fet0\sectd \linex0\endnhere\titlepg {\footer \pard\plain \s28\widctlpar\tqc\tx4320\tqr\tx8640\pvpara\phmrg\posxc\posy0 \f4\fs20 {\field{\*\fldinst {\cs29 PAGE  }}{\fldrslt {\cs29\lang1024 2}}}{
\cs29 
\par }\pard \s28\widctlpar\tqc\tx4320\tqr\tx8640 
\par }{\*\pnseclvl1\pnucrm\pnstart1\pnindent720\pnhang{\pntxta .}}{\*\pnseclvl2\pnucltr\pnstart1\pnindent720\pnhang{\pntxta .}}{\*\pnseclvl3\pndec\pnstart1\pnindent720\pnhang{\pntxta .}}{\*\pnseclvl4\pnlcltr\pnstart1\pnindent720\pnhang{\pntxta )}}{\*\pnseclvl5
\pndec\pnstart1\pnindent720\pnhang{\pntxtb (}{\pntxta )}}{\*\pnseclvl6\pnlcltr\pnstart1\pnindent720\pnhang{\pntxtb (}{\pntxta )}}{\*\pnseclvl7\pnlcrm\pnstart1\pnindent720\pnhang{\pntxtb (}{\pntxta )}}{\*\pnseclvl8\pnlcltr\pnstart1\pnindent720\pnhang
{\pntxtb (}{\pntxta )}}{\*\pnseclvl9\pnlcrm\pnstart1\pnindent720\pnhang{\pntxtb (}{\pntxta )}}\pard\plain \s19\sb360\widctlpar\tqr\tldot\tx8640 \b\caps\f5 {\field\fldedit{\*\fldinst {\scaps\caps0  TOC \\o "1-3" \\t "Class,1" }}{\fldrslt {
\scaps\caps0\lang1024 Object\tab }{\field\flddirty{\*\fldinst {\scaps\caps0\lang1024  GOTOBUTTON _Toc360619918  }{\field\flddirty{\*\fldinst {\scaps\caps0\lang1024  PAGEREF _Toc360619918 }}{\fldrslt {\scaps\caps0\lang1024 1}}}}}{\scaps\caps0\lang1024 

\par }\pard\plain \s16\sb240\sa60\keepn\pagebb\widctlpar \b\f5\fs28\kerning28 }}'!

endDocument
	self line: ' \par }'! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

RTFStream class
	instanceVariableNames: ''!


!RTFStream class methodsFor: 'class initialization'!

at: aCharacter put: anObject
	CharacterMap at: aCharacter asInteger put: anObject.
	^anObject!

initialize
	"Initialize the table for mapping characters to LaTeX. Characters 
	for which special (or no) action is to be taken have nil entries. 
		We won't have to handle NUL characters, so the table size is 127."

	CharacterMap := Array new: 127.
	1 to: CharacterMap size do: [:i | CharacterMap at: i put: (String with: (Character value: i))].	

"Characters requiring special handling are mapped to nil."
	self at: $" put: nil.
	self at: $\ put: '\\'.
	self at: Character cr put: (String with: Character cr), '{\cs30\f11 \line }'! !

!RTFStream class methodsFor: 'instance creation'!

for: aString do: aBlock 
	| fileName fileStream |
	fileName := Dialog
			requestFileName: 'Print RTF on'
			default: (Filename defaultClass suggest: (Filename defaultClass filterSeps: aString, '.tex')). 
	fileName = '' ifTrue: [^nil].
	fileStream :=  fileName asFilename rtfStream.
	[aBlock value: fileStream]
		valueNowOrOnUnwindDo: [fileStream close]! !
LatexStream initialize!

RTFStream initialize!


