EnvironmentEditor subclass: #DoubleEnvironmentEditor
	instanceVariableNames: 'classEnvironment '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Code-Printing'!
DoubleEnvironmentEditor comment:
'DoubleEnvironmentEditor added behavior to EnvironmentEditor
specific for selecting classes and adding menus specific for the
Latex printing environment.  We wanted to be able to italicize
items that printed their definitions etc.

Instance Variables:
	classEnvironment	<Object>	description of classEnvironment

Original Design and Ideas by Jeff Barcalow (barcalow@uiuc.edu) and Joe Yoder (j-yoder@uiuc.edu)
Developed by Jeff Barcalow (barcalow@uiuc.edu) and Joe Yoder (j-yoder@uiuc.edu)
Upgraded to VW 3.0 by Joseph W. Yoder (j-yoder@uiuc.edu)

'!


!DoubleEnvironmentEditor methodsFor: 'accessing'!

classEnvironment
	classEnvironment isNil ifTrue: [classEnvironment := ClassEnvironment new].
	^classEnvironment! !

!DoubleEnvironmentEditor methodsFor: 'private'!

checkCategory: aCategory 
	(environment classNamesFor: aCategory)
		do: [:each | self classEnvironment addClass: (Smalltalk at: each)].
	super checkCategory: aCategory!

checkClass: aName 
	self classEnvironment addClass: (Smalltalk at: aName).
	super checkClass: aName!

uncheckCategory: aCategory 
	(environment classNamesFor: aCategory)
		do: [:each | self classEnvironment removeClass: (Smalltalk at: each)].
	super uncheckCategory: aCategory!

uncheckClass: aName 
	self classEnvironment removeClass: (Smalltalk at: aName).
	super uncheckClass: aName! !

!DoubleEnvironmentEditor methodsFor: 'actions'!

addClassToDef
	self updateDisplayAfter: [self classEnvironment addClass: self nonMetaClass]!

checkClass
	self classEnvironment addClass: self selectedClass.
	super checkClass!

removeClassFromDef
	self updateDisplayAfter: [self classEnvironment removeClass: self nonMetaClass]!

uncheckClass
	self classEnvironment removeClass: self selectedClass.
	super uncheckClass! !

!DoubleEnvironmentEditor methodsFor: 'interface opening'!

postBuildClassList: aBuilder 
	| visualBlock |
	visualBlock := 
			[:v :i | 
			| class labelString |
			class := Smalltalk at: (v sequence at: i).
			labelString := (class notNil and: [self classEnvironment includesClass: class])
						ifTrue: [(v displayStringAt: i) asText emphasizeAllWith: #italic]
						ifFalse: [v displayStringAt: i].
			(class notNil and: [self meta value
					ifTrue: [selectedEnvironment includesClass: class class]
					ifFalse: [selectedEnvironment includesClass: class]])
				ifTrue: [(LabelAndIcon with: labelString attributes: v textStyle) beCheckMark]
				ifFalse: [Label
						with: labelString
						attributes: v textStyle
						offset: 16 @ 0]].
	(aBuilder componentAt: #classList) widget visualBlock: [:v :i | BoundingWrapper on: (visualBlock value: v value: i)]; selectedVisualBlock: (self selectedVisualBlock: visualBlock)! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DoubleEnvironmentEditor class
	instanceVariableNames: ''!


!DoubleEnvironmentEditor class methodsFor: 'resources'!

classMenu
	| menu |
	menu := super classMenu.
	menu addItemGroupLabels: #('add definition' 'remove definition') values: #(#addClassToDef #removeClassFromDef).
	^menu! !

ApplicationModel subclass: #CodeSelector
	instanceVariableNames: 'environmentEditor '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Code-Printing'!
CodeSelector comment:
'CodeSelector pops up an Interface for selecting the
classes, protocols, and/or methods you would like to
print.  It uses the Environments from the RefactoryBrowser
developed by John Brant and Don Roberts.

Instance Variables:
	environmentEditor	<DoubleEnvironmentEditor>	description of environmentEditor

Original Design and Ideas by Jeff Barcalow (barcalow@uiuc.edu) and Joe Yoder (j-yoder@uiuc.edu)
Developed by Jeff Barcalow (barcalow@uiuc.edu) and Joe Yoder (j-yoder@uiuc.edu)
Minor Coding by John Brant (brant@cs.uiuc.edu)
Upgraded to VW 3.0 by Joseph W. Yoder (j-yoder@uiuc.edu)
'!


!CodeSelector methodsFor: 'accessing'!

environmentEditor
	environmentEditor isNil
		ifTrue: 
			[environmentEditor := DoubleEnvironmentEditor new.
			environmentEditor environment: RestrictedEnvironment new].
	^environmentEditor!

selectedClassEnvironment
	^self environmentEditor classEnvironment!

selectedEnvironment
	^self environmentEditor selectedEnvironment! !

!CodeSelector methodsFor: 'actions'!

cancel
	self closeRequest!

printsave
	self selectedEnvironment classNames isEmpty ifTrue: [^Dialog warn: 'You must first select something to print!!'].
	(CodePrinter for: self selectedEnvironment with: self selectedClassEnvironment) open! !

!CodeSelector methodsFor: 'updating'!

changeMode! !

!CodeSelector methodsFor: 'interface opening'!

postBuildWith: aBuilder 
	super postBuildWith: aBuilder.
	(self builder componentAt: #italics) widget label text
		emphasizeAllWith: #italic! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CodeSelector class
	instanceVariableNames: ''!


!CodeSelector class methodsFor: 'interface specs'!

windowSpec
	"UIPainter new openOnClass: self andSelector: #windowSpec"

	<resource: #canvas>
	^#(#FullSpec 
		#window: 
		#(#WindowSpec 
			#label: 'LaTeX Selector' 
			#min: #(#Point 320 280 ) 
			#bounds: #(#Rectangle 409 279 930 709 ) 
			#isEventDriven: true ) 
		#component: 
		#(#SpecCollection 
			#collection: #(
				#(#SubCanvasSpec 
					#layout: #(#LayoutFrame 0 0 0 0 0 1 -60 1 ) 
					#name: #environmentEditor 
					#flags: 0 
					#majorKey: #DoubleEnvironmentEditor 
					#minorKey: #windowSpec 
					#clientKey: #environmentEditor ) 
				#(#ActionButtonSpec 
					#layout: #(#AlignmentOrigin 0 0.591603 -3 1 0.5 1 ) 
					#name: #cancel 
					#model: #cancel 
					#label: 'Cancel' 
					#defaultable: true ) 
				#(#ActionButtonSpec 
					#layout: #(#AlignmentOrigin 0 0.368321 -3 1 0.5 1 ) 
					#name: #print 
					#model: #printsave 
					#label: 'Print/Save' 
					#defaultable: true ) 
				#(#LabelSpec 
					#layout: #(#AlignmentOrigin 0 0.490494 -39 1 0.5 1 ) 
					#name: #italics 
					#label: '*** Class names that are italicized will print their definitions' ) ) ) )! !

ApplicationModel subclass: #CodePrinter
	instanceVariableNames: 'environment fileType mode destination tmpDirectory scriptStream tmpFile classDefinitionEnvironment '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Code-Printing'!
CodePrinter comment:
'CodePrinter chooses which printer format that you want to print out.
You can choose to print out directory to a file.  It uses the Environments
from the RefactoryBrowser developed by John Brant and Don Roberts to
go through and selectively print out to whatever format you want.  The
possible output formats are ps, rtf, tex, dvi, and st.  Printing, PostScript
and DVI outputs are currently disabled if you are not on a unix machine
and it is assumed you have latex and dvips installed with your PATH
environment variable set if you are on a unix machine.

Instance Variables:
	destination		<ValueHolder on: String>	the output designated by the user
	environment		<EnvironmentEditor>	the actual environment that was
												selected by CodeSelector
	fileType			<aValueHoler on: Symbol>	the type of output file you want to
												print or save to (#rtf #tex #dvi ...)
	mode			<ValueHolder on: Symbol>	the print mode (#save or #print)
	tmpDirectory 	<aString> the directory where you want the output to be put
	scriptStream 	<aWriteStream> the Stream where the file goes
	tmpFile			<aFileName> the temorary output file name used to stream out on
	classDefinitionEnvironment <ClassEnvironment>	the classes that were selected to be
												by the CodeSelector

Original Design and Ideas by Jeff Barcalow (barcalow@uiuc.edu) and Joe Yoder (j-yoder@uiuc.edu)
Developed by Jeff Barcalow (barcalow@uiuc.edu) and Joe Yoder (j-yoder@uiuc.edu)
Minor Coding from John Brant (brant@cs.uiuc.edu)
Upgraded to VW 3.0 by Joseph W. Yoder (j-yoder@uiuc.edu)'!


!CodePrinter methodsFor: 'accessing'!

classDefinitionEnvironment
	^classDefinitionEnvironment!

classDefinitionEnvironment: anObject
	classDefinitionEnvironment := anObject!

environment
	^environment!

environment: anEnvironment 
	environment := anEnvironment! !

!CodePrinter methodsFor: 'actions-file processing'!

createSortBlock
	^SortedCollection sortBlock: 
			[:v :e | 
			| common e1 v1 |
			v = e or: 
					[(v includesBehavior: e) not and: 
							[(e includesBehavior: v) or: 
									[common := e commonSuperclass: v.
									e1 := e.
									[e1 superclass = common] whileFalse: [e1 := e1 superclass].
									v1 := v.
									[v1 superclass = common] whileFalse: [v1 := v1 superclass].
									v1 name < e1 name]]]]!

dvi
	self latex.
	scriptStream nextPutAll: 'latex output.tex;'; cr.
	scriptStream nextPutAll: 'latex output.tex;'; cr.
	tmpFile := 'output.dvi'.!

getBaseClasses
	^(self createSortBlock)
		addAll: (environment classNames collect: 
						[:sym | 
						| class |
						class := Smalltalk at: sym.
						(environment includesClass: class) ifTrue: [class]]);
		yourself!

getSelectedClasses
	| classes |
	classes := OrderedCollection new.
	self getBaseClasses do: 
			[:class | 
			classes add: class.
			(environment includesClass: class class) ifTrue: [classes add: class class]].
	^classes!

headlessLatex
	self save: #headlessLatex for: 'output.tex'!

headlessSaveOn: fileStream 
	| classes classDefs |
	classes := self getSelectedClasses.
	classDefs := self classDefinitionEnvironment classNames asOrderedCollection collect: [:each | Smalltalk at: each].
	classes
		do: 
			[:class | 
			fileStream anotherClass.
			(classDefs includes: class)
				ifTrue: 
					[self saveDefinitionFor: class on: fileStream.
					classDefs remove: class]
				ifFalse: [fileStream printClassHeading: class printString].
			(environment protocolsFor: class)
				do: 
					[:heading | 
					fileStream anotherProtocol.
					fileStream printProtocolHeader: heading.
					(environment selectorsFor: heading in: class)
						do: 
							[:sel | 
							fileStream anotherMethod.
							class fileOutMethodChunk: sel on: fileStream]]].
	classDefs
		do: 
			[:class | 
			fileStream anotherClass.
			self saveDefinitionFor: class on: fileStream]!

latex
	self save: #latex for: 'output.tex'!

mpage2
	self postscript.
	scriptStream
		 nextPutAll: 'mpage -2P output.ps > output.ps2;';
		 cr;
		 nextPutAll: 'mv output.ps2 output.ps;';
		 cr!

mpage4
	self postscript.
	scriptStream
		 nextPutAll: 'mpage -4P output.ps > output.ps4;';
		 cr;
		 nextPutAll: 'mv output.ps4 output.ps;';
		 cr!

postscript
	self dvi.
	scriptStream nextPutAll: 'dvips -q -o output.ps output.dvi;'; cr.
	tmpFile := 'output.ps'.!

rtf
	self save: #rtf for: 'output.rtf'!

save: type for: filename 
	| stream |
	stream := self stream: type for: filename.
	[type = #headlessLatex ifTrue: [^self headlessSaveOn: stream].
	type = #source ifTrue: [^self sourceSaveOn: stream].
	self saveOn: stream]
		valueNowOrOnUnwindDo: [stream close]!

saveDefinitionFor: class on: fileStream 
	fileStream printClassDefinitionFor: class printString from: class definitionAsTable.
	fileStream printClassComment: class comment!

saveOn: fileStream 
	fileStream beginDocument.
	fileStream tableOfContents.
	self headlessSaveOn: fileStream.
	fileStream endDocument!

source
	self save: #source for: 'output.st'!

sourceSaveOn: fileStream 
	| classes changeSet classDefs |
	changeSet := ChangeSet new.
	classes := self getSelectedClasses.
	classDefs := self classDefinitionEnvironment classNames collect: [:each | Smalltalk at: each].
	classDefs
		do: 
			[:class | 
			changeSet changeClass: class.
			changeSet commentClass: class].
	classes do: [:class | (environment selectorsForClass: class)
			do: [:sel | changeSet addSelector: sel class: class]].
	changeSet fileOutOn: fileStream!

stream: type for: filename 
	(filename includes: $:)
		ifTrue: [tmpFile := filename asFilename]
		ifFalse: 
			[tmpDirectory isNil
				ifTrue: [tmpFile := '.']
				ifFalse: [tmpFile := tmpDirectory].
			tmpFile := tmpFile asFilename construct: filename].
	tmpFile head asFilename exists ifFalse:[^Dialog warn: 'The directory ' , tmpFile head , ' doesn''t exist!!'].
	type = #source ifTrue: [^SourceCodeStream write: tmpFile].
	type = #rtf ifTrue: [^tmpFile asFilename rtfStream].
	type = #headlessLatex ifTrue: [^tmpFile asFilename latexStream].
	type = #latex ifTrue: [^tmpFile asFilename latexStream].
	self error: 'Invalid stream type'! !

!CodePrinter methodsFor: 'actions-script'!

finishUp
	(tmpDirectory asFilename construct: 'latexscript')
		setProtection: 493.
	self runScript!

makeUniqueTmpDirectory
	| i name |
	i := 0.
	
	[name := '/tmp/' , i printString.
	name asFilename exists]
		whileTrue: [i := i + 1].
	name asFilename makeDirectory.
	^name!

moveScript
	scriptStream
		 nextPutAll: 'mv ';
		 nextPutAll: tmpFile;
		 nextPut: Character space;
		 nextPutAll: (destination value asFilename directory constructString: destination value asFilename tail);
		 cr!

printScript
	destination value: (destination value copyUpTo: $ ).
	destination value isEmpty
		ifTrue: [scriptStream nextPutAll: 'lp output.ps'; cr]
		ifFalse: [scriptStream
				 nextPutAll: 'lp -d';
				 nextPutAll: destination;
				 nextPutAll: ' output.ps';
				 cr]!

removeScript
	scriptStream
		 nextPutAll: 'cd /tmp'; 
		 cr;
		 nextPutAll: 'rm -rf ';
		 nextPutAll: tmpDirectory.!

runScript
	| process |
	process := UnixProcess forkJob: 'csh' arguments: (Array with: (tmpDirectory asFilename constructString: 'latexscript')).
	process wait!

setup
	tmpDirectory := self makeUniqueTmpDirectory.
	scriptStream := (tmpDirectory asFilename construct: 'latexscript') writeStream.
	scriptStream nextPutAll: 'cd '; nextPutAll: tmpDirectory; cr.! !

!CodePrinter methodsFor: 'actions'!

cancel
	self closeRequest!

ok
	Cursor wait showWhile: [self perform: mode value]!

print
	self setup.
	
	[self perform: fileType value.
	self printScript.
	self removeScript]
		valueNowOrOnUnwindDo: 
			[scriptStream close.
			self finishUp].
	self closeRequest!

save
	destination value: (destination value copyUpTo: Character space).
	destination value isEmpty ifTrue: [^Dialog warn: 'Must provide a filename'].
	destination value asFilename exists
		ifTrue: 
			[| answer |
			answer := Dialog confirm: 'A file with that name already exists.  \Do you want to overwrite that file?' withCRs initialAnswer: false.
			answer ifFalse: [^self]].
	self isUnixPlatform ifFalse: [self save: fileType value for: destination value]
		ifTrue: 
			[self setup.
			
			[self perform: fileType value.
			self moveScript.
			self removeScript]
				valueNowOrOnUnwindDo: 
					[scriptStream close.
					self finishUp]].
	self closeRequest! !

!CodePrinter methodsFor: 'aspects'!

destination
	"This method was generated by UIDefiner.  Any edits made here
	may be lost whenever methods are automatically defined.  The
	initialization provided below may have been preempted by an
	initialize method."

	^destination isNil
		ifTrue:
			[destination := String new asValue]
		ifFalse:
			[destination]!

fileType
	"This method was generated by UIDefiner.  Any edits made here
	may be lost whenever methods are automatically defined.  The
	initialization provided below may have been preempted by an
	initialize method."

	^fileType isNil
		ifTrue:
			[fileType := #postscript asValue]
		ifFalse:
			[fileType]!

mode
	"This method was generated by UIDefiner.  Any edits made here
	may be lost whenever methods are automatically defined.  The
	initialization provided below may have been preempted by an
	initialize method."

	^mode isNil
		ifTrue:
			[mode := #save asValue]
		ifFalse:
			[mode]! !

!CodePrinter methodsFor: 'testing'!

isUnixPlatform
	^(ExternalInterface currentPlatform at: 1) = #unix! !

!CodePrinter methodsFor: 'interface opening'!

postBuildWith: aBuilder 
	super postBuildWith: aBuilder.
	self isUnixPlatform ifTrue: [^self].
	self mode value: #save.
	self fileType value: #latex.
	(aBuilder componentAt: #print) disable.
	(aBuilder componentAt: #save) disable.
	(aBuilder componentAt: #dvi) disable.
	(aBuilder componentAt: #postscript) disable.
	(aBuilder componentAt: #mpage2) disable.
	(aBuilder componentAt: #mpage4) disable! !

!CodePrinter methodsFor: 'updating'!

changeMode
	mode value = #save
		ifTrue: [self saveEnables]
		ifFalse: [self printDisables]!

filenameLabel
	| txt |
	txt := 'Filename: ' asText allBold.
	^Label with: txt!

printDisables
	(builder componentAt: #source) disable.
	(builder componentAt: #latex) disable.
	(builder componentAt: #rtf) disable.
	(builder componentAt: #headlessLatex) disable.
	(builder componentAt: #dvi) disable.
	(builder componentAt: #outputDestination)
		label: self printerLabel.
	(fileType value ~= #postscript) & (fileType value ~= #mpage2) & (fileType value ~= #mpage4) 
		ifTrue: [fileType value: #postscript].
	destination value: String new!

printerLabel
	| txt |
	txt := 'Printer: ' asText allBold.
	^Label with: txt!

saveEnables
	(builder componentAt: #source) enable.
	(builder componentAt: #latex) enable.
	(builder componentAt: #rtf) enable.
	(builder componentAt: #headlessLatex) enable.
	(builder componentAt: #dvi) enable.
	(builder componentAt: #outputDestination)
		label: self filenameLabel.
	destination value: String new.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CodePrinter class
	instanceVariableNames: ''!


!CodePrinter class methodsFor: 'interface specs'!

windowSpec
	"UIPainter new openOnClass: self andSelector: #windowSpec"

	<resource: #canvas>
	^#(#FullSpec 
		#window: 
		#(#WindowSpec 
			#label: 'LaTeX Print Window' 
			#min: #(#Point 248 333 ) 
			#max: #(#Point 248 333 ) 
			#bounds: #(#Rectangle 389 509 637 842 ) 
			#isEventDriven: true ) 
		#component: 
		#(#SpecCollection 
			#collection: #(
				#(#RadioButtonSpec 
					#layout: #(#Point 57 15 ) 
					#name: #save 
					#model: #mode 
					#callbacksSpec: 
					#(#UIEventCallbackSubSpec 
						#valueChangeSelector: #changeMode ) 
					#label: 'Save' 
					#select: #save ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 128 15 ) 
					#name: #print 
					#model: #mode 
					#callbacksSpec: 
					#(#UIEventCallbackSubSpec 
						#valueChangeSelector: #changeMode ) 
					#label: 'Print' 
					#select: #print ) 
				#(#InputFieldSpec 
					#layout: #(#Rectangle 81 251 238 272 ) 
					#model: #destination ) 
				#(#ActionButtonSpec 
					#layout: #(#Rectangle 60 288 115 318 ) 
					#model: #ok 
					#label: 'OK' 
					#defaultable: true ) 
				#(#ActionButtonSpec 
					#layout: #(#Rectangle 131 288 186 319 ) 
					#model: #cancel 
					#label: 'Cancel' 
					#defaultable: true ) 
				#(#LabelSpec 
					#layout: #(#Point 13 251 ) 
					#name: #outputDestination 
					#label: 'Filename:' ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 95 ) 
					#name: #latex 
					#model: #fileType 
					#label: 'LaTeX' 
					#select: #latex ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 118 ) 
					#name: #headlessLatex 
					#model: #fileType 
					#label: 'LaTeX, no headers' 
					#select: #headlessLatex ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 141 ) 
					#name: #dvi 
					#model: #fileType 
					#label: 'DVI' 
					#select: #dvi ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 164 ) 
					#name: #postscript 
					#model: #fileType 
					#label: 'Postscript' 
					#select: #postscript ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 187 ) 
					#name: #mpage2 
					#model: #fileType 
					#label: 'Postscript - 2 pg/sheet' 
					#select: #mpage2 ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 210 ) 
					#name: #mpage4 
					#model: #fileType 
					#label: 'Postscript - 4 pg/sheet' 
					#select: #mpage4 ) 
				#(#GroupBoxSpec 
					#layout: #(#Rectangle 34 43 226 239 ) ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 72 ) 
					#name: #rtf 
					#model: #fileType 
					#label: 'RTF' 
					#select: #rtf ) 
				#(#RadioButtonSpec 
					#layout: #(#Point 43 49 ) 
					#name: #source 
					#model: #fileType 
					#label: 'Source' 
					#select: #source ) ) ) )! !

!CodePrinter class methodsFor: 'instance creation'!

for: environment
	^self new environment: environment!

for: environment with: classDefEnv
	^self new environment: environment; classDefinitionEnvironment: classDefEnv! !

!CodePrinter class methodsFor: 'parcel load/unload/save'!

postLoadActionFor: aParcel
	| comment |
	comment := aParcel comment.
	comment := comment
					copyFrom: (comment indexOfSubCollection: 'SUMMARY' startingAt: 1) + 8
					to: comment size.
	(ComposedTextView
			createOn: (TextCollector with: comment)
			label: 'Latex/RTF Printing Installation Workspace'
			icon: (Icon constantNamed: #workspace))
		openIn: (30@250 extent: 500@300)

	"self postLoadActionFor: (Parcel parcelNamed: 'RefactoringBrowser')"! !

