
nil subclass: #DOProxy
  instanceVariableNames: 
    ' diskObject manager userPolicy '
  classVariableNames: ''
  poolDictionaries: ''    !
Object subclass: #DiskObjectServices
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''  !
DiskObjectServices subclass: #DOObjectsPerFilePolicy
  instanceVariableNames: 
    ' userPolicy manager '
  classVariableNames: ''
  poolDictionaries: ''    !
DOObjectsPerFilePolicy subclass: #DOMultiObjectsPerFilePolicy
  instanceVariableNames: 
    ' objectsPerFile '
  classVariableNames: ''
  poolDictionaries: ''   !
DOObjectsPerFilePolicy subclass: #DOSingleObjectPerFilePolicy
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: '' !
DiskObjectServices subclass: #DOService
  instanceVariableNames: 
    ' userPolicy pathName '
  classVariableNames: ''
  poolDictionaries: ''    !
DOService subclass: #DOMultiObjectService
  instanceVariableNames: 
    ' nextIdDOManager fileNamePattern objectsPerFilePolicy '
  classVariableNames: ''
  poolDictionaries: '' !
DOService subclass: #DOSingleObjectService
  instanceVariableNames: 
    ' fileName '
  classVariableNames: ''
  poolDictionaries: ''    !
DiskObjectServices subclass: #DOUserPolicy
  instanceVariableNames: 
    ' changedFiles transactionReadCache transactionWriteCache writeCacheType '
  classVariableNames: ''
  poolDictionaries: ''  !
DOUserPolicy subclass: #DOMultiUserPolicy
  instanceVariableNames: 
    ' lockedFiles '
  classVariableNames: ''
  poolDictionaries: ''  !
DOUserPolicy subclass: #DOSingleUserPolicy
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''    !
Exception subclass: #DOObjectIsDeletedError
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''   !
Error subclass: #DOInvalidVersionError
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: ''    !
Error subclass: #DOLockStuckError
  instanceVariableNames: ''
  classVariableNames: ''
  poolDictionaries: '' !


!DOProxy class methods !
 
Comment
"
	This class has never been tested.
"!
  
DiskObjectServices
	"This is a dummy method is used by the source code filer to 
	denote a Disk Object Service class and it's subclasses."!
   
new
	"Signal an error.  The #newOn: method should be used instead."

	^Error signal: 'Use the #newOn: method to instantiate new objects.'!
   
newOn: anObject
	"Create a new instance of myself and initialize it."

	^self basicNew
		intialize;
		diskObject: anObject;
		yourself.!
  
testMultiUser
	"Test myself with the multiple user policy."

	| mgr |
	mgr := self newOn: 'abcdefg';
		userPolicy: DOMultiUserPolicy new;
		dosFileName: 'multiusr.obj';
		dosPathName: 'C:\'
		yourself.
	mgr store
	mgr read.
	^mgr!
   
testSingleUser
	"Test myself with the single user policy."

	| mgr |
	mgr := self newOn: 'abcdefg';
		userPolicy: DOSingleUserPolicy new;
		dosFileName: 'singleusr.obj';
		dosPathName: 'C:\'
		yourself.
	mgr store
	mgr read.
	^mgr! !



!DiskObjectServices class methods !
 
Comments
"
	Here are the transaction rules for the Disk Object System.
		1) All reads during a transaction are executed as if they were readForUpdate.
		2)	The objects read during a transaction are cached.  Because they have
			been locked, the system does not have to go back to disk during the transaction.
			This is an efficiency feature.
		3) Transactions are meant to be short.  A transaction that includes waiting for user
			input will lock other users out of those objects.
		4) For most uses, an explicit readForUpdate will not be necessary.  Optimistic
			locking is the default.  When a conflict occurs, DOInvalidVersionError will be
			signaled.

	This is how data integrity is maintained in this multi-user environment.
		1) When an object is stored for the first time, it is given a DiskObject Id and a 
			version number.
		2) The DiskObject Id ensures it's identity.  The version number ensures that
			the object has not been changed since it was read.
		3) If an object is changed and a copy is made, make sure the original object
			is stored back to disk instead of the copy.  When the copy is stored to disk 
			it will be stored as a brand new object.
"!
  
DiskObjectServices
	"This is a dummy method is used by the source code filer to 
	denote a Disk Object Service class and it's subclasses."!
   
ExceptionsToExpect
"
You can expect the following exceptions to be signaled when...

DOObjectIsDeletedError			
		When trying to read a deleted object.
	
DOInvalidVersionError
		When trying to update an object that has been updated since 
		it was last read.

DOLockStuckError
		When the system can't read an object because it has been locked 
		for a period of time.

"!
   
new
	"Answer a new, initialized instance of myself."

	^super new initialize!

Overview
"
	The Disk Object System is intended to provide multi-user access to 
	Visual Smalltalk ObjectFiler files.  Also provided is the capability to 
	store multiple objects, either in their own files, or with n objects in 
	each file.

	The MinneStore System uses the Disk Object System to store
	objects and set up indexes for each kind of object.

	The author of this code is Jonathan A. Carlson in Minneapolis, Minnesota.
"! !



!DOObjectsPerFilePolicy class methods !

Comment
"
	These classes handle the number of objects stored in each file.  
	The manager and user policy logic is thereby insulated from any of this knowledge.
"! !



!DOMultiObjectsPerFilePolicy class methods !
   
new: anInteger
	"Create a new instance and set the objectsPerFile instance variable."

	^self new 
		objectsPerFile: anInteger;
		yourself.! !



!DOSingleObjectPerFilePolicy class methods ! !



!DOService class methods !
 
Comment
"
	The DOManager classes make use of the user policy and the
	objects-per-file policy.  They handle the number of objects to
	be stored.  If we have many to store, the DOMultiObjectManager 
	is used.
"

	!
   
constructEventsTriggered
	"Private - answer the set of events that instances of the
        receiver can trigger."

	^super constructEventsTriggered 
		add: #needsUserName;
		yourself.!
 
initializeTestPath: aString
	"Create path.  If it already exists, remove all files in it."

	| dir |
	dir := Directory pathName: aString.
	dir exists
		ifFalse: [ dir create ]
		ifTrue: [ dir isEmpty ifFalse: [ dir allFilesDo: [ :each | each remove ] ]].!
  
newMultiUser
	"Answer a new instance of myself with a multi user policy."

	^self new
		userPolicy: (DOMultiUserPolicy new);
		yourself.!
  
newSingleUser
	"Answer a new instance of myself with a single user policy."

	^self new
		userPolicy: (DOSingleUserPolicy new);
		yourself.!
   
openOn: fullFileName
	"Read myself in from fullFileName"

	^(DOSingleObjectService newSingleUserOn: fullFileName) read! !



!DOMultiObjectService class methods !
   
newMultiUserOn: pathName
	"Answer a new instance of myself with a multi user policy."

	^self newMultiUser
		pathName: pathName;
		yourself.!
  
newSingleUserOn: pathName
	"Answer a new instance of myself with a single user policy."

	^self newSingleUser
		pathName: pathName;
		yourself.! !



!DOSingleObjectService class methods !
   
newMultiUserOn: fullFileName
	"Answer a new instance of myself with a multi user policy."

	^self newMultiUser
		fullFileName: fullFileName;
		yourself.!
  
newSingleUserOn: fullFileName
	"Answer a new instance of myself with a single user policy."

	^self newSingleUser
		fullFileName: fullFileName;
		yourself.!
   
openOn: fullFileName

	^ObjectFiler loadFromPath: fullFileName!
   
testMultiUser
	"Answer a new instance of myself with policies."

	| mgr |
	mgr := self newMultiUserOn: 'c:\junk.obj'.
	mgr store: 'hijklmn'.
	mgr read.
	^mgr!
   
testSimple
	"Answer a new instance of myself with policies."

	| mgr |
	mgr := self newSingleUserOn: 'C:\JUNK.OBJ'.
	mgr store: 'abcdefg'.
	mgr read.
	mgr store: 'hijkl'.
	^mgr! !



!DOUserPolicy class methods !

Comment
	"	Instance of this class store and retrieve objects at the file level. Dealing with the
	individual objects is is left up to the objects-per-file policy.

		A certain amount of transaction processing is supported.	Use beginTransaction
	and commitTransaction (or rollbackTransaction) to use it.
	
		During a transaction all read and store commands will be saved in the cache
	until committed or rolled back.	If multi-user, all read files will be locked until 
	commit or rollback.  This is to help avoid extra disk reads and writes for 
	multi-object processing"! !



!DOMultiUserPolicy class methods !
   
testMultiObjectsPerFile!

testSingleObjectPerFile!

testTheConcept1
	"Test to see if my multi-user concurrency concept will even work.
	Start this method on machine one.  Then start testTheConcept2 on machine two."

	| me lockName |
	lockName := 'K:\TEST.LOK'.
	me := self new.
	[(File fromPath: lockName) exists] whileFalse: [me wait].
	self testTheConcept2.!

testTheConcept2
	"Start this method on machine two."

	| me lockName |
	lockName := 'K:\TEST.LOK'.
	me := self new.
	1 to: 10 do: [ :x |
		me tryToLock: lockName times: 10.
		Transcript nextPutAll: 'Set lock: '; nextPutAll: Time current asString; cr.
		me wait; wait.
		(File fromPath: lockName) remove.
		Transcript nextPutAll: 'Removed lock at: '; nextPutAll: Time current asString; cr.]! !



!DOSingleUserPolicy class methods ! !



!DOObjectIsDeletedError class methods !
 
Comment
"
	This exception is thrown when an object you are trying to read has already been deleted.
"!
   
DiskObjectServices
	"This is a dummy method is used by the source code filer to 
	denote a Disk Object Service class and it's subclasses."! !



!DOInvalidVersionError class methods !
   
Comment
"
	This exception is thrown when an object you are trying to read
	has been locked for a period of time.
"!
 
DiskObjectServices
	"This is a dummy method is used by the source code filer to 
	denote a Disk Object Service class and it's subclasses."! !



!DOLockStuckError class methods !

Comment
"
	This exception is thrown when an object you are trying to update
	has been changed since it was last read.
"!

DiskObjectServices
	"This is a dummy method is used by the source code filer to 
	denote a Disk Object Service class and it's subclasses."! !



!DOProxy methods !
   
delete
	"Delete my disk object. This method only applies to single objects."
	
	^manager delete.!

diskObject
	"Private - If diskObject is not retrieved, retrieve it from disk."

	diskObject isNil
		ifTrue: [ diskObject := self read ].
	^diskObject.!

diskObject: anObject

	diskObject := anObject.!
   
doesNotUnderstand: aMessage
	"Forward the message on to my diskObject."

	| result |
	result := self diskObject 
		perform: aMessage selector
		withArguments: aMessage arguments.
	result == diskObject  "If true, return myself instead of the result"
		ifTrue: [ ^self ].
	^result.!
   
dosFileName: aString
	"Set the file name in my single object policy. (dos stands for Disk Object System)"

	manager fileName: aString.!
  
dosPathName: aString
	"Set the path name in my single object policy. (dos stands for Disk Object System)"

	manager pathName: aString.!
  
initialize

	manager := DOSingleObjectService new.!
   
isProxy
	"Answer true if this is a proxy object."

	^true.!
  
manager: aDOManager
	"Set the object policy and its user policy."

	manager := aDOManager!
   
printOn: aStream
	"Remove this method when this class descends from nil"

	aStream nextPut: ${.
	diskObject printOn: aStream.
	aStream nextPut: $}.!
   
read
	"Answer my disk object after reading it from disk."
	
	^manager read!
  
store
	"Save my disk-object-to-be to disk."

	diskObject isNil
		ifTrue: [ ^nil ].
	manager store: diskObject.
		!

userPolicy: aDOUserPolicy
	"Set the user policy...and inform the object policy about it."

	| policy |
	manager isNil
		ifFalse: [manager userPolicy: aDOUserPolicy].
	userPolicy := aDOUserPolicy.!
  
vmInterrupt: aSymbol
        "Private - Process virtual machine interrupt.  This method is called
        by the virtual machine.  It is entered with interrupts disabled.  The
        interrupt handler should enable interrupts at the appropriate time."
    Process perform: aSymbol.
    ^self! !



!DiskObjectServices methods !
   
initialize
	"May be reimplemented by my subclasses"! !



!DOObjectsPerFilePolicy methods !

basicIdFor: anObject
	"Answer the database assigned id for this object or nil if it has none."

	^manager basicIdFor: anObject!
  
fullFileNameFor: anObject
	"Answer the full file name for anObject."

	^self fullFileNameForId: (self idFor: anObject)!
  
idFor: anObject
	"Answer the internal id for this object.  If it has none, give it one."

	^manager idFor: anObject!
 
manager: aDOManager
	"Setter"

	manager := aDOManager!
   
readAll
	"Answer all the stored objects."

	self implementedBySubclass.!
 
store: anObject
	"Store an object to disk."

	self implementedBySubclass.!
   
storeAll: aCollection
	"Store a collection of objects."

	aCollection do: [ :each | self store: each]!
   
userPolicy
	"Getter - If the userPolicy is nil, get it from the objectPolicy."

	userPolicy == nil
		ifTrue: [userPolicy := manager userPolicy].
	^userPolicy.!

userPolicy: aPolicy
	"Setter"

	userPolicy := aPolicy! !



!DOMultiObjectsPerFilePolicy methods !
   
delete: anObject
	"Remove object with id of anInteger from disk.  Make sure it hasn't been changed by someone first.
	Theoretically my userPolicy should decide whether to check if this object has been changed
	by someone else or not, but it doesn't know how many objects I have per file like I do."

	| id fileName fileContents oldObject index |
	(id := self basicIdFor: anObject) == nil
		ifTrue: [Error signal: 'You are trying to delete an object that has no database id.'].
	fileName := self fullFileNameForId: id.
	fileContents := self readForUpdate: fileName.
	index := self indexForId: id.
	(oldObject := fileContents at: index) == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	self userPolicy compare: oldObject to: anObject. "Okay to delete?  SingleUserPolicy won't check."
	fileContents at: index put: #deleted.
	self userPolicy store: fileContents to: fileName.!
  
deleteId: anInteger
	"Remove object with id of anInteger from disk."

	| fileName fileContents |
	fileName := self fullFileNameForId: anInteger.
	fileContents := self readForUpdate: fileName.
	fileContents at: (self indexForId: anInteger) put: #deleted.
	self userPolicy store: fileContents to: fileName.!

fileNumberForId: anInteger

	^((anInteger - 1) // objectsPerFile) + 1.!
   
fullFileNameForId: anInteger
	"Answer the full file name for id of anInteger."

	| fileNumber fileName |
	fileNumber := self fileNumberForId: anInteger.
	fileName := manager fileNamePattern replaceAll: '9999' with: (fileNumber asZeroFilledString: 4).
	^manager pathName, fileName.!
 
indexFor: anObject
	"Answer the remainder of the id divided by the number of objects per file."

	^self indexForId: (manager idFor: anObject)!
   
indexForId: anInteger
	"Answer the remainder of the id divided by the number of objects per file."

	^((anInteger + -1) \\ objectsPerFile) + 1.!
 
objectsPerFile: anInteger
	"Setter"

	objectsPerFile := anInteger.!
  
readAll
	"Answer all the stored objects."

	| result anOC fileName previousFileName file |
	anOC := OrderedCollection new.
	previousFileName := nil.
	1 to: manager lastId
		do:
			[:id |
			fileName := self fullFileNameForId: id.
			fileName = previousFileName
				ifFalse: [file := self readFileNamed: (self fullFileNameForId: id)].
			result := file at: (self indexForId: id).
			result ~~ #deleted
				ifTrue: [anOC add: result].
			].
	^anOC!
  
readFileNamed: fileName
	"Pass message on to my userPolicy.  if nil is returned to me, answer an array."

	| fileContents |
	^(fileContents := self userPolicy readFileNamed: fileName) isNil
		ifTrue: [Array new: objectsPerFile]
		ifFalse: [fileContents].!
   
readForUpdate: fileName
	"Pass message on to my userPolicy.  if nil is returned to me, answer an array."

	| fileContents |
	^(fileContents := self userPolicy readForUpdate: fileName) isNil
		ifTrue: [Array new: objectsPerFile]
		ifFalse: [fileContents].!
   
readId: anInteger

	| result |
	result := (self readFileNamed: (self fullFileNameForId: anInteger))
						at: (self indexForId: anInteger).
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.!

readIdForUpdate: anInteger

	| result |
	result := (self readForUpdate: (self fullFileNameForId: anInteger))
			at: (self indexForId: anInteger).
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.!
  
store: anObject

	| id fileName index fileContents oldObject |
	id := self idFor: anObject.
	fileName := self fullFileNameForId: id.
	index := self indexForId: id.
	fileContents := self readForUpdate: fileName.
	(oldObject := fileContents at: index) == #deleted
		ifTrue: [DOObjectIsDeletedError signal]
		ifFalse: [userPolicy compare: oldObject to: anObject]. "Ok to update?"
	fileContents at: index put: anObject.
	manager recordUserOn: anObject.
	self userPolicy store: fileContents to: fileName.! !



!DOSingleObjectPerFilePolicy methods !

delete: anObject
	"Ask my user policy to delete this object."

	self userPolicy delete: anObject from: (self fullFileNameFor: anObject)!
 
deleteId: anInteger
	"Remove object with id of anInteger from disk."

	self userPolicy deleteFileNamed: (self fullFileNameForId: anInteger).!

fullFileNameForId: anInteger
	"Answer the full file name for anObject."

	| fileName |
	fileName := manager fileNamePattern replaceAll: '9999' with: (anInteger asZeroFilledString: 4).
	^manager pathName, fileName.!
 
readForUpdate: fileName
	"Pass message on to my userPolicy."

	^self userPolicy readForUpdate: fileName!
 
readId: anInteger

	| result |
	result := self userPolicy readFileNamed: (self fullFileNameForId: anInteger).
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.!
   
readIdForUpdate: anInteger

	| result |
	result := self readForUpdate: (self fullFileNameForId: anInteger).
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.!
 
store: anObject
	"Ask my user policy to store this object."

	self userPolicy update: anObject to: (self fullFileNameFor: anObject)! !



!DOService methods !
   
allowChangesBeforeCommit
	"The objects themselves will be held in the write cache until commit time.  The 
	danger (or benefit) here is that the object may change before being committed."

	self storeObjectsInWriteCache!

beginTransaction
	"Use transactions carefully.  They should be very short periods of time for 
	multi-user because all read files will be locked until committed or rolled back.
	All read and stored objects will be saved in the cache until commit time."
	
	userPolicy beginTransaction.!
  
commitTransaction
	"Abort any changes and unlock all files lock by this user."
	
	userPolicy commitTransaction.!
 
duringTransaction
	"Answer true or false."

	^userPolicy duringTransaction!
  
initialize
	"Default to doing nothing right now."!
 
isDOService

	^true!
  
pathName
	"Answer the path string to store objects in.  Default to the current path"

	pathName isNil
		ifTrue: [ pathName := '.\' ].
	^pathName!
  
pathName: aString
	"Set the pathName instance variable to aString."

	(pathName := aString) last = $\
		ifFalse: [pathName := pathName,'\'].
	!

protect: aBlock
	"Private - Protect a block of code (probably a save command).  Rollback changes if an error occurs.
	If a transaction is already in progress, don't start a new one or commit the old one."

	| myTransaction result |
	myTransaction := false.
	userPolicy duringTransaction  "begin a transaction if not already done."
		ifFalse: [
			myTransaction := true.
			userPolicy beginTransaction].
	[result := aBlock value] 
		on: Error 
		do: [ :error |
			userPolicy rollbackTransaction.
			error pass "pass the error up to whoever is listening."].
	myTransaction 
		ifTrue: [userPolicy commitTransaction].
	^result.!
   
readAll
	"Answer all objects that I have stored."

	self implementedBySubclass
	!
   
recordUserOn: anObject
	"Set the last-updated-by userName property on anObject."

	| userName |
	userName := self triggerEvent: #needsUserName.
	userName == nil
		ifFalse: [anObject propertyAt: #DOUserName put: userName].!

rollbackTransaction
	"Abort any changes and unlock all files lock by this user."
	
	userPolicy rollbackTransaction.!
 
storeMyselfOn: fullFileName
	"Store myself on fullFileName"

	(DOSingleObjectService newSingleUserOn: fullFileName) store: self.!

storeObjectsInWriteCache
	"The objects themselves will be held in the write cache until commit time.  The 
	danger (or benefit) here is that the object may change before being committed."

	userPolicy writeCacheType: #object!
   
storeStreamInWriteCache
	"This is the default.  Objects will be saved to a stream immediately during 
	storing.  The benefit here is that the object can now change before being 
	committed without the changes being saved to disk."

	userPolicy writeCacheType: #stream!
   
userFor: anObject
	"Answer the user that last updated anObject (if none, answer nil)"

	^anObject propertyAt: #DOUserName.!
  
userPolicy
	"Private - Getter"

	^userPolicy!

userPolicy: aDOUserPolicy
	"Private - Set my userPolicy instance variable."

	userPolicy := aDOUserPolicy.!
  
writeCacheType: aSymbol
	"aSymbol can be either #stream or #object.
	#stream means objects will be saved to a stream immediately during storing.
	#object means the objects themselves will be held in the write cache until commit time."

	userPolicy writeCacheType: aSymbol! !



!DOMultiObjectService methods !
  
basicIdFor: anObject
	"Answer the internal database id for this object or give it one if it has none."

	| id |
	^anObject	
		propertyAt: #DOId
		ifAbsent: [nil].!
   
delete: anObject
	"Remove anObject from the database."

	objectsPerFilePolicy delete: anObject.!
 
deleteId: anInteger
	"Remove object with id of anInteger from disk."

	objectsPerFilePolicy deleteId: anInteger!
 
fileNamePattern
	"Private - Get the pattern for the file names.  Default is 'File9999.obj'."

	fileNamePattern == nil
		ifTrue: [ fileNamePattern := 'File9999.obj' ].
	^fileNamePattern!
  
fileNamePattern: aString
	"Set the pattern for the file names.  Default is 'File9999.obj'."

	fileNamePattern := aString.!
   
idFor: anObject
	"Answer the internal database id for this object or give it one if it has none."

	| id |
	^anObject	
		propertyAt: #DOId
		ifAbsent: 
			[anObject propertyAt: #DOId put: (id := self nextId).
			id].!
   
initialize
	"Default the objectsPerFile policy to 1."

	self objectsPerFile: 1!
  
lastId
	"Public - Answer the last id assigned or nil.  This number should generally not be
	trusted.  To be sure you get the correct number, send #read to the nextIdDOManager."

	^self nextIdDOManager read.!
 
lock: anObject
	"Lock this object.  If it has been changed since it was read, signal DOInvalidVersionError."

	#needsWork.
	Error signal: 'This method has not been coded yet.'.
	objectsPerFilePolicy lock: anObject.!

nextId
	"Private - Answer the next available id."

	| lastId |
	lastId := self nextIdDOManager readForUpdate.
	lastId isNil
		ifTrue: [lastId := 0].
	nextIdDOManager store: (lastId := lastId + 1).
	^lastId.!
 
nextIdDOManager
	"Private - Get the next available id."

	nextIdDOManager isNil
		ifTrue:
			[nextIdDOManager := DOSingleObjectService new
				userPolicy: self userPolicy class new;
				fileName: 'nextId.obj';
				pathName: self pathName;
				yourself].
	^nextIdDOManager.!
 
objectsPerFile: anInteger
	"Set the objects-per-file policy based on the value of anInteger."

	| policy |
	anInteger > 1
		ifTrue: [policy := DOMultiObjectsPerFilePolicy new: anInteger]
		ifFalse: [policy := DOSingleObjectPerFilePolicy new].
	objectsPerFilePolicy := policy.
	objectsPerFilePolicy manager: self.
	objectsPerFilePolicy userPolicy: userPolicy.!

read: anObject
	"Answer the disk version of this object or nil if this object has no database id."

	| id |
	(id := self idFor: anObject) == nil
		ifTrue: [^nil].
	^objectsPerFilePolicy readId: id
	!
  
readAll
	"Answer all objects that I have stored."

	^objectsPerFilePolicy readAll
	!

readForUpdate: anObject
	"Answer the disk version of this object or nil if this object has no database id."

	| id result |
	(id := self idFor: anObject) == nil
		ifTrue: [^nil].
	result := self protect: [objectsPerFilePolicy readIdForUpdate: id]
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.!
   
readId: anInteger
	"Answer one object read from disk or nil if none was found."

	| result |
	result := objectsPerFilePolicy readId: anInteger.
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.
	!
 
readIdForUpdate: anInteger
	"Answer one object read from disk or nil if none was found."

	^self protect: [objectsPerFilePolicy readIdForUpdate: anInteger]!
 
store: anObject
	"Store anObject.  If an error occurs, the changes will be rolled back."

	self protect: [objectsPerFilePolicy store: anObject].
	!
 
storeAll: aCollection
	"Store a collection of objects.  If an error occurs, the changes will be rolled back."

	self protect: [objectsPerFilePolicy storeAll: aCollection].! !



!DOSingleObjectService methods !
   
delete
	"Ask my user policy to delete the object from disk."

	^self userPolicy deleteFileNamed: self fullFileName.!
 
deleteId: anInteger
	"Signal an error."

	Error signal: 'The deleteId: method is used with DOMultiObjectPolicy only.'!
   
fileName
	"Answer the name of the file."

	^fileName!

fileName: aString
	"Set the fileName instance variable to aString."

	fileName := aString.!
  
fullFileName
	"Answer the path name and file name as one string."

	^self pathName, self fileName!
   
fullFileName: aString
	"Set the fileName instance variable to aString."

	| i |
	( i := aString last: $\) == nil
		ifTrue: [Error signal: 'Invalid file name.  You must include the full path and drive.'].
	pathName := aString copyFrom: 1 to: i.
	fileName := aString copyFrom: i+1 to: aString size.!

lock: anObject
	"Ask my user policy to lock this object.  If it has changed, DOInvalidVersionError will be signaled.
	This should be the same object that was read from disk earlier."

	| oldObject |  #needsWork. "? if compare fails, should I unlock the file?"
	oldObject := self userPolicy readForUpdate: self fullFileName.
	self userPolicy compare: oldObject to: anObject.  "Make sure the version hasn't changed."
	^self userPolicy.!
   
read
	"Ask my user policy to read the object from disk."

	| result |
	result := self userPolicy readFileNamed: self fullFileName.
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.
!
   
readAll
	"Ask my user policy to read the object from disk."

	^self read.!
   
readForUpdate
	"Answer the object from the disk.  Give my userPolicy a chance to lock it, if it needs to."

	| result |
	result := userPolicy readForUpdate: self fullFileName.
	result == #deleted
		ifTrue: [DOObjectIsDeletedError signal].
	^result.!

readId: anInteger
	"Signal an error."

	Error signal: 'readId: method is used with DOMultiObjectPolicy only.'!
   
store: anObject
	"Pass this object to my user policy along with it's filename to be stored to disk."
	
	self recordUserOn: anObject.
	userPolicy update: anObject to: self fullFileName.! !



!DOUserPolicy methods !
  
backupFileNamed: fileNameString

	| backupName |
	#needsWork.
	backupName := (fileNameString upTo: $.) , '.BAK'.!
   
beginTransaction
	"Set the transactionCache instance variables to a new Dictionary.
	All read and store commands will be saved in the cache until commit time.
	If multi-user, all read files will be locked until commit or rollback."

	transactionWriteCache isDictionary
		ifTrue:
			[transactionWriteCache isEmpty
				ifTrue: [^self]
				ifFalse: [Error signal: 'Transaction must be committed or rolled back before beginning a new one.']].
	transactionWriteCache := Dictionary new.
	transactionReadCache := Dictionary new.
	changedFiles := Set new.!

commitTransaction
	"Save the changed files to disk and wipe out the transactionCaches.
	If I want to get fancy I could backup each file before writing them, and then 
	delete the backup after all are saved okay (I'm afraid this would be too inefficient).
	For now we'll just live with the possibility of a power failure during the commit."

	self duringTransaction
		ifFalse: [Error signal: 'You must begin a transaction before committing it.'].
	transactionWriteCache
		associationsDo:
			[:each |
			each value isStream "If true, copy stream to the file."
				ifTrue: [each value dumpToNewFile: each key]
				ifFalse: [ObjectFiler dump: each value newFile: each key]].
	transactionWriteCache := nil.
	transactionReadCache := nil.
	changedFiles := nil.!
  
deleteFileNamed: aString
	"Store the symbol #deleted to file named aString."

	self store: #deleted to: aString!
 
duringTransaction
	"Answer true or false."

	^transactionWriteCache ~= nil.!
 
initialize
	"Default writeCacheType to #stream."

	writeCacheType := #stream.!
   
isMultiUserPolicy

	^false!
   
readFileNamed: aString
	"Check the write cache.  If nothing found, check the read cache.  
	If nothing found there either, read the file named aString."

	^self duringTransaction
		ifTrue:
			[self
				transactionWriteCacheAt: aString 
				ifAbsent:
					[self
						transactionReadCacheAt: aString
						ifAbsentPut: [ObjectFiler loadFromPathName: aString].
					].
			]
		ifFalse:
			[ObjectFiler loadFromPathName: aString].!
  
rollbackTransaction
	"Wipe out the transactionCache instance variable."

	transactionWriteCache := nil.
	transactionReadCache := nil.
	changedFiles := nil.!
   
store: fileContents
to: fileNameString
	"Save anObject to fileNameString."

	self duringTransaction
		ifTrue:
			[ "Save these changes in the transactionWriteCache to write at commit time."
			self transactionWriteCacheAt: fileNameString put: fileContents.
			changedFiles add: fileNameString]
		ifFalse:
			[ObjectFiler dump: fileContents newFile: fileNameString].
	!
 
transactionReadCacheAt: aString ifAbsentPut: aBlock
	"Answer the value if found, or the block if not."

	^transactionReadCache 
		at: aString 
		ifAbsentPut: aBlock.!
 
transactionWriteCacheAt: aString
ifAbsent: aBlock
	"The transaction write cache keeps the objects in ObjectFiler format.  It used to 
	keep the actual objects but then I ran into problems with them changing before 
	they were committed."

	| result temp |
	temp := transactionWriteCache at: aString ifAbsent: [nil].
	temp == nil
		ifTrue: [^aBlock value].
	^temp isStream
		ifTrue: 
			[result := ObjectFiler loadFrom: temp.
			temp reset. "So we can read from it again if needed."
			result]
		ifFalse: [temp]!
  
transactionWriteCacheAt: aString
put: anObject
	"The transaction write cache keeps the objects in ObjectFiler format.  It used to 
	keep the actual objects but then I ran into problems with them changing before 
	they were committed."

	| stream |
	writeCacheType == #object
		ifTrue: [^transactionWriteCache at: aString ifAbsentPut: [anObject]].
	self ASSERT: (writeCacheType == #stream).
	stream := transactionWriteCache at: aString ifAbsentPut: [ReadWriteStream on: (String new: 300)].
	stream reset.
	ObjectFiler dump: anObject on: stream.
	stream truncate.  "So we don't write out the whole string."
	stream reset. "So we are ready to read or write over."!
 
writeCacheType: aSymbol
	"aSymbol can be either #stream or #object.
	#stream means objects will be saved to a stream immediately during storing.
	#object means the objects themselves will be held in the write cache until commit time."

	(#(stream object) includes: aSymbol)
		ifFalse: [Error signal: 'Invalid writeCacheType of ', aSymbol].
	writeCacheType := aSymbol.! !



!DOMultiUserPolicy methods !
   
basicVersionFor: anObject
	"Answer the version for this object or nil if it has none."

	^anObject propertyAt: #DOVersion!
   
commitTransaction
	"Save my changes and remove all of the locks I have placed."

	super commitTransaction.
	self removeLocks.!
  
compare: oldObject to: newObject
	"Compare the version number of oldObject to newObjects version.
	If they don't match signal DOInvalidVersionError."

	oldObject isNil
		ifTrue: [^nil].
	(self basicVersionFor: newObject) isNil  "We're replacing an old object with a new one."
		ifTrue: [^nil].
	(self versionFor: oldObject) == (self basicVersionFor: newObject)
		ifTrue: [self incrementVersionFor: newObject]
		ifFalse: [DOInvalidVersionError signal]!
   
delete: anObject
from: fileNameString
	"Do object version checking.  This method to be used only when there is one object per file."

	| oldObject |
	(oldObject := self readForUpdate: fileNameString) == #deleted
		ifTrue: [DOObjectIsDeletedError signal]
		ifFalse: [self compare: oldObject to: anObject].
	self store: #deleted to: fileNameString.!
 
incrementVersionFor: anObject
	"Increment the version number for this object."

	anObject	
		propertyAt: #DOVersion
		put: (self versionFor: anObject) + 1.!
   
isMultiUserPolicy

	^true!

lockedFiles
	"Private - Answer the set of file names that have been locked by me."

	lockedFiles == nil
		ifTrue: [lockedFiles := Set new: 10].
	^lockedFiles!
 
lockFileNamed: aString
	"Set the lock on this file name string."

	| lockName |
	lockName := (aString upTo: $.) , '.LOK'.
	(self lockedFiles includes: aString)
		ifTrue: [^self].
	self tryToLock: lockName times: 10.
	self lockedFiles add: aString.!

readFileNamed: aString
	"If a transaction is going on, I need to lock the file anyways since I'll depend on it not changing."

	self duringTransaction
		ifTrue: [self lockFileNamed: aString].
	^super readFileNamed: aString.!
   
readForUpdate: aFullFileName
	"Read anObject from aFullFileName.  This will lock this file until you store or commit it."

	self lockFileNamed: aFullFileName.
	^super readFileNamed: aFullFileName.!
   
removeLocks
	"Remove all locks that I have set.  I use a copy of the lockedFiles collection because
	the #unlockFileNamed: method removes items from the unlockedFiles collection."

	self lockedFiles copy do: [ :each | 
		self unlockFileNamed: each.]!
 
rollbackTransaction
	"Remove all of the locks I have placed.  Since nothing is saved yet, I don't have to explicitly roll anything back.."

	self duringTransaction
		ifFalse: [Error signal: 'You must begin a transaction before rolling it back.'].
	self removeLocks.
	super rollbackTransaction.!

store: fileContents
to: fileNameString
	"Save anObject to fileNameString.  This method does no version checking."

	self lockFileNamed: fileNameString.
	super store: fileContents to: fileNameString.
	self duringTransaction 
		ifFalse: [self unlockFileNamed: fileNameString].!
  
tryToLock: aLockFileName
times: anInteger
	"Private - Set the lock on this file name string. Signal DOLockStuckError if I can't get it."

	anInteger < 1
		ifTrue: [DOLockStuckError signal].
	[File createFileNamed: aLockFileName]
		on: FileAlreadyExists
		do:
			[self wait.  "Wait a second, then call myself again."
			self tryToLock: aLockFileName times: anInteger + -1].
	!
  
unlockFileNamed: aString
	"Remove the lock on this file name string."

	| lockName |
	lockName := (aString upTo: $.) , '.LOK'.
	(File fromPath: lockName) remove.
	self lockedFiles remove: aString.!
 
update: anObject
to: fileNameString
	"Do object version checking.  This method to be used only when there is one object per file."

	| oldObject |
	(oldObject := self readForUpdate: fileNameString) == #deleted
		ifTrue: [DOObjectIsDeletedError signal]
		ifFalse: [self compare: oldObject to: anObject].
	self store: anObject to: fileNameString.!
   
versionFor: anObject
	"Answer the version for this object.  If it has none, give it a one."

	^anObject
		propertyAt: #DOVersion
		ifAbsent: [ "IfAbsentPut: is not working properly for me."
			anObject propertyAt: #DOVersion put: 1.
			1 ].!

wait
	"Private - Loop until the next second."

	| endTime |
	endTime := Time now asSeconds + 1.
	[Time now asSeconds < endTime]
		whileTrue: [].
	! !



!DOSingleUserPolicy methods !
   
compare: oldObject to: newObject
	"Do nothing.  Only the multi-user policy cares about object versioning."!

delete: anObject
from: fileNameString
	"No version checking is needed.  Pass control directly to #store:to:"

	self store: #deleted to: fileNameString.!

readForUpdate: aFullFileNameString
	"Read whatever is in aFileNameString.  Used by DOObjectPolicy>>readForUpdate:.
	The default is no locking.  DOMultiUserPolicy overrides this."

	^self readFileNamed: aFullFileNameString!
  
update: anObject
to: fileNameString
	"No version checking is needed.  Pass control directly to #store:to:"

	self store: anObject to: fileNameString! !



!DOObjectIsDeletedError methods ! !



!DOInvalidVersionError methods ! !



!DOLockStuckError methods ! !



!Object methods !
  
isDOService
	"Answer false."

	#DiskObjectServices.
	^false! !



!IndexedCollection methods !
  
last: anObject
	"Answer the index of the last object equal to anObject.
	Answer nil if no object matches."

	| i |
	#DiskObjectServices.
	i := self size.
	self reverseDo:
		[:each |
		each = anObject ifTrue: [ ^i ].
		i := i - 1].
	^nil.! !



!Integer methods !
   
asZeroFilledString: anInteger
    "Answer the receiver as a string of length anInteger.
    Add leading zeroes to fill it out."

    #DiskObjectServices.
    ^self asString: anInteger fillWith: $0.
! !



!Integer methods !
   
asString: anInteger fillWith: aCharacter
    "Answer the receiver as a string of length anInteger.
	fill leading blanks with aCharacter"

    | string fill |
    #DiskObjectServices.
    string := self asString.
    string size >= anInteger ifTrue: [ ^string ].
    fill := (String new: (anInteger - string size)) atAllPut: aCharacter.
    ^fill, string.! !



!Integer methods !

asString: anInteger
    "Answer the receiver as a string of length anInteger.
    Add leading blanks to fill it out."

    #DiskObjectServices.
    ^self asString: anInteger fillWith: $ .! !



!ObjectFiler class methods !
 
sizeOf: anObject
	"Answer the size in bytes of an object when 'ObjectFiled'."

    | stream |
	#DiskObjectServices.
    stream := ReadWriteStream on: (String new: 300).
    self dump: anObject on: stream.
    stream truncate.
    ^stream size.! !
