TextDocs.NewDoc     [P   Syntax10.Scn.Fnt    Oberon10.Scn.Fnt  2       ~        MODULE PacMan; (* Copyright (C) 1997 by John Drake *)
	IMPORT Files, Objects, Input, Display, Pictures, Display3, Fonts, Effects, Strings, Gadgets, Texts, Oberon, 
				 Documents, Sprites, RandomNumbers, Desktops, Stacks, Attributes, Modules, Out;
	(* PacMan for Oberon Version 1.0 *)

	CONST
		left = 2;
		Left = CHR(196);
		Right = CHR(195);
		Up = CHR(193);
		Down = CHR(194); 
		border = 1;
		
		DocGen = "PacMan.NewDoc";
		DefaultBackPict = "PacMaze.gif";
		SpritePict = "PacSprites.gif";
		
		PacSpriteH = 16;
		PacSpriteW = 14;
		GhostSpriteH = 16;
		GhostSpriteW = 15;
		
		mouthclosed = 1;
		inGenerator = 1;
		scared = 2;
		eaten = 3;
		generatorExitCol = 13;
		
		Maze = "PacData.Text";
		RetPathMaze = "RetPath.Text";
		MaxM = 31; MaxN = 28;
		
		(* Field values for PacMan 'world' *)
		Wall = '0';
		Dot = '1';
		PowerDot = '2';
		Empty = '3'; 
		PacField = '4';
		Door = '5';
		Ghost1 = '6';
		Ghost2 = '7';
		Ghost3 = '8';
		Ghost4 = '9';
		ghostFields = {6..9};
		
		DotColor = Display.FG;
		DotXOfs = 3; DotYOfs = 3;
		cellH = 8;
		cellW = 8;
		PDXOfs = cellW DIV 2; PDYOfs = cellH DIV 2;
		PacYOfs = 6;
		PacXOfs = 4;
		PDR = cellW DIV 2;
		DotPattern = -1;
		scoreH = 40;
		ghostXOfs = cellW DIV 2 - GhostSpriteW DIV 2;
		ghostYOfs = cellH DIV 2 - GhostSpriteH DIV 2;
				
		(* Headings *)
		uph = 0;
		righth = 1;
		downh = 2;
		lefth = 3;
		
		defaultDelayFactor = 0.05;
		scaredDelayFactor = 200;
		
		Start = 1;
		Run = 2;
		
		maxTasks = 20;
		DocMenu = "PacMan.Close[Close] System.Time[Time]";

	TYPE
		Frame* = POINTER TO FrameDesc;
		GameModel = POINTER TO GameModelDesc;

		Task = POINTER TO TaskDesc;
		TaskDesc = RECORD (Oberon.TaskDesc)
			game : GameModel;
		END;
		
		SpriteTask = POINTER TO SpriteTaskDesc;
		SpriteTaskDesc = RECORD (TaskDesc)
			sprite : Sprites.Sprite;
		END;
		
		PacMan = POINTER TO PacManDesc;
		Ghost = POINTER TO GhostDesc;
		
		GameModelDesc = RECORD (Gadgets.ObjDesc)
			level : INTEGER;
			lives : INTEGER;
			score : LONGINT;
			world : ARRAY MaxM+2, MaxN+2 OF CHAR;
			retpath : ARRAY MaxM+2, MaxN+2 OF CHAR;
			rows, cols : INTEGER;
			pacman : PacMan;
			ghosts : ARRAY 4 OF Ghost;
			nghosts : INTEGER;
			nDots, nPowerDots : INTEGER;
			state : INTEGER;
			backPictName : ARRAY 32 OF CHAR;
			done : BOOLEAN;
			backPict: Pictures.Picture;
			taskStack : Stacks.Stack;
		END;

		SpriteDesc = RECORD (Sprites.SpriteDesc)
			state : SET;
			heading, row, col : INTEGER;
			game : GameModel;
			task : SpriteTask;
		END;

		Sprite = POINTER TO SpriteDesc;
				
		GhostDesc = RECORD (SpriteDesc)
			savedChar, ghostCell : CHAR;
			number : INTEGER;
			gendelay, scaredUntilTime : LONGINT;
			blink : BOOLEAN;
		END;
		
		ModelProc = PROCEDURE(g : GameModel);
		
		DelayTask = POINTER TO DelayTaskDesc;
		
		DelayTaskDesc = RECORD(TaskDesc)
			Command : ModelProc;  
			taskStack : Stacks.Stack;
		END;
		
		SequenceTask = POINTER TO RECORD(DelayTaskDesc)
			seqnumber, maxnumber : INTEGER;
		END;	
		
		PacManDesc = RECORD (SpriteDesc)
			inputChar : CHAR;
		END;		
		
		TaskElement = POINTER TO RECORD (Stacks.ElementDesc)
			task : Task;
		END;
		
		FrameDesc* = RECORD (Gadgets.FrameDesc)
			focus : BOOLEAN;
		END;
		
		UpdateMsg = RECORD (Display.FrameMsg)
			message : BOOLEAN;
			messageData : ARRAY 40 OF CHAR;
			messageColor : INTEGER;
			points: INTEGER;
			new: ARRAY 10 OF RECORD x, y: INTEGER END
		END;
		
		SequenceProc = PROCEDURE(g : GameModel);
		
	VAR
		statusBarH: INTEGER;
		delay : LONGINT;
		deathSequence : SequenceProc;
		scaredGhost, eyes : Sprites.Sprite;
		arcadeFont : Fonts.Font;
		pacTask : SpriteTask;
		ghostTask : ARRAY 4 OF SpriteTask;
		delayFactor : REAL;

	PROCEDURE Model(F : Frame):GameModel;
	BEGIN
		RETURN F.obj(GameModel);
	END Model;
				
	PROCEDURE SleepTask(taskElement : Stacks.Element);		
	BEGIN
		IF taskElement # NIL THEN
			WITH taskElement : TaskElement DO;
				Oberon.Remove(taskElement.task);
			END;
		END;
	END SleepTask;

	
	PROCEDURE DrawScore(game : GameModel; M : Display3.Mask; x, y : INTEGER; score : LONGINT);
	VAR
		str : ARRAY 10 OF CHAR;
		xo, yo, w, h, dsr : INTEGER;
		
	BEGIN
		Strings.IntToStr(score, str);
		xo := x+6*cellW; yo := y+(game.rows+3)*cellH - 4;
		Display3.StringSize(str, arcadeFont, w, h, dsr);
		Display3.ReplConst(M, Display3.FG, xo, yo, w, h - 3, Display3.paint);
		Display3.String(M, Display3.white, xo, yo, arcadeFont, str, Display3.paint);
	END DrawScore;

	PROCEDURE ShowLives(M : Display3.Mask; x, y, lives : INTEGER);
	VAR
		x0, y0, i : INTEGER;
		
	BEGIN
		y0 := y + 4;
		IF lives < 3 THEN
			x0 := x + 4 * cellW;
			FOR i := 1 TO (3 - lives) DO;
				Display3.ReplConst(M, Display3.FG, x0, y0, 14, 12, Display3.paint);
				DEC(x0, 16);
			END;
		END;
	END ShowLives;
				
	PROCEDURE RestoreField(game : GameModel; field: CHAR; Q: Display3.Mask; x, y, w, h, xo, yo: INTEGER);
		PROCEDURE RestorePacField(x, y : INTEGER);
		BEGIN
			game.pacman.x := x-PacXOfs;
			game.pacman.y := y-PacYOfs;
			IF mouthclosed IN game.pacman.state THEN
				game.pacman.cell := 0;
				game.pacman.Draw(Q, Display3.paint);
			ELSE
				game.pacman.cell := game.pacman.heading + 1;
				game.pacman.Draw(Q, Display3.paint);
			END;
		END RestorePacField;
		
		PROCEDURE RestoreGhostField(ghostID : CHAR; x, y : INTEGER);
		VAR
			ghostIndex : INTEGER;
			ghost : Ghost;
			
		BEGIN
			ghostIndex := ORD(ghostID) - ORD(Ghost1);
			ghost := game.ghosts[ghostIndex];
			ghost.x := x + ghostXOfs;
			ghost.y := y + ghostYOfs;;
			ghost.cell := ghost.heading;
			IF (~(scared IN ghost.state) & ~(eaten IN ghost.state)) OR ghost.blink THEN
				ghost.Draw(Q, Display3.paint); 
			ELSIF scared IN ghost.state THEN
				scaredGhost.x := ghost.x;
				scaredGhost.y := ghost.y;
				scaredGhost.Draw(Q, Display3.paint);
			ELSIF eaten IN ghost.state THEN
				eyes.x := ghost.x;
				eyes.y := ghost.y;
				eyes.cell := ghost.cell;
				eyes.Draw(Q, Display3.paint);
			END;
		END RestoreGhostField;
		
	BEGIN
		Oberon.RemoveMarks(x, y, w, h);
		CASE field OF
			Dot : Display3.ReplConst(Q, Display.FG, x, y, w, h, Display3.paint);
					Display3.ReplConst(Q, Display.BG, x+DotXOfs, y+DotYOfs, 2, 2, Display3.paint); 		
			|PowerDot : Display3.Circle(Q, 4, Display.solid, x+PDXOfs, y+PDYOfs, PDR, 1, {Display3.filled}, Display3.paint);
			| Empty :  Display3.ReplConst(Q, Display.FG, x, y, w, h, Display3.paint); 
			| PacField :   RestorePacField(x, y);  
			| Wall, Door :  Display3.Pict(Q, game.backPict, x - xo, y-yo, w, h, x, y, Display3.replace); 
			| Ghost1..Ghost4 : RestoreGhostField(field, x, y) ;
		ELSE
		END;
	END RestoreField;
 
	PROCEDURE CalcSize(x, y: INTEGER; VAR xo, yo, dx, dy: INTEGER);
	BEGIN
		dx := cellW;
		xo := x+border;
		dy := cellH;
		yo := y+scoreH-cellH*3;
	END CalcSize; 

	PROCEDURE Restore(F: Frame; Q: Display3.Mask; x, y, w, h: INTEGER);
		VAR 
			i, j, dx, dy, xo, yo, xj, yi: INTEGER;
			game : GameModel;
		
	BEGIN
		game := Model(F);
		IF F.focus & ~game.done THEN
			(* Restore background *)
			CalcSize(x, y, xo, yo, dx, dy);  
			Display3.Pict(Q, game.backPict, 0, 0, F.W, F.H, x+border, y+border, Display3.replace);
			yi := yo;	
			FOR i := game.rows-1 TO 0 BY- 1 DO
				xj := xo;
				FOR j := 0 TO game.cols - 1 DO
					ASSERT(game.world[i,j] # CHR(0));
					IF game.world[i, j] # CHR(0) THEN
						RestoreField (game, game.world[i, j], Q, xj, yi, dx, dy, x, y); 
					END;
					
					INC(xj, dx);
				END;
				INC(yi, dy);
			END;
			IF game.state = Start THEN
				Display3.CenterString(Q, Display3.white, x, y-cellH, w, h-statusBarH, arcadeFont, "READY!", Display.paint);
			END;		
		ELSE
			Display3.Pict(Q, game.backPict, 0, 0, F.W, F.H, x+border, y+border, Display3.replace);	
			IF game.done THEN
				Display3.CenterString(Q, Display3.white, x, y-cellH, w, h-statusBarH, arcadeFont, "GAME OVER", Display.paint);
			ELSIF ~F.focus THEN
				Display3.CenterString(Q, Display3.white, x, y+statusBarH, w, h-statusBarH, arcadeFont, "PAUSED", Display.paint);
			END
		END;
		IF Gadgets.selected IN F.state THEN
			Display3.FillPattern(Q, Display3.white, Display3.selectpat, x, y, x, y, w, h, Display.paint)
		END;
		DrawScore(game, Q, x, y, game.score);
		ShowLives(Q, x, y, game.lives);
	END Restore;

	PROCEDURE (man : PacMan) Eat(ch : CHAR);
	VAR
		i, ghostfield : INTEGER;
		
	BEGIN
		CASE ch OF
			Dot : INC(man.game.score, 10);
				DEC(man.game.nDots);|
			PowerDot : INC(man.game.score, 50);
				FOR i := 0 TO man.game.nghosts -1 DO;
					IF man.game.ghosts[i].state * {inGenerator, eaten} = {} THEN
						INCL(man.game.ghosts[i].state, scared);
						man.game.ghosts[i].scaredUntilTime := Input.Time()+ENTIER(delay*(scaredDelayFactor/man.game.level));
						man.game.ghosts[i].blink := FALSE;
					END;
				END;
				DEC(man.game.nPowerDots);
			|Ghost1, Ghost2, Ghost3, Ghost4 :
				ghostfield := ORD(ch) - ORD(Ghost1);
				man.game.ghosts[ghostfield].state := {eaten};
				man.game.ghosts[ghostfield].blink := FALSE;
				INC(man.game.score, 100);
		ELSE
		END;
	END Eat;
	
	PROCEDURE (man : Sprite) Move(oldCell, newCell : CHAR; VAR outCell : CHAR);
	VAR 
		U : UpdateMsg;
		facing : CHAR;
		lookahead : INTEGER;
		
		PROCEDURE SavePoints(points : SET);
		(* The variable points represents points in the matrix :
			012
			345
			678
		*)
		VAR 
			i, row, col, index : INTEGER;
			
		BEGIN
			index := 0;
			i :=0;
			FOR row := 1 TO 3 DO;
				FOR col := 1 TO 3 DO;
					IF index IN points THEN
						U.new[i].x := (man.col-2)+col;
						U.new[i].y := (man.row-2)+row;
						INC(i);
					END;
					INC(index);
				END;
			END; 
			U.points := i;
		END SavePoints;
		
	BEGIN
		IF (man IS Ghost) THEN
			WITH man : Ghost DO;
				IF (inGenerator IN man.state) THEN
					lookahead := 2;
				ELSE
					lookahead := 1;
				END;
			END;
		ELSE
			lookahead := 1;
		END;
		CASE man.heading OF
			lefth: 
				IF (man.col > 1) & (man.game.world[man.row, man.col-lookahead] # Wall) THEN				
					man.game.world[man.row, man.col] := oldCell;
					
					SavePoints({1,2,4,5,7,8});
								
					DEC(man.col);				
					outCell := man.game.world[man.row, man.col];
					man.game.world[man.row, man.col] := newCell;						
					U.new[6].x := man.col; U.new[6].y := man.row;
					U.points := 7;
					Display.Broadcast(U);
				ELSE
					IF(man.col = 1) & (man.game.world[man.row, man.col - lookahead] # Wall) THEN
						man.game.world[man.row, man.col] := Empty;
					
						SavePoints({0..8});
						
						man.col := man.game.cols;
						outCell := man.game.world[man.row, man.col];
						man.game.world[man.row, man.col] := newCell;						
						U.new[9].x := man.col; U.new[9].y := man.row;
						U.points := 10;
						Display.Broadcast(U);
					END;
				END;
			|righth:
				IF(man.col < man.game.cols-1) & (man.game.world[man.row, man.col+lookahead] # Wall) THEN
					(* Handle "Tunnel through" case *)
					man.game.world[man.row, man.col] := oldCell;
														
					SavePoints({0,1,3,4,6,7});
								
					INC(man.col);				
					outCell := man.game.world[man.row, man.col];
					man.game.world[man.row, man.col] := newCell;						
					U.new[7].x := man.col; U.new[7].y := man.row;
					U.points := 8;
													
					Display.Broadcast(U);
				ELSE
					IF(man.col = man.game.cols - 1) & (man.game.world[man.row, man.col - lookahead] # Wall) THEN	
						(* Handle "Tunnel through" case *)
						man.game.world[man.row, man.col] := oldCell;
												
						SavePoints({0..8});
															
						man.col := 1;
						outCell := man.game.world[man.row, man.col];
						man.game.world[man.row, man.col] := newCell;						
						U.new[9].x := man.col; U.new[9].y := man.row;
						U.points := 10;
						
						Display.Broadcast(U);
					END;					
				END;
			|uph:
				IF(man.row > 0) & (man.game.world[man.row-lookahead, man.col] # Wall)  THEN
					man.game.world[man.row, man.col] := oldCell;
									
					SavePoints({3..8});								
				
					DEC(man.row);
					outCell := man.game.world[man.row, man.col];
					man.game.world[man.row, man.col] := newCell;
					U.new[7].x := man.col; U.new[7].y := man.row;
					U.points := 8;
					
					Display.Broadcast(U);
				END;
			|downh:
				facing := man.game.world[man.row+lookahead, man.col];
				IF (man.row < man.game.rows) & (~((facing = Wall) OR (facing = Door)) OR ((man IS Ghost) & ~(facing = Wall))) THEN
					man.game.world[man.row, man.col] := oldCell;
					
					SavePoints({0..5});
				
					INC(man.row);
					outCell := man.game.world[man.row, man.col];
					
					man.game.world[man.row, man.col] := newCell;
					U.new[7].x := man.col; U.new[7].y := man.row;
					U.points := 8;
					
					Display.Broadcast(U);						
				END (* IF *);
		END (* CASE *);
	END Move;

	PROCEDURE DelayHandler(me : Oberon.Task);
	BEGIN
		WITH me : DelayTask DO;
			me.game.state := Run;
			me.game.taskStack.Copy(me.taskStack);
			ASSERT(~me.taskStack.Empty());
			me.taskStack.Clear();
			Oberon.Remove(me);
			me.Command(me.game);
		END;
	END DelayHandler;
	
	PROCEDURE InitGhost(ghost : Ghost; startrow, startcol, number : INTEGER; gendelay : LONGINT);
	BEGIN
		ghost.row := startrow;
		ghost.col := startcol;
		ghost.number := number;
		ghost.ghostCell := CHR(number+ORD(Ghost1));
		ghost.Init(SpritePict, 0, (number+1)*GhostSpriteH, 4, GhostSpriteW, GhostSpriteH);
		ghost.gendelay := gendelay;
		ghost.state := {inGenerator};
		ghost.savedChar := Empty;
		ASSERT(inGenerator IN ghost.state);
	END InitGhost;

	PROCEDURE (g : GameModel) Reset();
	VAR
		i : INTEGER;
		
	BEGIN
		g.pacman.state := {};
		g.pacman.row := 23;
		g.pacman.col := 13; 
		g.pacman.heading := -1;
		g.world[g.pacman.row, g.pacman.col] := PacField;
		
		FOR i := 0 TO g.nghosts - 1 DO;
			InitGhost(g.ghosts[i], 14, 13, i, i*30);
			g.ghosts[i].game:= g;
			g.ghosts[i].heading := lefth;
			g.world[g.ghosts[i].row, g.ghosts[i].col] :=CHR(ORD( Ghost1)+i);
		END;
	END Reset;
		
	PROCEDURE DeathSpin(game : GameModel);
	VAR
		U : UpdateMsg;
		
	BEGIN
		EXCL(game.pacman.state, mouthclosed);
		ASSERT(game.pacman.heading < 4);
		game.pacman.heading := (game.pacman.heading + 1) MOD 4;
		U.new[0].x := game.pacman.col;
		U.new[0].y := game.pacman.row;
		U.points := 1;
		Display.Broadcast(U);
	END DeathSpin;
								
	PROCEDURE PacHandler (me : Oberon.Task);
	VAR
		man : PacMan;
		U : UpdateMsg;
		row, col : INTEGER;
		eatChar : CHAR;
		ghostfield : INTEGER;
		
	BEGIN
		WITH me : SpriteTask DO
			man := me.sprite(PacMan);
			IF ~(mouthclosed IN man.state) THEN			
				INCL(man.state, mouthclosed);
			ELSE
				EXCL(man.state, mouthclosed);
			END;
			row := man.row;
			col := man.col;
			CASE man.inputChar OF
				Left:
					IF (col > 0) & (man.game.world[row, col-1] # Wall) THEN
						man.heading := lefth;
					END;
				|Right: 
					IF (col < man.game.cols) & (man.game.world[row, col+1] # Wall) THEN
						man.heading := righth;
					END;
				|Up: 
					IF (row > 0) & (man.game.world[row-1, col] # Wall) THEN
						man.heading := uph;
					END;
				|Down: 
					IF (row < man.game.rows) & (man.game.world[row+1, col] # Wall) THEN
						man.heading := downh;
					END;
			ELSE
			END;
			IF man.heading IN {lefth, righth, uph, downh} THEN
				man.Move(Empty, PacField, eatChar);
				IF (eatChar >= Ghost1) & (eatChar <= Ghost4) THEN
					ghostfield := ORD(eatChar) - ORD(Ghost1);
					IF scared IN me.game.ghosts[ghostfield].state THEN
						man.Eat(eatChar);
						IF me.game.ghosts[ghostfield].savedChar # Empty THEN
							man.Eat(me.game.ghosts[ghostfield].savedChar);
							me.game.ghosts[ghostfield].savedChar := Empty;
						END;
					ELSIF eaten IN me.game.ghosts[ghostfield].state THEN
						IF me.game.ghosts[ghostfield].savedChar # Empty THEN
							man.Eat(me.game.ghosts[ghostfield].savedChar);
							me.game.ghosts[ghostfield].savedChar := Empty;
						END;
					ELSIF ~(eaten IN me.game.ghosts[ghostfield].state) THEN
						(* DEC(me.F.game.lives); *)
						deathSequence(me.game);						
					END
				ELSE
					man.Eat(eatChar);
					(* me.time := Input.Time() + delay; *)
				END;
			END;
			me.time := Input.Time() + delay;
		END (* WITH *);
	END PacHandler; 

	PROCEDURE GhostHandler (me : Oberon.Task);
	(* Control procedure for ghost sprite 
			Specs : 
				- Movement
					Normally a ghost will move in the direction of its heading.
					The ghost heading will change if the ghost runs into something 
						(wall or door if outside of generator)
					The ghost heading can also change at random, or change to
						follow or evade PacMan (depending upon vunerable state)
				- Generator
					Ghosts begin life in generator area.  At this point in life movement of
					ghost is similair to movement outside of generator, except ghost can
					move through door		
				- Changing direction
					Random change
						If moving horizontally then randomly choose new vertical heading
						If moving vertically then randomly choose new horizontal heading
						Reverse direction if no other option exists
					Evade change
						If ghost "sees" pacman than do a random change
					Pursue change
						If at position where pacman last seen, change heading to last
							PacMan heading
				- Looking
					Start at current ghost position
					Check each cell in direction of current heading
					Stop checking if offboard, wall seen or PacMan seen
					If PacMan is seen return position and heading
	*)
	

	VAR
		ghost : Ghost;
		oldcol, oldrow, temp : INTEGER;
		headings, tempset : SET;
		ghostfield : INTEGER;
		state : SET;
		heading : INTEGER;
		
		PROCEDURE GetNewHeading(possibleHeadings : SET);
		VAR
			choice, r, c : INTEGER;
			direction : CHAR;
			
		BEGIN
			IF eaten IN ghost.state THEN
				WITH me : SpriteTask DO;
					direction := me.game.retpath[ghost.row, ghost.col];
					r := ghost.row; c := ghost.col;
					CASE direction OF
						'>' : ghost.heading := righth; |
						'<' : ghost.heading := lefth; |
						'^' : ghost.heading := uph; |
						'V' : ghost.heading := downh;|
						'*' : ghost.state := {inGenerator}
					END;	
				END;
			ELSIF ghost.heading IN {lefth, righth} THEN
				IF possibleHeadings * {uph, downh} # {} THEN			
					choice := SHORT(ENTIER(RandomNumbers.Uniform() * 2));
					IF choice = 1 THEN
						IF uph IN possibleHeadings THEN
							ghost.heading := uph;
						ELSE
							ghost.heading := downh;
						END;
					ELSE
						IF downh IN possibleHeadings THEN
							ghost.heading := downh;
						ELSE
							ghost.heading := uph;
						END;
					END;
				ELSE
					IF ghost.heading = lefth THEN
						ghost.heading := righth 
					ELSE
						ghost.heading := lefth;
					END;
				END;
			ELSE 
					IF possibleHeadings * {uph, downh} # {} THEN			
						choice := SHORT(ENTIER(RandomNumbers.Uniform() * 2));
						IF choice = 1 THEN
							IF lefth IN possibleHeadings THEN
								ghost.heading := lefth;
							ELSE
								ghost.heading := righth;
							END;
						ELSE
							IF righth IN possibleHeadings THEN
								ghost.heading := righth;
							ELSE
								ghost.heading := lefth;
							END;
						END;
					ELSE
						IF ghost.heading = uph THEN
							ghost.heading := downh;
						ELSE
							ghost.heading := uph;
						END;
					END;
				END;
		END GetNewHeading;
	
	BEGIN
		WITH me : SpriteTask DO;
			ghost := me.sprite(Ghost);
			IF inGenerator IN ghost.state THEN
				IF (ghost.col = generatorExitCol) & (ghost.gendelay <= 0) THEN
					ghost.heading := uph;
					EXCL(ghost.state, inGenerator);
				END;
				IF ghost.gendelay > 0 THEN
					DEC(ghost.gendelay);
				END;
			END;
			state := ghost.state;
			heading := ghost.heading;
			tempset := ghost.state;
			oldcol := ghost.col; oldrow := ghost.row;
			ghost.Move(ghost.savedChar, ghost.ghostCell, ghost.savedChar);
			
			ghostfield := ORD(ghost.savedChar) - ORD('0');
			IF ghost.savedChar = PacField THEN
			 	ghost.savedChar := Empty;
			 	IF scared IN ghost.state THEN
			 		me.game.pacman.Eat(ghost.ghostCell);
			 		ghost.state := {eaten};
			 	ELSIF ~(eaten IN ghost.state) THEN
				 	ghost.game.world[ghost.row, ghost.col] := PacField;
					deathSequence(me.game);	
				END;
			ELSIF ghostfield IN ghostFields THEN
				ghost.savedChar := ghost.game.ghosts[ORD(ghost.savedChar)-ORD(Ghost1)].savedChar;
				ghostfield := ORD(ghost.savedChar) - ORD('0');
				ASSERT(~(ghostfield IN ghostFields));
			END;
			IF eaten IN ghost.state THEN
				GetNewHeading(headings);
			END;
			IF (ghost.col = oldcol) & (ghost.row = oldrow) THEN
				headings := {uph, downh, lefth, righth};
				IF (ghost.row > 0) & (ghost.game.world[ghost.row-1, ghost.col] = Wall) THEN
					EXCL(headings, uph);
				END;
				IF (ghost.row < ghost.game.rows) & (ghost.game.world[ghost.row+1, ghost.col] = Wall) THEN
					EXCL(headings,  downh); 
				END;
				IF (ghost.col > 0) & (ghost.game.world[ghost.row, ghost.col-1] = Wall) THEN
					EXCL(headings, lefth);
				END;
				IF (ghost.col < ghost.game.cols) & (ghost.game.world[ghost.row, ghost.col+1] = Wall) THEN
					EXCL(headings,  righth);
				END;
				GetNewHeading(headings);
			END;
		END;
		IF scared IN ghost.state THEN
			me.time := Input.Time() + delay*2;
			IF (ghost.scaredUntilTime - me.time) DIV delay <= 0 THEN
				EXCL(ghost.state, scared);
			ELSIF (ghost.scaredUntilTime - me.time) DIV delay < 50 THEN
				ghost.blink := ~ghost.blink;
			END;
		ELSE
			me.time := Input.Time() + delay;
		END;
	END GhostHandler;
												
	PROCEDURE InitTasks(game : GameModel);
	VAR
		i : INTEGER;
		taskSlot : TaskElement;
		genericTask : Oberon.Task; 
		
	BEGIN
		(* Init PacMan task *)
		IF game.pacman.task = NIL THEN
			game.pacman.task := pacTask;
			game.pacman.task.sprite := game.pacman;
			game.pacman.task.safe := FALSE;
			game.pacman.task.handle := PacHandler; 
			game.pacman.task.game := game;
			NEW(taskSlot);
			taskSlot.task := game.pacman.task;
			game.taskStack.Push(taskSlot);
		END;	
		
		(* Init ghost tasks *)
		FOR i := 0 TO game.nghosts - 1 DO;
			IF game.ghosts[i].task = NIL THEN
				game.ghosts[i].task := ghostTask[i];
				game.ghosts[i].task.sprite := game.ghosts[i];
				game.ghosts[i].task.safe := FALSE;
				game.ghosts[i].task.handle := GhostHandler;
				game.ghosts[i].task.game := game;
				NEW(taskSlot);
				taskSlot.task := game.ghosts[i].task;
				game.taskStack.Push(taskSlot);
			END;
		END;
	END InitTasks;																																																																																												
	PROCEDURE WakeTask(taskElement : Stacks.Element);		
	BEGIN
		IF taskElement # NIL THEN
			WITH taskElement : TaskElement DO;
				Oberon.Install(taskElement.task);
			END;
		END;
	END WakeTask;
	
	PROCEDURE WakeTasks(game : GameModel);
	BEGIN
		game.taskStack.ApplyAll(WakeTask);
	END WakeTasks;			
	
	PROCEDURE LoadMaze(VAR maze : ARRAY OF ARRAY OF CHAR; name : ARRAY OF CHAR):BOOLEAN;
		VAR
			i, j, rows, cols, levels: INTEGER;
			file: Files.File;
			R: Texts.Reader;
			T : Texts.Text;
			dummy : CHAR;
			
	BEGIN
		rows := MaxM;
		cols := MaxN;
		NEW(T);
		Texts.Open(T, name);
		Texts.OpenReader(R, T, 0);
		IF T # NIL THEN
			FOR  i := 0 TO rows - 1 DO;
				FOR j := 0 TO cols - 1 DO; 
					Texts.Read(R, maze[i,j]);
				END; 
				Texts.Read(R, dummy); (* Ignore EOLN char *)
			END;
			RETURN TRUE;
		ELSE
			RETURN FALSE;
		END;
	END LoadMaze;

	PROCEDURE CountDots(maze : ARRAY OF ARRAY OF CHAR; VAR dots, pdots : INTEGER); 
	VAR
		i, j : INTEGER;
		
	BEGIN
		dots := 0; pdots := 0;
		FOR i := 0 TO MaxM - 1 DO;
			FOR j := 0 TO MaxN - 1 DO;
				IF maze[i, j] = Dot THEN 
					INC(dots);
				ELSIF maze[i,j] = PowerDot THEN
					INC(pdots);
				END;
			END;
		END;
	END CountDots;
	
	PROCEDURE Startup(game : GameModel);
	VAR
		U : UpdateMsg;
		
	BEGIN
		U.messageData := "READY!";
		U.messageColor := Display3.black;
		Display.Broadcast(U);
		ASSERT(~game.taskStack.Empty());
		WakeTasks(game);
	END Startup;
	
	PROCEDURE BeginStartSequenceTask(game : GameModel);
	VAR
		startSequence : DelayTask;
		taskSlot : TaskElement;
		
	BEGIN
		NEW(startSequence);
		startSequence.safe := FALSE;
		startSequence.game := game;
		startSequence.handle := DelayHandler;
		startSequence.Command := Startup;
		startSequence.time := Input.Time() + delay * 20;
		
		NEW(startSequence.taskStack);
		startSequence.taskStack.Init(maxTasks);
		startSequence.taskStack.Copy(game.taskStack);
		ASSERT(~startSequence.taskStack.Empty());
		game.taskStack.ApplyAll(SleepTask);
		game.taskStack.Clear();
		
		NEW(taskSlot);
		taskSlot.task := startSequence;
		game.taskStack.Push(taskSlot);
		Oberon.Install(startSequence);
	END BeginStartSequenceTask;
															
	PROCEDURE DeathSequenceHandler(me : Oberon.Task);
	VAR
		m : Display.DisplayMsg;
		om : Objects.ObjMsg;		
		u : UpdateMsg;
		i : INTEGER;
			
	BEGIN
		WITH me : SequenceTask DO;
			IF me.seqnumber < me.maxnumber THEN
				INC(me.seqnumber);
				me.Command(me.game);
				me.time := Input.Time() + delay*2;
			ELSE
				Oberon.Remove(me);
				me.game.taskStack.Clear();
				me.game.world[me.game.pacman.row, me.game.pacman.col] := Empty;
				FOR i := 0 TO me.game.nghosts - 1 DO;
					me.game.world[me.game.ghosts[i].row, me.game.ghosts[i].col] := 
						me.game.ghosts[i].savedChar;
				END;
				IF me.game.lives > 0 THEN
					me.game.Reset();
					me.game.state := Start;
					m.id := Display.area;
					Display.Broadcast(m);
					me.game.taskStack.Copy(me.taskStack);
					me.taskStack.Clear();
					BeginStartSequenceTask(me.game);
				ELSE
					me.game.done := TRUE;
					m.id := Display.area;
					Display.Broadcast(m);
				END;
			END;
		END;
	END DeathSequenceHandler;	

	PROCEDURE DeathSequence(game : GameModel);
	VAR 		
		seqtask : SequenceTask;
		taskSlot : TaskElement;
		
	BEGIN
		DEC(game.lives);
		NEW(seqtask);
		seqtask.safe := FALSE;
		NEW(seqtask.taskStack);
		seqtask.taskStack.Init(maxTasks);
		seqtask.taskStack.Copy(game.taskStack);
		game.taskStack.ApplyAll(SleepTask);
		game.taskStack.Clear();
		
		seqtask.seqnumber := 0;
		seqtask.maxnumber := 10;
		seqtask.handle := DeathSequenceHandler;
		seqtask.Command := DeathSpin;
		seqtask.game := game;
		
		NEW(taskSlot);
		taskSlot.task := seqtask;
		game.taskStack.Push(taskSlot);
		Oberon.Install(seqtask);
	END DeathSequence;					
		
	PROCEDURE ClearBoardSequenceHandler(me : Oberon.Task);
	VAR
		m : Display.DisplayMsg;
		i : INTEGER;
		U : UpdateMsg;
		ok : BOOLEAN;
			
	BEGIN
		WITH me : SequenceTask DO;
			IF me.seqnumber < me.maxnumber THEN
				INC(me.seqnumber);
				U.messageData := "L E V E L    C L E A R E D ";
				IF (me.seqnumber MOD 2) = 0 THEN
					U.messageColor := Display3.white;
				ELSE
					U.messageColor := Display3.black;
				END;
				U.message := TRUE;
				Display.Broadcast(U);
				me.time := Input.Time() + delay*2;
			ELSE
				Oberon.Remove(me);
				me.game.taskStack.Clear();

				ok := LoadMaze(me.game.world, Maze);
				CountDots(me.game.world, me.game.nDots, me.game.nPowerDots);
				
				me.game.Reset();
				INC(me.game.level);
				me.game.state := Start;
				m.id := Display.area;
				Display.Broadcast(m);
				me.game.taskStack.Copy(me.taskStack);
				me.taskStack.Clear();
				BeginStartSequenceTask(me.game);
			END;
		END;
	END ClearBoardSequenceHandler;	
		
	PROCEDURE ClearBoardSequence(game : GameModel);
	VAR
		seqtask : SequenceTask;
				taskSlot : TaskElement;
		
	BEGIN
		NEW(seqtask);
		seqtask.safe := FALSE;
		NEW(seqtask.taskStack);
		seqtask.taskStack.Init(maxTasks);
		seqtask.taskStack.Copy(game.taskStack);
		game.taskStack.ApplyAll(SleepTask);
		game.taskStack.Clear();
		
		seqtask.seqnumber := 0;
		seqtask.maxnumber := 10;
		seqtask.handle := ClearBoardSequenceHandler;
		seqtask.game := game;
		
		NEW(taskSlot);
		taskSlot.task := seqtask;
		game.taskStack.Push(taskSlot);
		Oberon.Install(seqtask);
	END ClearBoardSequence;																									
		
	PROCEDURE ControlMan(game : GameModel; ch : CHAR);
	VAR
		U : UpdateMsg;
		row, col : INTEGER;
		
	BEGIN
		game.pacman.inputChar := ch;
	END ControlMan;

	PROCEDURE CopyModel(VAR m: Objects.CopyMsg; g1, g2 : GameModel);
	VAR
		i, j : INTEGER;
		
	BEGIN
		Gadgets.CopyObject(m, g1, g2);
		g2.cols := g1.cols;
		g2.rows := g1.rows;
		g2.done := g1.done;
		COPY(g1.backPictName, g2.backPictName);
		FOR i := 0 TO MaxN+1 DO
			FOR j := 0 TO MaxM+1 DO
				g2.world[i, j] := g1.world[i, j]
			END
		END;
		NEW(g2.backPict);
		Pictures.Open(g2.backPict, g2.backPictName, TRUE);
	END CopyModel;

	PROCEDURE CopyFrame(VAR m: Objects.CopyMsg; f1, f2: Frame);
	VAR
		g1, g2 : GameModel;
		
	BEGIN
		g1 := Model(f1); g2 := Model(f2);
		Gadgets.CopyFrame(m, f1, f2);
		CopyModel(m, g1, g2);
		f1.focus := FALSE;
	END CopyFrame;					
	
	PROCEDURE Close*;
	VAR
		D : Documents.Document;
		game : GameModel;
		
	BEGIN
		D := Desktops.CurDoc(Gadgets.context);
		IF (D # NIL) & (D.dsc IS Frame) THEN
			game := Model(D.dsc(Frame));
			game.taskStack.ApplyAll(SleepTask);
			game.taskStack.Clear();
			Desktops.Close;
		END;
	END Close; 

	PROCEDURE GameHandle*(g : Objects.Object; VAR M : Objects.ObjMsg);
	BEGIN
		Gadgets.objecthandle(g, M);
	END GameHandle;

	PROCEDURE FrameHandle*(F: Objects.Object; VAR M: Objects.ObjMsg);
		VAR
			x, y, w, h, xo, yo, ver, i: INTEGER;
			Q: Display3.Mask;
			keysum: SET;
			copy: Frame;
			dummy : INTEGER;
			game : GameModel;
			
	BEGIN
		WITH F: Frame DO
			game := Model(F);
			IF M IS Display.FrameMsg THEN
				WITH M: Display.FrameMsg DO
     		 	  IF (M.F = NIL) OR (M.F = F) THEN
						x := M.x + F.X;
						y := M.y + F.Y;
						w := F.W;
						h := F.H;
						IF M IS Display.DisplayMsg THEN
							WITH M: Display.DisplayMsg  DO
								IF M.device = Display.screen THEN
									IF (M.id = Display.full) OR (M.F = NIL) THEN
										Gadgets.MakeMask(F, x, y, M.dlink, Q);
										Restore(F, Q, x, y, w, h)
									ELSIF M.id = Display.area THEN
										Gadgets.MakeMask(F, x, y, M.dlink, Q);
										Display3.AdjustMask(Q, x + M.u, y + h - 1 + M.v, M.w, M.h);
										Restore(F, Q, x, y, w, h);
									END;
								ELSIF M.device = Display.printer THEN Gadgets.framehandle(F, M)
								END
							END
						ELSIF M IS UpdateMsg THEN
							WITH M: UpdateMsg DO
								Gadgets.MakeMask(F, x, y, M.dlink, Q);
								IF M.message THEN
									Display3.CenterString(Q, M.messageColor, x, y+statusBarH, w, h-statusBarH, Fonts.Default, M.messageData, Display.paint);
									M.res := -2
								END;
								IF M.points > 0 THEN
									CalcSize(x, y, xo, yo, w, h); 
									dummy := M.points;
									FOR i := 0 TO dummy-1 DO
										ASSERT(M.new[i].y <= MaxM + 1);
										ASSERT(M.new[i].x <= MaxN + 1);
										RestoreField(game, game.world[M.new[i].y, M.new[i].x], Q, xo+(M.new[i].x)*w, 
											yo+(game.rows-M.new[i].y-1)*h, w, h, x, y);
									END; 
									DrawScore(game, Q, x, y, game.score);
									IF(game.nDots = 0) & (game.nPowerDots = 0) & (game.state = Run) THEN
										ClearBoardSequence(game);
									END;
								END
							END
						ELSIF M IS Oberon.InputMsg THEN
							WITH M: Oberon.InputMsg DO
								IF M.id = Oberon.track THEN
									IF Gadgets.InActiveArea(F, M) THEN
										IF M.keys = {left} THEN
											keysum := M.keys;
											REPEAT
												Effects.TrackMouse(M.keys, M.X, M.Y, Effects.Arrow);
												keysum := keysum + M.keys
											UNTIL M.keys = {};
											M.res := 0;
											xo := M.X-(x+12+3*statusBarH);
											yo := M.Y-(y+3);
										ELSIF ~F.focus THEN
											Oberon.Defocus();
											F.focus := TRUE;
											game.taskStack.ApplyAll(WakeTask);  
											Gadgets.Update(F);
										ELSE
											Gadgets.framehandle(F, M)
										END
									ELSE
										Gadgets.framehandle(F, M)
									END 
								ELSIF (M.id = Oberon.consume) & (M.stamp # F.stamp) THEN
									F.stamp := M.stamp;
									IF F.focus THEN
										ControlMan(game, M.ch);
									ELSE
										Gadgets.framehandle(F, M)
									END
								ELSE
									Gadgets.framehandle(F, M)
								END
							END
						ELSIF M IS Oberon.ControlMsg THEN
							WITH  M: Oberon.ControlMsg DO
								IF ((M.id = Oberon.defocus) OR (M.id = Oberon.neutralize)) THEN
									IF F.focus THEN
										game.taskStack.ApplyAll(SleepTask); 
										F.focus := FALSE;
										Gadgets.Update(F)
									END
								END
							END
						ELSE
							Gadgets.framehandle(F, M)
						END
					END
				END
			ELSIF M IS Objects.AttrMsg THEN
				WITH M: Objects.AttrMsg DO
					IF M.id = Objects.get THEN
						IF M.name = "Gen" THEN
							M.class := Objects.String;
							M.s := "PacMan.NewFrame";
							M.res := 0
						ELSE
							Gadgets.framehandle(F, M)
						END
					END
				END
			ELSIF M IS Objects.LinkMsg THEN
				WITH M: Objects.LinkMsg DO;
						(* IF M.name = "Model" THEN
						F.obj := M.obj;
					ELSE
						Gadgets.framehandle(F, M);
					END; *)
				END; 
			ELSE
				Gadgets.framehandle(F, M)
			END
		END
	END FrameHandle;
	
	PROCEDURE NewFrame*;
	VAR
		F : Frame;
	BEGIN
		NEW(F);
		F.W := MaxN * 8+border;
		F.H := MaxM * 8+scoreH;
		F.handle := FrameHandle;
		F.focus := FALSE;
		Objects.NewObj := F;
	END NewFrame;
	
	PROCEDURE NewGame*;
		VAR 
			game : GameModel;
			i : INTEGER;	
			ok : BOOLEAN;
		
	BEGIN
		NEW(game);
		NEW(game.taskStack);
		game.taskStack.Init(maxTasks);
		
		game.level := 1;
		game.state := Start;
		ok := LoadMaze(game.world, Maze);
		ok := ok & LoadMaze(game.retpath, RetPathMaze);
		CountDots(game.world, game.nDots, game.nPowerDots);
		IF ~ok THEN Out.String("PacMaze.Text or RetPath.Text Files Not Loaded");  Out.Ln(); END;
		ASSERT(ok);
		game.rows := MaxM;
		game.cols := MaxN;
		game.done := FALSE;
		game.backPictName := DefaultBackPict;
		NEW(game.backPict);
		Pictures.Open(game.backPict, game.backPictName, TRUE);
		game.score := 0;
				
		NEW(game.pacman);
		game.handle := GameHandle;
		game.pacman.game := game;
		game.pacman.Init(SpritePict, 0, 0, 5, PacSpriteW, PacSpriteH); 
		
		game.nghosts := 4;
		game.lives := 3;
		FOR i := 0 TO game.nghosts - 1 DO;
			NEW(game.ghosts[i]);
		END;
		game.Reset();
		Objects.NewObj := game;
		InitTasks(game);
		BeginStartSequenceTask(game);
	END NewGame;

	PROCEDURE *LoadDoc(D: Documents.Document);
		VAR
			F: Files.File;
			R: Files.Rider;
			frame : Frame;
			tag, x, y, w, h, ref: INTEGER;
			gen: ARRAY 64 OF CHAR;
			lib: Objects.Library;
			len: LONGINT;
			obj: Objects.Object;
	BEGIN
		frame := NIL;
		F := Files.Old(D.name);
		IF F # NIL THEN
			Files.Set(R, F, 0);
			Files.ReadInt(R, tag);
			IF tag = Documents.Id THEN
				Files.ReadString(R, gen);
				Files.ReadInt(R, x);
				Files.ReadInt(R, y);
				Files.ReadInt(R, w);
				Files.ReadInt(R, h);
				Files.ReadInt(R, ref);
				NEW(lib);
				Objects.OpenLibrary(lib);
				Objects.LoadLibrary(lib, F, Files.Pos(R)+1, len);
				lib.GetObj(lib, ref, obj);
				frame := obj(Frame)
			END;
			Files.Close(F)
		END;
		IF frame = NIL THEN					
			D.name := "PacMan.Doc";
			NewFrame;
			frame := Objects.NewObj(Frame);
			NewGame;			
			frame.obj := Objects.NewObj;
			w := frame.W;
			h := frame.H
		END;
		D.W := w;
		D.H := h;
		Documents.Init(D, frame)
	END LoadDoc;

	PROCEDURE *StoreDoc(D: Documents.Document);
		VAR
			F: Files.File;
			R: Files.Rider;
			B: Objects.BindMsg;
			len: LONGINT;
	BEGIN
		F := Files.New(D.name);
		IF F # NIL THEN
			Files.Set(R, F, 0);
			Files.WriteInt(R, Documents.Id);
			Files.WriteString(R, DocGen);
			Files.WriteInt(R, D.X);
			Files.WriteInt(R, D.Y);
			Files.WriteInt(R, D.W);
			Files.WriteInt(R, D.H);
			NEW(B.lib);
			Objects.OpenLibrary(B.lib);
			D.dsc.handle(D.dsc, B);
			Files.WriteInt(R, D.dsc.ref);
			Objects.StoreLibrary(B.lib, F, Files.Pos(R), len);
			Files.Register(F);
			Files.Close(F)
		END
	END StoreDoc;

	
	PROCEDURE DocLink(D: Documents.Document; VAR M: Objects.LinkMsg);
	VAR
		f : Display.Frame;
		capt : ARRAY 64 OF CHAR;
		
	BEGIN
		(* IF (M.id = Objects.get) & ((M.name = "DeskMenu") OR (M.name = "SystemMenu") OR (M.name = "UserMenu")) THEN
			f := Desktops.NewMenu(""); M.res := 0;
			M.obj := f;
			f := f.dsc;
			WHILE (f # NIL) DO
				Attributes.GetString(f, "Caption", capt);
				IF (capt = "Close") THEN
					Attributes.SetString(f, "Cmd", "PacMan.Close");
					Gadgets.Update(f);
				END;
				IF f.slink # NIL THEN f := f.slink(Display.Frame) ELSE f := NIL; END;
			END;
		ELSE
			Documents.Handler(D,M);
		END; *)
	END DocLink; 

	PROCEDURE DocHandler*(D: Objects.Object; VAR M: Objects.ObjMsg);	
	BEGIN
		WITH D: Documents.Document DO
			IF M IS Objects.AttrMsg THEN
				WITH M: Objects.AttrMsg DO
					IF M.id = Objects.get THEN
						IF M.name = "Gen" THEN
							M.class := Objects.String; M.s := DocGen; M.res := 0
						ELSE
							Documents.Handler(D, M)
						END
					ELSE
						Documents.Handler(D, M)
					END
				END
			ELSIF M IS Objects.LinkMsg THEN
				WITH M: Objects.LinkMsg DO
					DocLink(D, M);					
				END
			ELSE
				Documents.Handler(D, M)
			END
		END
	END DocHandler;

	PROCEDURE SetSpeed*;
	VAR
		S : Texts.Scanner;
	BEGIN
		Texts.OpenScanner(S, Oberon.Par.text, Oberon.Par.pos);
		Texts.Scan(S);
		IF S.class = Texts.Real THEN
			delayFactor := S.x;
		ELSE
			delayFactor := defaultDelayFactor;
		END;
		delay := ENTIER(Input.TimeUnit * delayFactor);
	END SetSpeed;

	PROCEDURE NewDoc*;
		VAR D: Documents.Document;
	BEGIN
		NEW(D);
		D.Load := LoadDoc;
		D.Store := StoreDoc;
		D.handle := DocHandler;
		Objects.NewObj := D;
	END NewDoc;
	
	PROCEDURE CleanUp;
	VAR
		i : INTEGER;
		
	BEGIN
		Oberon.Remove(pacTask);
		FOR i := 0 TO 3 DO;
			Oberon.Remove(ghostTask[i]);
		END;
	END CleanUp;
		
BEGIN
	delayFactor := defaultDelayFactor;
	statusBarH := 2*Fonts.Default.height;
	delay := ENTIER(Input.TimeUnit * delayFactor);
	deathSequence := DeathSequence;
	NEW(scaredGhost);
	scaredGhost.Init(SpritePict, GhostSpriteW*4, GhostSpriteH, 1, GhostSpriteW, GhostSpriteH);
	scaredGhost.cell := 0;
	NEW(eyes);
	eyes.Init(SpritePict, 0, GhostSpriteH*5+2, 4, GhostSpriteW, GhostSpriteH);
	arcadeFont := Fonts.This("Arial12b.Scn.Fnt");
	NEW(pacTask);
	NEW(ghostTask[0]); NEW(ghostTask[1]); NEW(ghostTask[2]); NEW(ghostTask[3]);
	Modules.InstallTermHandler(CleanUp);
END PacMan.

Desktops.OpenDoc (PacMan.NewDoc)  ~

Sprites.Mod

